// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.io;

import org.refcodes.component.CloseException;
import org.refcodes.component.OpenException;

/**
 * Implementation of the {@link LoopbackShortTransceiver} interface.
 */
public class LoopbackShortTransceiverImpl extends AbstractShortReceiver implements LoopbackShortTransceiver {

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private LoopbackShortTransceiver _loopbackTransceiver = null;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new loopback short transceiver impl.
	 */
	public LoopbackShortTransceiverImpl() {
		super();
	}

	/**
	 * Instantiates a new loopback short transceiver impl.
	 *
	 * @param aCapacity the capacity
	 */
	public LoopbackShortTransceiverImpl( int aCapacity ) {
		super( aCapacity );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isOpenable( LoopbackShortTransceiver aLoopbackTransceiver ) {
		return !isOpened() && aLoopbackTransceiver != null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void open( LoopbackShortTransceiver aLoopbackTransceiver ) throws OpenException {
		if ( isOpened() ) {
			if ( _loopbackTransceiver == aLoopbackTransceiver && _loopbackTransceiver.isOpened() ) {
				return;
			}
			throw new OpenException( "Unable to open the connection is is is ALREADY OPEN; connection status is " + getConnectionStatus() + "." );
		}
		super.open();
		_loopbackTransceiver = aLoopbackTransceiver;
		if ( _loopbackTransceiver.isOpenable( this ) ) {
			_loopbackTransceiver.open( this );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void writeDatagram( short aDatagram ) throws OpenException {
		if ( !isOpened() ) {
			throw new OpenException( "Unable to write datagram <" + aDatagram + "> as the connection is NOT OPEN; connection status is " + getConnectionStatus() + "." );
		}
		_loopbackTransceiver.pushDatagram( aDatagram );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void pushDatagram( short aDatagram ) throws OpenException {
		super.pushDatagram( aDatagram );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void flush() throws OpenException {
		_loopbackTransceiver.flush();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void close() throws CloseException {
		super.close();
		if ( _loopbackTransceiver != null && !_loopbackTransceiver.isClosed() ) {
			_loopbackTransceiver.close();
			_loopbackTransceiver = null;
		}
	}
}
