// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////
package org.refcodes.logger.alt.console;

import java.io.PrintStream;

import org.refcodes.logger.ColumnLayout;
import org.refcodes.logger.ColumnLayoutAccessor.ColumnLayoutBuilder;
import org.refcodes.logger.ColumnLayoutAccessor.ColumnLayoutProperty;
import org.refcodes.logger.RuntimeLoggerSingleton;
import org.refcodes.textual.TableStyle;

/**
 * The Interface ConsoleLogger.
 *
 * @author steiner
 */
public interface ConsoleLogger extends FormattedLogger<Object>, ColumnLayoutProperty, ColumnLayoutBuilder<ConsoleLogger> {

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	default ConsoleLogger withColumnLayout( ColumnLayout aColumnLayout ) {
		setColumnLayout( aColumnLayout );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default ConsoleLogger withTableStyle( TableStyle aTableStyle ) {
		setTableStyle( aTableStyle );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default ConsoleLogger withRowWidth( int aRowWidth ) {
		setRowWidth( aRowWidth );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default FormattedLogger<Object> withEscapeCodes( boolean isEscCodesEnabled ) {
		setEscapeCodes( isEscCodesEnabled );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default FormattedLogger<Object> withLeftBorder( boolean hasLeftBorder ) {
		setLeftBorder( hasLeftBorder );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default FormattedLogger<Object> withRightBorder( boolean hasRightBorder ) {
		setRightBorder( hasRightBorder );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default FormattedLogger<Object> withErrorPrintStream( PrintStream aErrorPrintStream ) {
		setErrorPrintStream( aErrorPrintStream );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default FormattedLogger<Object> withStandardPrintStream( PrintStream aStandardPrintStream ) {
		setStandardPrintStream( aStandardPrintStream );
		return this;
	}

	/**
	 * As the underlying configuration framework used by the
	 * {@link RuntimeLoggerSingleton} cannot convert a String to an enum, we got
	 * to provide such a method ourselves. When configuring programmatically use
	 * {@link #setColumnLayout(ColumnLayout)}, when using the
	 * "<code>runtimelogger-config.xml</code>" use the property "loggerLayout"
	 * for this method to be invoked. Valid arguments are as of the
	 * {@link ColumnLayout} enumeration:
	 * 
	 * <ul>
	 * <li>SUPERUSER</li>
	 * <li>FALLBACK</li>
	 * <li>DEVELOPER</li>
	 * <li>DEVOPS</li>
	 * <li>END_USER</li>
	 * </ul>
	 *
	 * @param aLoggerLayout the new logger layout
	 */
	default void setLoggerLayout( String aLoggerLayout ) {
		setColumnLayout( ColumnLayout.toColumnLayout( aLoggerLayout ) );
	}

}