// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.logger.alt.console;

import static org.fusesource.jansi.Ansi.*;

import org.fusesource.jansi.Ansi.Color;
import org.refcodes.logger.ColumnLayout;
import org.refcodes.logger.LogPriority;
import org.refcodes.logger.LoggerField;
import org.refcodes.tabular.FormattedColumnDecorator;
import org.refcodes.tabular.FormattedHeaderImpl;
import org.refcodes.tabular.PrintStackTrace;
import org.refcodes.textual.ColumnSetupMetrics;
import org.refcodes.textual.ColumnSetupMetricsImpl;
import org.refcodes.textual.ColumnWidthType;
import org.refcodes.textual.EscapeCodeFactory;
import org.refcodes.textual.HorizAlignTextMode;
import org.refcodes.textual.MoreTextMode;
import org.refcodes.textual.SplitTextMode;
import org.refcodes.textual.TextFormatMode;

/**
 * The Class ConsoleLoggerHeaderImpl.
 */
public class ConsoleLoggerHeaderImpl extends FormattedHeaderImpl<Object> {

	private static final long serialVersionUID = 1L;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static EscapeCodeFactory LOG_PRIORITY_ESC_CODE_FACTORY = new EscapeCodeFactory() {
		@Override
		public String createInstance( Object aIdentifier ) {
			if ( aIdentifier != null ) {
				String theIdentifier = aIdentifier instanceof String ? (String) aIdentifier : aIdentifier.toString();
				if ( LogPriority.PANIC.name().equals( theIdentifier ) ) return ANSI_PRIORITY_PANIC;
				if ( LogPriority.ALERT.name().equals( theIdentifier ) ) return ANSI_PRIORITY_ALERT;
				if ( LogPriority.CRITICAL.name().equals( theIdentifier ) ) return ANSI_PRIORITY_CRITICAL;
				if ( LogPriority.ERROR.name().equals( theIdentifier ) ) return ANSI_PRIORITY_ERROR;
				if ( LogPriority.WARN.name().equals( theIdentifier ) ) return ANSI_PRIORITY_WARN;
				if ( LogPriority.NOTICE.name().equals( theIdentifier ) ) return ANSI_PRIORITY_NOTICE;
				if ( LogPriority.INFO.name().equals( theIdentifier ) ) return ANSI_PRIORITY_INFO;
				if ( LogPriority.DEBUG.name().equals( theIdentifier ) ) return ANSI_PRIORITY_DEBUG;
				if ( LogPriority.TRACE.name().equals( theIdentifier ) ) return ANSI_PRIORITY_TRACE;
			}
			return null;
		}
	};

	private static final String EXCEPTION = "Exception";

	private static final String MESSAGE = "Message";

	private static final String METHOD = "Method";

	private static final String CODE_LINE = "@";

	private static final String CLASS = "Class";

	private static final String REQUEST = "Request";

	private static final String SESSION = "Session";

	private static final String THREAD = "Thread";

	private static final String PRIORITY = "Level";

	private static final String DATE = "Date";

	private static final String LINE_NUMBER = "#";

	private static final String ANSI_LINE_NUMBER = ansi().fg( Color.MAGENTA ).bg( Color.DEFAULT ).toString();

	private static final String ANSI_DATE = ansi().fg( Color.WHITE ).toString();

	private static final String ANSI_THREAD = ansi().fg( Color.CYAN ).toString();

	private static final String ANSI_CORRELATION_ID = ansi().fg( Color.CYAN ).bg( Color.DEFAULT ).toString();

	private static final String ANSI_MESSAGE = ansi().fg( Color.DEFAULT ).bg( Color.DEFAULT ).toString();

	private static final String ANSI_EXCEPTION = ansi().fgBright( Color.RED ).toString();

	private static final String ANSI_CLASS = ansi().fg( Color.DEFAULT ).bg( Color.DEFAULT ).toString();

	private static final String ANSI_METHOD = ansi().bold().fg( Color.BLUE ).toString();

	private static final String ANSI_PRIORITY_WARN = ansi().bold().fg( Color.RED ).toString();

	private static final String ANSI_PRIORITY_NOTICE = ansi().bold().fg( Color.MAGENTA ).toString();

	private static final String ANSI_PRIORITY_INFO = ansi().bold().fg( Color.GREEN ).toString();

	private static final String ANSI_PRIORITY_PANIC = ansi().bold().fgBright( Color.WHITE ).bgBright( Color.RED ).toString();

	private static final String ANSI_PRIORITY_ALERT = ansi().bold().fgBright( Color.YELLOW ).bgBright( Color.RED ).toString();

	private static final String ANSI_PRIORITY_CRITICAL = ansi().bold().fgBright( Color.YELLOW ).bgBright( Color.MAGENTA ).toString();

	private static final String ANSI_PRIORITY_ERROR = ansi().bold().fgBright( Color.RED ).toString();

	private static final String ANSI_PRIORITY_DEBUG = ansi().bold().fgBright( Color.YELLOW ).bgBright( Color.BLUE ).toString();

	private static final String ANSI_PRIORITY_TRACE = ansi().bold().fg( Color.WHITE ).toString();

	private static final String ANSI_HEADER = ansi().bold().fg( Color.DEFAULT ).bg( Color.DEFAULT ).toString();

	private static final String ANSI_RESET = ansi().reset().toString();

	// @formatter:off
	private static final ColumnSetupMetrics[] SUPERUSER_COLUMN_METRICS = new ColumnSetupMetrics[] {
		/* Line number: */ new ColumnSetupMetricsImpl( 10, ColumnWidthType.ABSOLUTE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.RIGHT ).withRowEscapeCode( ANSI_LINE_NUMBER ).withRowMoreTextMode( MoreTextMode.NONE ).withName(LINE_NUMBER),
		/* Date:        */ new ColumnSetupMetricsImpl( 19, ColumnWidthType.ABSOLUTE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.RIGHT ).withRowEscapeCode( ANSI_DATE ).withRowMoreTextMode( MoreTextMode.NONE ).withName(DATE),
		/* Priority:    */ new ColumnSetupMetricsImpl(  7, ColumnWidthType.ABSOLUTE ).withTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.CENTER).withRowEscapeCodeFactory( LOG_PRIORITY_ESC_CODE_FACTORY ).withRowMoreTextMode( MoreTextMode.NONE ).withName(PRIORITY),
		/* Thread:      */ new ColumnSetupMetricsImpl( 10, ColumnWidthType.ABSOLUTE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.LEFT ).withRowEscapeCode( ANSI_THREAD ).withRowMoreTextMode( MoreTextMode.LEFT ).withName(THREAD),
		/* Session:     */ new ColumnSetupMetricsImpl(  8, ColumnWidthType.ABSOLUTE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.CENTER ).withRowEscapeCode( ANSI_CORRELATION_ID ).withRowMoreTextMode( MoreTextMode.LEFT ).withName(SESSION),
		/* Request:     */ new ColumnSetupMetricsImpl(  8, ColumnWidthType.ABSOLUTE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.CENTER ).withRowEscapeCode( ANSI_CORRELATION_ID ).withRowMoreTextMode( MoreTextMode.LEFT ).withName(REQUEST),
		/* Class:       */ new ColumnSetupMetricsImpl(  1, ColumnWidthType.RELATIVE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.LEFT ).withRowEscapeCode(ANSI_CLASS ).withRowMoreTextMode( MoreTextMode.LEFT ).withName(CLASS),
		/* Code line:   */ new ColumnSetupMetricsImpl(  4, ColumnWidthType.ABSOLUTE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.RIGHT ).withRowEscapeCode( ANSI_CLASS ).withRowMoreTextMode( MoreTextMode.LEFT ).withName(CODE_LINE),
		/* Method:      */ new ColumnSetupMetricsImpl( 30, ColumnWidthType.ABSOLUTE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.LEFT ).withRowEscapeCode( ANSI_METHOD ).withRowMoreTextMode( MoreTextMode.LEFT ).withName(METHOD),
		/* Message:     */ new ColumnSetupMetricsImpl(  2, ColumnWidthType.RELATIVE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.LEFT ).withRowEscapeCode( ANSI_MESSAGE ).withRowMoreTextMode( MoreTextMode.LEFT ).withName(MESSAGE),
		/* Exception:   */ new ColumnSetupMetricsImpl(  1, ColumnWidthType.RELATIVE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.NONE ).withRowHorizAlignTextMode( HorizAlignTextMode.LEFT).withRowEscapeCode( ANSI_EXCEPTION ).withRowSplitTextMode( SplitTextMode.AT_FIRST_END_OF_LINE ).withName( EXCEPTION ),
	};
	// @formatter:on

	// @formatter:off
	private static final ColumnSetupMetrics[] GRANDPA_COLUMN_METRICS = new ColumnSetupMetrics[] {
		/* Line number: */ new ColumnSetupMetricsImpl().withHide(), 
		/* Date:        */ new ColumnSetupMetricsImpl(  19, ColumnWidthType.ABSOLUTE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.RIGHT).withRowEscapeCode( ANSI_DATE ).withRowMoreTextMode( MoreTextMode.LEFT ).withName( DATE), 
		/* Priority:    */ new ColumnSetupMetricsImpl(  7, ColumnWidthType.ABSOLUTE ).withTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.CENTER ).withRowEscapeCodeFactory( LOG_PRIORITY_ESC_CODE_FACTORY ).withRowMoreTextMode( MoreTextMode.NONE ).withName(PRIORITY),
		/* Thread:      */ new ColumnSetupMetricsImpl(  8, ColumnWidthType.ABSOLUTE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.LEFT).withRowEscapeCode( ANSI_THREAD ).withRowMoreTextMode( MoreTextMode.LEFT ).withName( THREAD ), 
		/* Session:     */ new ColumnSetupMetricsImpl().withHide(),
		/* Request:     */ new ColumnSetupMetricsImpl().withHide(),
		/* Class:       */ new ColumnSetupMetricsImpl().withHide(), 
		/* Code line:   */ new ColumnSetupMetricsImpl().withHide(),
		/* Method:      */ new ColumnSetupMetricsImpl().withHide(),
		/* Message:     */ new ColumnSetupMetricsImpl(  7, ColumnWidthType.RELATIVE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.LEFT).withRowEscapeCode( ANSI_MESSAGE ).withRowMoreTextMode( MoreTextMode.LEFT ).withName( MESSAGE ),
		/* Exception:   */ new ColumnSetupMetricsImpl(  1, ColumnWidthType.RELATIVE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.NONE ).withRowHorizAlignTextMode( HorizAlignTextMode.LEFT).withRowEscapeCode( ANSI_EXCEPTION ).withRowSplitTextMode( SplitTextMode.AT_FIRST_END_OF_LINE ).withName( EXCEPTION ),
	};
	// @formatter:on

	// @formatter:off
	private static final ColumnSetupMetrics[] DEVELOPER_COLUMN_METRICS = new ColumnSetupMetrics[] {
		/* Line number: */ new ColumnSetupMetricsImpl().withHide(), 
		/* Date:        */ new ColumnSetupMetricsImpl( 19, ColumnWidthType.ABSOLUTE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.RIGHT).withRowEscapeCode( ANSI_DATE ).withRowMoreTextMode( MoreTextMode.NONE ).withName(DATE), 
		/* Priority:    */ new ColumnSetupMetricsImpl(  7, ColumnWidthType.ABSOLUTE ).withTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.CENTER ).withRowEscapeCodeFactory( LOG_PRIORITY_ESC_CODE_FACTORY ).withRowMoreTextMode( MoreTextMode.NONE ).withName(PRIORITY), 
		/* Thread:      */ new ColumnSetupMetricsImpl(  8, ColumnWidthType.ABSOLUTE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.LEFT).withRowEscapeCode( ANSI_THREAD ).withRowMoreTextMode( MoreTextMode.LEFT ).withName( THREAD ),
		/* Session:     */ new ColumnSetupMetricsImpl().withHide(),
		/* Request:     */ new ColumnSetupMetricsImpl(  8, ColumnWidthType.ABSOLUTE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.CENTER ).withRowEscapeCode( ANSI_CORRELATION_ID ).withRowMoreTextMode( MoreTextMode.LEFT ).withName(REQUEST),
		/* Class:       */ new ColumnSetupMetricsImpl(  3, ColumnWidthType.RELATIVE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.LEFT).withRowEscapeCode( ANSI_CLASS ).withRowMoreTextMode( MoreTextMode.LEFT ).withName( CLASS ), 
		/* Code line:   */ new ColumnSetupMetricsImpl(  4, ColumnWidthType.ABSOLUTE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.RIGHT ).withRowEscapeCode( ANSI_CLASS ).withRowMoreTextMode( MoreTextMode.LEFT ).withName(CODE_LINE),
		/* Method:      */ new ColumnSetupMetricsImpl(  2, ColumnWidthType.RELATIVE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.LEFT).withRowEscapeCode( ANSI_METHOD ).withRowMoreTextMode( MoreTextMode.LEFT ).withName( METHOD ),
		/* Message:     */ new ColumnSetupMetricsImpl(  7, ColumnWidthType.RELATIVE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.LEFT).withRowEscapeCode( ANSI_MESSAGE ).withRowMoreTextMode( MoreTextMode.LEFT ).withName( MESSAGE ),
		/* Exception:   */ new ColumnSetupMetricsImpl(  1, ColumnWidthType.RELATIVE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.NONE ).withRowHorizAlignTextMode( HorizAlignTextMode.LEFT).withRowEscapeCode( ANSI_EXCEPTION ).withRowSplitTextMode( SplitTextMode.AT_FIRST_END_OF_LINE ).withName( EXCEPTION ),
	};
	// @formatter:on

	// @formatter:off
		private static final ColumnSetupMetrics[] DEVOPS_COLUMN_METRICS = new ColumnSetupMetrics[] {
			/* Line number: */ new ColumnSetupMetricsImpl().withHide(), 
			/* Date:        */ new ColumnSetupMetricsImpl( 19, ColumnWidthType.ABSOLUTE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.RIGHT).withRowEscapeCode( ANSI_DATE ).withRowMoreTextMode( MoreTextMode.NONE ).withName(DATE), 
			/* Priority:    */ new ColumnSetupMetricsImpl(  7, ColumnWidthType.ABSOLUTE ).withTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.CENTER ).withRowEscapeCodeFactory( LOG_PRIORITY_ESC_CODE_FACTORY ).withRowMoreTextMode( MoreTextMode.NONE ).withName(PRIORITY), 
			/* Thread:      */ new ColumnSetupMetricsImpl(  8, ColumnWidthType.ABSOLUTE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.LEFT).withRowEscapeCode( ANSI_THREAD ).withRowMoreTextMode( MoreTextMode.LEFT ).withName( THREAD ),
			/* Session:     */ new ColumnSetupMetricsImpl(  8, ColumnWidthType.ABSOLUTE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.LEFT).withRowEscapeCode( ANSI_CORRELATION_ID).withRowMoreTextMode( MoreTextMode.LEFT ).withName( SESSION),
			/* Request:     */ new ColumnSetupMetricsImpl(  8, ColumnWidthType.ABSOLUTE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.CENTER ).withRowEscapeCode( ANSI_CORRELATION_ID ).withRowMoreTextMode( MoreTextMode.LEFT ).withName(REQUEST),
			/* Class:       */ new ColumnSetupMetricsImpl(  3, ColumnWidthType.RELATIVE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.LEFT).withRowEscapeCode( ANSI_CLASS ).withRowMoreTextMode( MoreTextMode.LEFT ).withName( CLASS ), 
			/* Code line:   */ new ColumnSetupMetricsImpl().withHide(),
			/* Method:      */ new ColumnSetupMetricsImpl(  2, ColumnWidthType.RELATIVE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.LEFT).withRowEscapeCode( ANSI_METHOD ).withRowMoreTextMode( MoreTextMode.LEFT ).withName( METHOD ),
			/* Message:     */ new ColumnSetupMetricsImpl(  7, ColumnWidthType.RELATIVE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.LEFT).withRowEscapeCode( ANSI_MESSAGE ).withRowMoreTextMode( MoreTextMode.LEFT ).withName( MESSAGE ),
			/* Exception:   */ new ColumnSetupMetricsImpl(  1, ColumnWidthType.RELATIVE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.NONE ).withRowHorizAlignTextMode( HorizAlignTextMode.LEFT).withRowEscapeCode( ANSI_EXCEPTION ).withRowSplitTextMode( SplitTextMode.AT_FIRST_END_OF_LINE ).withName( EXCEPTION ),
		};
		// @formatter:on

	// @formatter:off
		private static final ColumnSetupMetrics[] ENDUSER_COLUMN_METRICS = new ColumnSetupMetrics[] {
			/* Line number: */ new ColumnSetupMetricsImpl().withHide(), 
			/* Date:        */ new ColumnSetupMetricsImpl( 19, ColumnWidthType.ABSOLUTE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.RIGHT).withRowEscapeCode( ANSI_DATE ).withRowMoreTextMode( MoreTextMode.NONE ).withName(DATE), 
			/* Priority:    */ new ColumnSetupMetricsImpl(  7, ColumnWidthType.ABSOLUTE ).withTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.CENTER ).withRowEscapeCodeFactory( LOG_PRIORITY_ESC_CODE_FACTORY ).withRowMoreTextMode( MoreTextMode.NONE ).withName(PRIORITY), 
			/* Thread:      */ new ColumnSetupMetricsImpl().withHide(),
			/* Session:     */ new ColumnSetupMetricsImpl().withHide(),
			/* Request:     */ new ColumnSetupMetricsImpl().withHide(),
			/* Class:       */ new ColumnSetupMetricsImpl().withHide(), 
			/* Code line:   */ new ColumnSetupMetricsImpl().withHide(),
			/* Method:      */ new ColumnSetupMetricsImpl().withHide(),
			/* Message:     */ new ColumnSetupMetricsImpl(  1, ColumnWidthType.RELATIVE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.LEFT).withRowEscapeCode( ANSI_MESSAGE ).withRowMoreTextMode( MoreTextMode.LEFT ).withName( MESSAGE ),
			/* Exception:   */ new ColumnSetupMetricsImpl(  1, ColumnWidthType.RELATIVE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.NONE ).withRowHorizAlignTextMode( HorizAlignTextMode.LEFT).withRowEscapeCode( ANSI_EXCEPTION ).withRowSplitTextMode( SplitTextMode.AT_FIRST_END_OF_LINE ).withName( EXCEPTION ),
		};
		// @formatter:on
	// @formatter:off
		private static final ColumnSetupMetrics[] ANALYST_COLUMN_METRICS = new ColumnSetupMetrics[] {
			/* Line number: */ new ColumnSetupMetricsImpl( 10, ColumnWidthType.ABSOLUTE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.RIGHT ).withRowEscapeCode( ANSI_LINE_NUMBER ).withRowMoreTextMode( MoreTextMode.NONE ).withName(LINE_NUMBER),
			/* Date:        */ new ColumnSetupMetricsImpl( 19, ColumnWidthType.ABSOLUTE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.RIGHT ).withRowEscapeCode( ANSI_DATE ).withRowMoreTextMode( MoreTextMode.NONE ).withName(DATE),
			/* Priority:    */ new ColumnSetupMetricsImpl(  7, ColumnWidthType.ABSOLUTE ).withTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.CENTER).withRowEscapeCodeFactory( LOG_PRIORITY_ESC_CODE_FACTORY ).withRowMoreTextMode( MoreTextMode.NONE ).withName(PRIORITY),
			/* Thread:      */ new ColumnSetupMetricsImpl( 10, ColumnWidthType.ABSOLUTE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.LEFT ).withRowEscapeCode( ANSI_THREAD ).withRowSplitTextMode( SplitTextMode.AT_FIXED_WIDTH ).withName(THREAD),
			/* Session:     */ new ColumnSetupMetricsImpl(  8, ColumnWidthType.ABSOLUTE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.CENTER ).withRowEscapeCode( ANSI_CORRELATION_ID ).withRowSplitTextMode( SplitTextMode.AT_FIXED_WIDTH ).withName(SESSION),
			/* Request:     */ new ColumnSetupMetricsImpl(  8, ColumnWidthType.ABSOLUTE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.CENTER ).withRowEscapeCode( ANSI_CORRELATION_ID ).withRowSplitTextMode( SplitTextMode.AT_FIXED_WIDTH ).withName(REQUEST),
			/* Class:       */ new ColumnSetupMetricsImpl(  1, ColumnWidthType.RELATIVE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.LEFT ).withRowEscapeCode(ANSI_CLASS ).withRowSplitTextMode( SplitTextMode.AT_FIXED_WIDTH ).withName(CLASS),
			/* Code line:   */ new ColumnSetupMetricsImpl(  4, ColumnWidthType.ABSOLUTE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.RIGHT ).withRowEscapeCode( ANSI_CLASS ).withRowSplitTextMode( SplitTextMode.AT_FIXED_WIDTH ).withName(CODE_LINE),
			/* Method:      */ new ColumnSetupMetricsImpl( 30, ColumnWidthType.ABSOLUTE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.LEFT ).withRowEscapeCode( ANSI_METHOD ).withRowSplitTextMode( SplitTextMode.AT_FIXED_WIDTH ).withName(METHOD),
			/* Message:     */ new ColumnSetupMetricsImpl(  2, ColumnWidthType.RELATIVE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.RIGHT ).withRowHorizAlignTextMode( HorizAlignTextMode.LEFT ).withRowEscapeCode( ANSI_MESSAGE ).withRowSplitTextMode( SplitTextMode.AT_FIRST_END_OF_LINE ).withName(MESSAGE),
			/* Exception:   */ new ColumnSetupMetricsImpl(  1, ColumnWidthType.RELATIVE ).withHeaderTextFormatMode( TextFormatMode.CELL ).withHeaderHorizAlignTextMode( HorizAlignTextMode.CENTER ).withHeaderEscapeCode( ANSI_HEADER ).withHeaderMoreTextMode( MoreTextMode.NONE ).withRowHorizAlignTextMode( HorizAlignTextMode.LEFT).withRowEscapeCode( ANSI_EXCEPTION ).withRowSplitTextMode( SplitTextMode.AT_FIRST_END_OF_LINE ).withName( EXCEPTION ),
		};
		// @formatter:on

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new console logger header impl.
	 */
	public ConsoleLoggerHeaderImpl() {
		this( PrintStackTrace.EXPLODED, DEVELOPER_COLUMN_METRICS );
	}

	/**
	 * Instantiates a new console logger header impl.
	 *
	 * @param aLayout the a layout
	 */
	public ConsoleLoggerHeaderImpl( ColumnLayout aLayout ) {
		this( PrintStackTrace.EXPLODED, toColumnSetupMetrics( aLayout ) );
	}

	/**
	 * To column setup metrics.
	 *
	 * @param aLayout the a layout
	 * @return the column setup metrics[]
	 */
	private static ColumnSetupMetrics[] toColumnSetupMetrics( ColumnLayout aLayout ) {
		switch ( aLayout ) {
		case SUPERUSER:
			return SUPERUSER_COLUMN_METRICS;
		case ANALYST:
			return ANALYST_COLUMN_METRICS;
		case GRANDPA:
			return GRANDPA_COLUMN_METRICS;
		case DEVOPS:
			return DEVOPS_COLUMN_METRICS;
		case END_USER:
			return ENDUSER_COLUMN_METRICS;
		case DEVELOPER:
		default:
			return DEVELOPER_COLUMN_METRICS;
		}
	}

	/**
	 * Instantiates a new console logger header impl.
	 *
	 * @param aPrintStackTrace the a print stack trace
	 * @param aColumnSetupMetrics the a column setup metrics
	 */
	@SuppressWarnings("unchecked")
	public ConsoleLoggerHeaderImpl( PrintStackTrace aPrintStackTrace, ColumnSetupMetrics[] aColumnSetupMetrics ) {
		// @formatter:off
		super( 
			new FormattedColumnDecorator<>( LoggerField.LOG_LINE_NUMBER.getColumn(), aColumnSetupMetrics[0]),
			new FormattedColumnDecorator<>( LoggerField.LOG_DATE.getColumn(), aColumnSetupMetrics[1]),
			new FormattedColumnDecorator<>( LoggerField.LOG_PRIORITY.getColumn(), aColumnSetupMetrics[2]),
			new FormattedColumnDecorator<>( LoggerField.LOG_THREAD_NAME.getColumn(), aColumnSetupMetrics[3]),
			new FormattedColumnDecorator<>( LoggerField.LOG_SESSION_ID.getColumn(), aColumnSetupMetrics[4]),
			new FormattedColumnDecorator<>( LoggerField.LOG_REQUEST_ID.getColumn(), aColumnSetupMetrics[5]),
			new FormattedColumnDecorator<>( LoggerField.LOG_FULLY_QUALIFIED_CLASS_NAME.getColumn(), aColumnSetupMetrics[6]),
			new FormattedColumnDecorator<>( LoggerField.LOG_CLASS_LINE_NUMBER.getColumn(), aColumnSetupMetrics[7]),
			new FormattedColumnDecorator<>( LoggerField.LOG_METHODE_NAME.getColumn(), aColumnSetupMetrics[8]),
			new FormattedColumnDecorator<>( LoggerField.LOG_MESSAGE.getColumn(), aColumnSetupMetrics[9]),
			new FormattedColumnDecorator<>( LoggerField.toExceptionLoggerField( aPrintStackTrace ).getColumn(),  aColumnSetupMetrics[10])
		);
		// @formatter:on
		setResetEscapeCode( ANSI_RESET );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////
}
