// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////
package org.refcodes.logger.alt.console;

import org.refcodes.data.ConsoleDimension;
import org.refcodes.data.EnvironmentProperty;
import org.refcodes.data.Field;
import org.refcodes.data.SystemProperty;
import org.refcodes.logger.ColumnLayout;
import org.refcodes.logger.IllegalRecordRuntimeException;
import org.refcodes.logger.LogPriority;
import org.refcodes.logger.LoggerField;
import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.UnexpectedLogRuntimeException;
import org.refcodes.runtime.OperatingSystem;
import org.refcodes.runtime.SystemUtility;
import org.refcodes.tabular.ColumnMismatchException;
import org.refcodes.tabular.FormattedColumn;
import org.refcodes.tabular.FormattedHeader;
import org.refcodes.tabular.Record;
import org.refcodes.textual.MoreTextMode;
import org.refcodes.textual.TableBuilder;
import org.refcodes.textual.TableBuilderImpl;
import org.refcodes.textual.TableStyle;

/**
 * Specialized subclass of the {@link FormattedLoggerImpl} for handling
 * {@link RuntimeLogger}'s logs with ANSI Escape-Codes.
 */
public class ConsoleLoggerImpl extends FormattedLoggerImpl<Object> implements ConsoleLogger {

	// /////////////////////////////////////////////////////////////////////////
	// ENUM:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * The Enum TableLayout.
	 */
	private enum TableLayout {
		NONE,

		MESSAGE,

		EXCEPTION
	}

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private LogPriority _lastPriority;

	private int _exceptionIndex;

	private FormattedColumn<Exception> _errorColumn;

	protected TableBuilder _errorBuilder;

	protected TableBuilder _standardBuilder;

	protected FormattedHeader<Object> _standardHeader;

	private TableLayout _lastLayout = TableLayout.NONE;

	private ColumnLayout _columnLayout;

	private boolean _isPrintSeparator = false;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new console logger impl.
	 */
	public ConsoleLoggerImpl() {
		this( toLoggerLayout( null ) );
	}

	/**
	 * Instantiates a new console logger impl.
	 *
	 * @param aLoggerLayout the a logger layout
	 */
	public ConsoleLoggerImpl( ColumnLayout aLoggerLayout ) {
		super( new ConsoleLoggerHeaderImpl( toLoggerLayout( aLoggerLayout ) ) );
		init( toLoggerLayout( aLoggerLayout ) );
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ColumnLayout getColumnLayout() {
		return _columnLayout;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setColumnLayout( ColumnLayout aColumnLayout ) {
		if ( toLoggerLayout() == null ) {
			ColumnLayout theColumnLayout = toLoggerLayout( aColumnLayout );
			if ( theColumnLayout != _columnLayout ) {
				_columnLayout = theColumnLayout;
				_errorColumn = null;
				_header = new ConsoleLoggerHeaderImpl( _columnLayout );
				init();
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void log( Record<? extends Object> aRecord ) throws IllegalRecordRuntimeException, UnexpectedLogRuntimeException {
		try {
			LogPriority thePriority = (LogPriority) LoggerField.LOG_PRIORITY.getColumn().get( aRecord );

			// Do we have an exception? |-->
			Exception theException = null;
			if ( aRecord.size() > _exceptionIndex ) {
				theException = (Exception) aRecord.get( Field.LOG_EXCEPTION.getName() );
			}
			// <--| Do we have an exception?

			// Doing logger's state brainf*ck |-->
			if ( thePriority == null || thePriority.getPriority() > LogPriority.DISCARD.getPriority() ) {
				if ( _lastPriority != thePriority || _columnLayout == ColumnLayout.ANALYST || thePriority.getPriority() >= LogPriority.WARN.getPriority() || _isPrintSeparator ) {
					_isPrintSeparator = false;
					if ( _lastLayout == TableLayout.EXCEPTION ) {
						if ( theException != null ) {
							_standardBuilder.printRowEnd( _errorBuilder );
						}
						else {
							if ( thePriority.getPriority() >= LogPriority.WARN.getPriority() ) {
								_standardBuilder.printRowEnd( _errorBuilder );
							}
							else {
								_tableBuilder.printRowEnd( _errorBuilder );
							}
						}
					}
					else if ( _lastLayout != TableLayout.NONE ) {
						_tableBuilder.printRowBegin();
					}
				}
				if ( thePriority.getPriority() >= LogPriority.WARN.getPriority() ) {
					log( aRecord, _standardHeader, _standardBuilder );
				}
				else {
					super.log( aRecord );
				}

				if ( theException != null ) {
					printException( theException );
					_lastLayout = TableLayout.EXCEPTION;
				}
				else {
					_lastLayout = TableLayout.MESSAGE;
				}
				// <--| Doing logger's state brainf*ck
			}

			// Log exceptions even if logs are disabled |-->
			else {
				if ( theException != null ) {
					System.err.println( theException.getMessage() );
					theException.printStackTrace();
				}
			}
			// <--| Log exceptions even if logs are disabled
			_lastPriority = thePriority;
		}
		catch ( ColumnMismatchException |

				ClassCastException e ) {
			throw new IllegalRecordRuntimeException( aRecord, e );
		}
	}

	/**
	 * Prints the exception.
	 *
	 * @param theException the the exception
	 */
	protected void printException( Exception theException ) {
		String theText = _errorColumn.toPrintable( theException );
		theText = theText.replaceAll( "\t", "--> " );
		_errorBuilder.printRowEnd( _standardBuilder );
		_errorBuilder.printRowContinue( theText );
		_errorBuilder.withPrintStream( _errStream );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void printSeparator() {
		_isPrintSeparator = true;
	}

	// /////////////////////////////////////////////////////////////////////////
	// UTILTY:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Determines the {@link ColumnLayout} by evaluating the
	 * {@link SystemProperty#LOGGER_LAYOUT} and the
	 * {@link EnvironmentProperty#LOGGER_LAYOUT} (in this order). If none
	 * (valid) layout was determinable, then the {@link ColumnLayout#SUPERUSER}
	 * is returned.
	 *
	 * @param aLoggerLayout the a logger layout
	 * @return The {@link ColumnLayout} to be used.
	 */
	protected static ColumnLayout toLoggerLayout( ColumnLayout aLoggerLayout ) {
		if ( aLoggerLayout != null ) {
			if ( aLoggerLayout != ColumnLayout.SUPERUSER || SystemUtility.toConsoleWidth() >= ConsoleDimension.NORM_WIDTH.getValue() ) return aLoggerLayout;
		}
		ColumnLayout theLoggerLayout = toLoggerLayout();
		if ( theLoggerLayout == null ) theLoggerLayout = ColumnLayout.SUPERUSER;
		if ( theLoggerLayout == ColumnLayout.SUPERUSER && SystemUtility.toConsoleWidth() < ConsoleDimension.NORM_WIDTH.getValue() ) return ColumnLayout.GRANDPA;
		return theLoggerLayout;
	}

	/**
	 * To logger layout.
	 *
	 * @return the column layout
	 */
	protected static ColumnLayout toLoggerLayout() {
		ColumnLayout theLoggerLayout = null;
		String theResult = SystemUtility.toPropertyValue( SystemProperty.LOGGER_LAYOUT, EnvironmentProperty.LOGGER_LAYOUT );
		if ( theResult != null ) {
			try {
				theLoggerLayout = ColumnLayout.toColumnLayout( theResult );
				if ( theLoggerLayout == null ) theLoggerLayout = ColumnLayout.SUPERUSER;
			}
			catch ( IllegalArgumentException | NullPointerException e ) {}
		}
		return theLoggerLayout;
	}

	/**
	 * Determines the {@link TableStyle} by evaluating the
	 * {@link SystemProperty#LOGGER_STYLE} and the
	 * {@link EnvironmentProperty#LOGGER_STYLE} (in this order). If none (valid)
	 * style was determinable, then a proposed style is returned.
	 * 
	 * @return The {@link TableStyle} to be used.
	 */
	protected static TableStyle toLoggerStyle() {
		String theResult = SystemUtility.toPropertyValue( SystemProperty.LOGGER_STYLE, EnvironmentProperty.LOGGER_STYLE );
		if ( theResult != null ) {
			try {
				TableStyle theTableStyle = TableStyle.toTableStyle( theResult );
				if ( theTableStyle != null ) return theTableStyle;
			}
			catch ( IllegalArgumentException | NullPointerException e ) {}
		}
		return OperatingSystem.toOperatingSystem() == OperatingSystem.WINDOWS ? TableStyle.ASCII : TableStyle.SINGLE;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	@SuppressWarnings("unchecked")
	protected void init() {
		_tableStyle = toLoggerStyle();
		if ( _errorColumn == null ) {
			_exceptionIndex = _header.indexOf( Field.LOG_EXCEPTION.getName() );
			_errorColumn = (FormattedColumn<Exception>) _header.delete( Field.LOG_EXCEPTION.getName() );
		}
		_errorBuilder = new TableBuilderImpl( _rowWidth ).addColumn().withColumnFormatMetrics( _errorColumn ).withPrintStream( _errStream ).withTableStyle( _tableStyle ).withLeftBorder( _hasLeftBorder ).withRightBorder( _hasRightBorder );
		_errorBuilder.withLineBreak( SystemUtility.toLineBreak( _rowWidth ) );
		if ( _columnLayout != null ) {
			init( _columnLayout );
		}
		super.init();
	}

	/**
	 * Inits the.
	 *
	 * @param aColumnLayout the a column layout
	 */
	private void init( ColumnLayout aColumnLayout ) {
		_columnLayout = aColumnLayout;
		_standardHeader = toVerboseHeader();
		_standardBuilder = toPreConfiguredTableBuilder( _standardHeader, OutputPrintStream.ERROR );
	}

	/**
	 * To verbose header.
	 *
	 * @return the formatted header
	 */
	private FormattedHeader<Object> toVerboseHeader() {
		FormattedHeader<Object> theVerboseHeader = new ConsoleLoggerHeaderImpl( _columnLayout );
		FormattedColumn<?> theMessageColumn = _header.get( Field.LOG_MESSAGE.getName() );
		theMessageColumn.setMoreTextMode( MoreTextMode.NONE );
		theVerboseHeader.delete( Field.LOG_EXCEPTION.getName() );
		return theVerboseHeader;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
