// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////
package org.refcodes.logger.alt.console;

import java.io.PrintStream;

import org.refcodes.logger.Logger;
import org.refcodes.logger.RuntimeLoggerSingleton;
import org.refcodes.mixin.ErrorPrintStreamAccessor.ErrorPrintStreamBuilder;
import org.refcodes.mixin.ErrorPrintStreamAccessor.ErrorPrintStreamProperty;
import org.refcodes.mixin.RowWidthAccessor.RowWidthBuilder;
import org.refcodes.mixin.RowWidthAccessor.RowWidthProperty;
import org.refcodes.mixin.StandardPrintStreamAccessor.StandardPrintStreamBuilder;
import org.refcodes.mixin.StandardPrintStreamAccessor.StandardPrintStreamProperty;
import org.refcodes.tabular.Record;
import org.refcodes.textual.TableStyle;
import org.refcodes.textual.TableStyleAccessor.TableStyleBuilder;
import org.refcodes.textual.TableStyleAccessor.TableStyleProperty;

/**
 * The {@link FormattedLogger} interface defines the methods to do pretty
 * logging to the console's output- or error-streams.
 *
 * @param <T> The type of the {@link Record} instances managed by the
 *        {@link Logger}.
 */
public interface FormattedLogger<T> extends Logger<T>, RowWidthProperty, RowWidthBuilder<FormattedLogger<?>>, TableStyleProperty, TableStyleBuilder<FormattedLogger<?>>, StandardPrintStreamProperty, StandardPrintStreamBuilder<FormattedLogger<T>>, ErrorPrintStreamProperty, ErrorPrintStreamBuilder<FormattedLogger<T>> {

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * As the underlying configuration framework used by the
	 * {@link RuntimeLoggerSingleton} cannot convert a String to an enum, we got
	 * to provide such a method ourselves. When configuring programmatically use
	 * {@link #setTableStyle(TableStyle)}, when using the
	 * "<code>runtimelogger-config.xml</code>" use the property "loggerStyle"
	 * for this method to be invoked. Valid arguments are as of the
	 * {@link TableStyle} enumeration:
	 * 
	 * <ul>
	 * <li>SINGLE</li>
	 * <li>DOUBLE</li>
	 * <li>DOUBLE_SINGLE</li>
	 * <li>SINGLE_DOUBLE</li>
	 * <li>SINGLE_DASHED</li>
	 * <li>ASCII</li>
	 * <li>BLANK</li>
	 * <li>SINGLE_BLANK</li>
	 * <li>ASCII_BLANK</li>
	 * </ul>
	 * 
	 * @param aTableStyle The style to use for the logger.
	 */
	void setLoggerStyle( String aTableStyle );

	/**
	 * Sets the escape codes.
	 *
	 * @param isEscCodesEnabled the new escape codes
	 */
	void setEscapeCodes( boolean isEscCodesEnabled );

	/**
	 * With escape codes.
	 *
	 * @param isEscCodesEnabled the is esc codes enabled
	 * @return the formatted logger
	 */
	default FormattedLogger<T> withEscapeCodes( boolean isEscCodesEnabled ) {
		setEscapeCodes( isEscCodesEnabled );
		return this;
	}

	/**
	 * Checks for escape codes.
	 *
	 * @return true, if successful
	 */
	boolean hasEscapeCodes();

	/**
	 * Checks for left border.
	 *
	 * @return true, if successful
	 */
	boolean hasLeftBorder();

	/**
	 * Sets the left border.
	 *
	 * @param hasLeftBorder the new left border
	 */
	void setLeftBorder( boolean hasLeftBorder );

	/**
	 * With left border.
	 *
	 * @param hasLeftBorder the has left border
	 * @return the formatted logger
	 */
	default FormattedLogger<T> withLeftBorder( boolean hasLeftBorder ) {
		setLeftBorder( hasLeftBorder );
		return this;
	}

	/**
	 * Checks for right border.
	 *
	 * @return true, if successful
	 */
	boolean hasRightBorder();

	/**
	 * Sets the right border.
	 *
	 * @param hasRightBorder the new right border
	 */
	void setRightBorder( boolean hasRightBorder );

	/**
	 * With right border.
	 *
	 * @param hasRightBorder the has right border
	 * @return the formatted logger
	 */
	default FormattedLogger<T> withRightBorder( boolean hasRightBorder ) {
		setRightBorder( hasRightBorder );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default FormattedLogger<T> withErrorPrintStream( PrintStream aErrorPrintStream ) {
		setErrorPrintStream( aErrorPrintStream );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default FormattedLogger<T> withStandardPrintStream( PrintStream aStandardPrintStream ) {
		setStandardPrintStream( aStandardPrintStream );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default FormattedLogger<?> withTableStyle( TableStyle aTableStyle ) {
		setTableStyle( aTableStyle );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	default FormattedLogger<?> withRowWidth( int aRowWidth ) {
		setRowWidth( aRowWidth );
		return this;
	}

}