// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.logger.alt.console;

import java.io.PrintStream;
import java.util.ArrayList;
import java.util.List;

import org.fusesource.jansi.AnsiConsole;
import org.refcodes.component.Destroyable;
import org.refcodes.data.SystemProperty;
import org.refcodes.data.Text;
import org.refcodes.logger.IllegalRecordRuntimeException;
import org.refcodes.logger.Logger;
import org.refcodes.logger.UnexpectedLogRuntimeException;
import org.refcodes.runtime.OperatingSystem;
import org.refcodes.runtime.Shell;
import org.refcodes.runtime.SystemUtility;
import org.refcodes.tabular.ColumnMismatchException;
import org.refcodes.tabular.FormattedHeader;
import org.refcodes.tabular.Header;
import org.refcodes.tabular.HeaderMismatchException;
import org.refcodes.tabular.Record;
import org.refcodes.tabular.Row;
import org.refcodes.textual.ColumnSetupMetrics;
import org.refcodes.textual.TableBuilder;
import org.refcodes.textual.TableBuilderImpl;
import org.refcodes.textual.TableStyle;

/**
 * The {@link FormattedLoggerImpl} implements the {@link Logger} interface for
 * providing logging functionality with extended pimped console output (via
 * {@link System#out} and (via sub-classing also {@link System#err}).
 * <p>
 * The {@link FormattedLoggerImpl} by default uses the most promising width in
 * characters of the system's terminal in use by calling the method
 * {@link SystemUtility#toConsoleWidth()}. In case you pass a
 * "-Dconsole.width=n" JVM argument, then your width is taken, else the actual
 * console's width is being tried to be determined. See
 * {@link SystemProperty#CONSOLE_WIDTH}.
 * 
 * You can also use the {@link #setRowWidth(int)} or {@link #withRowWidth(int)}
 * method in order to programmatically set the console's row width.
 *
 * @param <T> The type of the {@link Record} instances managed by the
 *        {@link Logger}.
 */
public class FormattedLoggerImpl<T> implements Destroyable, FormattedLogger<T> {

	// /////////////////////////////////////////////////////////////////////////
	// ENUM:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * The Enum OutputPrintStream.
	 */
	public enum OutputPrintStream {
		STANDARD,

		ERROR
	}

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	protected FormattedHeader<T> _header;

	protected TableBuilder _tableBuilder;

	protected PrintStream _stdStream = Shell.toShell() == Shell.WIN_CMD ? AnsiConsole.out : System.out;

	protected PrintStream _errStream = Shell.toShell() == Shell.WIN_CMD ? AnsiConsole.err : System.err;

	protected int _rowWidth = SystemUtility.toConsoleWidth();

	protected TableStyle _tableStyle;

	protected boolean _hasLeftBorder = false;

	protected boolean _hasRightBorder = false;

	private List<String> _columnNames;

	private boolean _isEscCodesEnabled = SystemUtility.toAnsiConsole();

	// /////////////////////////////////////////////////////////////////////////
	// STATE:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Initially enables or disables ANSI escape sequences as of detection of
	 * terminal's ANSI support. You can overrule this setting by calling
	 * {@link #setEscapeCodes(boolean)}.
	 *
	 * @param aHeader the a header
	 * @see SystemUtility#isAnsiTerminal()
	 */
	public FormattedLoggerImpl( FormattedHeader<T> aHeader ) {
		_header = aHeader;
		_tableStyle = OperatingSystem.toOperatingSystem() == OperatingSystem.WINDOWS ? TableStyle.ASCII : TableStyle.SINGLE;
		init();
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void log( Record<? extends T> aRecord ) throws IllegalRecordRuntimeException, UnexpectedLogRuntimeException {
		log( aRecord, _header, _tableBuilder );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public PrintStream getStandardPrintStream() {
		return _stdStream;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setStandardPrintStream( PrintStream aOutStream ) {
		_stdStream = aOutStream;

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public PrintStream getErrorPrintStream() {
		return _errStream;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setErrorPrintStream( PrintStream aErrStream ) {
		_errStream = aErrStream;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TableStyle getTableStyle() {
		return _tableStyle;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setTableStyle( TableStyle aTableStyle ) {
		_tableStyle = aTableStyle;
		init();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setLoggerStyle( String aTableStyleName ) {
		setTableStyle( TableStyle.valueOf( aTableStyleName ) );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setEscapeCodes( boolean isEscCodesEnabled ) {
		_isEscCodesEnabled = isEscCodesEnabled;
		TableBuilder theTableBuilder = _tableBuilder;
		if ( theTableBuilder != null ) {
			theTableBuilder.withEscapeCodes( isEscCodesEnabled );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean hasLeftBorder() {
		return _hasLeftBorder;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setLeftBorder( boolean hasLeftBorder ) {
		_hasLeftBorder = hasLeftBorder;
		init();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean hasRightBorder() {
		return _hasRightBorder;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setRightBorder( boolean hasRightBorder ) {
		_hasRightBorder = hasRightBorder;
		init();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getRowWidth() {
		return _rowWidth;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setRowWidth( int aRowWidth ) {
		_rowWidth = aRowWidth;
		init();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean hasEscapeCodes() {
		return _isEscCodesEnabled;
	}

	// /////////////////////////////////////////////////////////////////////////
	// LIFECYCLE:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void destroy() {
		if ( Shell.toShell() == Shell.WIN_CMD ) AnsiConsole.systemUninstall();
		_tableBuilder.printTail();
	}

	/**
	 * Inits the.
	 */
	protected void init() {
		if ( Shell.toShell() == Shell.WIN_CMD ) AnsiConsole.systemInstall();
		TableBuilder theTableBuilder = toPreConfiguredTableBuilder( _header, OutputPrintStream.STANDARD );
		_columnNames = new ArrayList<>();
		for ( ColumnSetupMetrics eMetrics : _header ) {
			if ( eMetrics.isVisible() ) {
				_columnNames.add( eMetrics.getName() );
			}
		}
		out: {
			for ( String eName : _columnNames ) {
				if ( eName != null && eName.length() != 0 ) {
					break out;
				}
			}
			_columnNames.clear();
			_columnNames = null;
		}

		_tableBuilder = theTableBuilder;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Creates a pre-configured {@link TableBuilder}, can be used by sub-classes
	 * in case them require additional {@link TableBuilder} instances.
	 *
	 * @param aHeader the a header
	 * @param aOutputPrintStream the a output print stream
	 * @return A pre-configured {@link TableBuilder} (as of the attributes state
	 *         of the {@link FormattedLoggerImpl}).
	 */
	protected TableBuilder toPreConfiguredTableBuilder( FormattedHeader<?> aHeader, OutputPrintStream aOutputPrintStream ) {
		TableBuilder theTableBuilder = new TableBuilderImpl( _rowWidth ).withTableStyle( _tableStyle ).withPrintStream( aOutputPrintStream == OutputPrintStream.STANDARD ? _stdStream : _errStream ).withResetEscapeCode( _header.getResetEscapeCode() ).withLeftBorder( _hasLeftBorder ).withRightBorder( _hasRightBorder ).withEscapeCodes( _isEscCodesEnabled ).withLineBreak( SystemUtility.toLineBreak( _rowWidth ) );
		for ( ColumnSetupMetrics eMetrics : aHeader ) {
			if ( eMetrics.isVisible() ) {
				theTableBuilder.addColumn().withColumnFormatMetrics( eMetrics );
			}
		}
		return theTableBuilder;
	}

	/**
	 * Prints out a log-line with regard to the provided {@link Header} unsing
	 * the provided {@link TableBuilder}.
	 *
	 * @param aRecord The record to log.
	 * @param aHeader The {@link Header} with which to determine the visibility
	 *        of the {@link Record}'s elements.
	 * @param aTableBuilder The {@link TableBuilder} to use for printing.
	 * @throws ClassCastException the class cast exception
	 */
	protected void log( Record<? extends T> aRecord, FormattedHeader<T> aHeader, TableBuilder aTableBuilder ) {

		if ( _columnNames != null ) {
			_tableBuilder.printHeaderBegin();
			_tableBuilder.printHeaderContinue( _columnNames );
			_tableBuilder.printHeaderComplete();
			_columnNames.clear();
			_columnNames = null;
		}

		try {
			Row<String> theRow = aHeader.toPrintableRow( aRecord );
			for ( int i = aHeader.size() - 1; i >= 0; i-- ) {
				if ( !aHeader.get( i ).isVisible() && theRow.size() > i ) {
					theRow.remove( i );
				}
			}
			for ( int i = 0; i < theRow.size(); i++ ) {
				if ( theRow.get( i ) == null ) {
					theRow.remove( i );
					theRow.add( i, Text.NULL_VALUE.getText() );
				}

			}
			aTableBuilder.printRowContinue( theRow );
		}
		catch ( ColumnMismatchException | ClassCastException | HeaderMismatchException e ) {
			throw new IllegalRecordRuntimeException( aRecord, e );
		}

	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
