// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.logger;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.util.Iterator;

import org.refcodes.component.CompositeComponent;
import org.refcodes.logger.impls.TestTrimLoggerImpl;

/**
 * Utility class helpful for test cases regarding the {@link TrimLogger} and in
 * conjunction with other {@link Logger} instances.
 */
public final class TestTrimLoggerUtility {

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Private empty constructor to prevent instantiation as of being a utility
	 * with just static public methods.
	 */
	private TestTrimLoggerUtility() {}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Asserts the state change invocation counts regarding the
	 * {@link CompositeComponent}'s supported state change methods.
	 * 
	 * @param aInitializeCount The according invocation count.
	 * @param aStartCount The according invocation count.
	 * @param aPauseCount The according invocation count.
	 * @param aResumeCount The according invocation count.
	 * @param aStopCount The according invocation count.
	 * @param aDestroyCount The according invocation count.
	 * @param aDecomposeCount The according invocation count.
	 * @param aFlushCount The according invocation count.
	 */
	public static void doAssertCompositeComponentInvocation( int aInitializeCount, int aStartCount, int aPauseCount, int aResumeCount, int aStopCount, int aDestroyCount, int aDecomposeCount, int aFlushCount, TestTrimLoggerImpl[] aTrimLoggers ) {
		for ( int i = 0; i < aTrimLoggers.length; i++ ) {
			assertEquals( aInitializeCount, aTrimLoggers[i].getInitializeCount() );
			assertEquals( aStartCount, aTrimLoggers[i].getStartCount() );
			assertEquals( aPauseCount, aTrimLoggers[i].getPauseCount() );
			assertEquals( aResumeCount, aTrimLoggers[i].getResumeCount() );
			assertEquals( aStopCount, aTrimLoggers[i].getStopCount() );
			assertEquals( aDestroyCount, aTrimLoggers[i].getDestroyCount() );
			assertEquals( aDecomposeCount, aTrimLoggers[i].getDecomposeCount() );
			assertEquals( aFlushCount, aTrimLoggers[i].getFlushCount() );
		}
	}

	/**
	 * Asserts the method invocation counts regarding the {@link TrimLogger} 
	 * 'supported log related methods.
	 * 
	 * @param aOverallLogCount The according invocation count.
	 * @param aOverallFindLogsCount The according invocation count.
	 * @param aOverallDeleteLogsCount The according invocation count.
	 * @param aOverallClearCount The according invocation count.
	 */
	public static void doAssertLoggerInvocation( int aOverallLogCount, int aOverallFindLogsCount, int aOverallDeleteLogsCount, int aOverallClearCount, TestTrimLoggerImpl[] aTrimLoggers ) {
		int theOverallLogCount = 0;
		int theOverallFindLogsCount = 0;
		int theOverallDeleteLogsCount = 0;
		int theOverallClearCount = 0;
		for ( int i = 0; i < aTrimLoggers.length; i++ ) {
			theOverallLogCount += aTrimLoggers[i].getLogCount();
			theOverallFindLogsCount += aTrimLoggers[i].getFindLogsCount();
			theOverallDeleteLogsCount += aTrimLoggers[i].getDeleteLogsCount();
			theOverallClearCount += aTrimLoggers[i].getClearCount();
		}
		assertEquals( aOverallLogCount, theOverallLogCount );
		assertEquals( aOverallFindLogsCount, theOverallFindLogsCount );
		assertEquals( aOverallDeleteLogsCount, theOverallDeleteLogsCount );
		assertEquals( aOverallClearCount, theOverallClearCount );
	}

	/**
	 * Asserts the method invocation on all loggers regarding the
	 * {@link TrimLogger} 'supported log related methods.
	 * 
	 * @param hasLog True in case the according method is expected to have been
	 *        invoked on each logger.
	 * @param hasFindLogs True in case the according method is expected to have
	 *        been invoked on each logger.
	 * @param hasDeleteLogs True in case the according method is expected to
	 *        have been invoked on each logger.
	 * @param hasClear True in case the according method is expected to have
	 *        been invoked on each logger.
	 */
	public static void doAssertLoggerTouch( boolean hasLog, boolean hasFindLogs, boolean hasDeleteLogs, boolean hasClear, TestTrimLoggerImpl[] aTrimLoggers ) {
		for ( int i = 0; i < aTrimLoggers.length; i++ ) {
			assertTrue( aTrimLoggers[i].getClearCount() != 0 == hasClear );
			assertTrue( aTrimLoggers[i].getDeleteLogsCount() != 0 == hasDeleteLogs );
			assertTrue( aTrimLoggers[i].getFindLogsCount() != 0 == hasFindLogs );
			assertTrue( aTrimLoggers[i].getLogCount() != 0 == hasLog );
		}
	}

	/**
	 * Iterates through the {@link Iterator} and counts the elements, returns
	 * the element count.
	 * 
	 * @return The number of elements originally contained in the
	 *         {@link Iterator}.
	 */
	public static int getSize( Iterator<?> e ) {
		int theSize = 0;
		while ( e.hasNext() ) {
			theSize++;
			e.next();
		}
		return theSize;
	}
}
