// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.logger;

import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import org.refcodes.controlflow.InvocationStrategy;
import org.refcodes.criteria.AndCriteria;
import org.refcodes.criteria.Criteria;
import org.refcodes.criteria.CriteriaUtility;
import org.refcodes.criteria.EqualWithCriteria;
import org.refcodes.criteria.ExpressionQueryFactoryImpl;
import org.refcodes.criteria.OrCriteria;
import org.refcodes.tabular.Column;
import org.refcodes.tabular.Header;
import org.refcodes.tabular.Record;
import org.refcodes.tabular.Records;
import org.refcodes.textual.VerboseTextBuilder;

/**
 * The {@link AbstractPartedQueryLogger} extends the
 * {@link AbstractPartedLogger} with the functionality of a {@link QueryLogger}.
 * Any query operations, such as {@link #findLogs(Criteria)}, are targeted at
 * that partition containing the queried data. For this to work, the query must
 * obey some rules: The query is to contain an {@link EqualWithCriteria}
 * addressing the partition in an unambiguous way; by being part of a root level
 * {@link AndCriteria} or an unambiguously nested {@link AndCriteria} hierarchy.
 * More than one partition gets detected when unambiguous {@link OrCriteria} are
 * applied to the partition {@link Criteria}. In such cases, the query is
 * addressed to the potential partitions. In case it was not possible to
 * identify any partitions, then as a fallback, all partitions are queried.
 * Query results are taken from from the the invoked partitions (in normal cases
 * this would be a single partition) round robin. the first result is taken from
 * the first queried partition's result set ({@link Records}), the next result
 * from the next queried partition and so on to start over again with the first
 * queried partition. Round robin has been used to prevent invalidation of
 * physical data sinks's result sets as of timeouts. (you may specify the actual
 * {@link Record} type (generic parameter) accepted by the {@link QueryLogger}
 * instances to enforce a dedicated {@link Record} type which contains the
 * required partition {@link Column})
 *
 * @param <L> The type of the {@link QueryLogger} to be created.
 * @param <T> The type of the {@link Record} instances managed by the
 *        {@link Logger}.
 * @param <P> The type of the {@link Column}'s value used for partitioning the
 *        {@link QueryLogger}.
 */
abstract class AbstractPartedQueryLogger<L extends QueryLogger<T>, T, P extends T> extends AbstractPartedLogger<L, T, P> implements QueryLogger<T> {

	private static final InvocationStrategy RETRIEVAL_STRATEGY = InvocationStrategy.ROUND_ROBIN;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new abstract parted query logger.
	 *
	 * @param aPartitionColumn the partition column
	 * @param aLoggerFactory the logger factory
	 * @param isPartitionAutoInitialize the is partition auto initialize
	 * 
	 * @see AbstractPartedLogger#AbstractPartedLogger(Column, LoggerFactory,
	 *      boolean)
	 */
	public AbstractPartedQueryLogger( Column<P> aPartitionColumn, LoggerFactory<L> aLoggerFactory, boolean isPartitionAutoInitialize ) {
		super( aPartitionColumn, aLoggerFactory, isPartitionAutoInitialize );
	}

	/**
	 * Instantiates a new abstract parted query logger.
	 *
	 * @param aPartitionColumn the partition column
	 * @param aDefaultLoggerName the default logger name
	 * @param aLoggerFactory the logger factory
	 * @param isPartitionAutoInitialize the is partition auto initialize
	 * 
	 * @see AbstractPartedLogger#AbstractPartedLogger(Column, String,
	 *      LoggerFactory, boolean)
	 */
	public AbstractPartedQueryLogger( Column<P> aPartitionColumn, String aDefaultLoggerName, LoggerFactory<L> aLoggerFactory, boolean isPartitionAutoInitialize ) {
		super( aPartitionColumn, aDefaultLoggerName, aLoggerFactory, isPartitionAutoInitialize );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Records<T> findLogs() {
		return findLogs( null, null, -1 );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Records<T> findLogs( int aLimit ) {
		return findLogs( null, null, aLimit );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Records<T> findLogs( final Criteria aCriteria ) {
		return findLogs( aCriteria, null, -1 );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Records<T> findLogs( Header<T> aHeader, int aLimit ) {
		return findLogs( null, aHeader, aLimit );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Records<T> findLogs( final Criteria aCriteria, int aLimit ) {
		return findLogs( aCriteria, null, aLimit );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Records<T> findLogs( final Criteria aCriteria, final Header<T> aHeader ) {
		return findLogs( aCriteria, aHeader, -1 );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Records<T> findLogs( Criteria aCriteria, Header<T> aHeader, int aLimit ) {

		Set<P> thePartitions = getPartitions( aCriteria );

		// ---------------------------------------------------------------------
		// Single partition to be addressed:
		// ---------------------------------------------------------------------
		if ( thePartitions.size() == 1 ) {
			P ePartition = thePartitions.iterator().next();
			L theLogger = getPartitionLogger( ePartition );
			if ( theLogger != null ) {
				// Criteria, , aPartitions
				aCriteria = CriteriaUtility.doRemovePartitionCriteria( aCriteria, getPartitionColumn().getKey(), thePartitions );
				return theLogger.findLogs( aCriteria, aHeader, aLimit );
			}
			LOGGER.warn( "No logger found for partition \"" + getPartitionUid( ePartition ) + "\": Now querying all partitions (fallback)!" );
		}
		else if ( thePartitions.size() == 0 && getFallbackLogger() != null ) {
			LOGGER.debug( "No partition found for provided criteria, now querying default logger instance to find logs in question!" );
			return getFallbackLogger().findLogs( aCriteria, aHeader, aLimit );
		}

		// #####################################################################
		// In happy cases we should end the query here! Examine the logs in
		// order to determine on how to improve your partitioning (partitioning)
		// or your queries in order to always target at just one partition!
		// #####################################################################

		Collection<L> theLoggers = getPartitionLoggers( thePartitions );

		// -------------
		// Do the query:
		// -------------
		return LoggerUtility.findLogs( aCriteria, aHeader, aLimit, theLoggers, RETRIEVAL_STRATEGY );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Retrieves a collection with {@link Logger} instances managing the
	 * provided partitions.
	 * 
	 * @param aPartitions The partitions for which to get the {@link Logger}
	 *        instances.
	 * 
	 * @return A {@link Collection} with none to many {@link Logger} instances
	 */
	protected Collection<L> getPartitionLoggers( Set<P> aPartitions ) {

		Collection<L> theLoggers;
		// ---------------------------------------------------------------------
		// None partitions detected at all:
		// ---------------------------------------------------------------------
		if ( aPartitions.isEmpty() ) {
			theLoggers = getLoggers();
			LOGGER.warn( "None partitions at all detected for the query (criteria): Now querying all partitions (fallback)!" );
		}
		else {
			// -----------------------------------------------------------------
			// Multiple partitions to be addressed:
			// -----------------------------------------------------------------
			theLoggers = new HashSet<L>();
			if ( aPartitions.size() != 1 ) {
				L ePartitionLogger;
				for ( P ePartition : aPartitions ) {
					ePartitionLogger = getPartitionLogger( ePartition );
					if ( ePartitionLogger != null ) {
						theLoggers.add( ePartitionLogger );
					}
					else {
						LOGGER.warn( "Determined not exisiting partition " + getPartitionUid( ePartition ) + " to be addressed for the query (criteria), no logger instance present for that partition! Now querying all partitions (fallback)!" );
					}
				}
			}

			// -----------------------------------------------------------------
			// All partitions to be addressed:
			// -----------------------------------------------------------------
			if ( theLoggers.size() == 0 || theLoggers.size() != aPartitions.size() ) {
				theLoggers = getLoggers();
				LOGGER.warn( "None existing partitions detected (partitions with no logger) for the query (criteria): Now querying all partitions (fallback)!" );
			}
		}
		return theLoggers;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Tries to determine the partitions being addressed by the query
	 * represented by the given {@link Criteria}.
	 * -------------------------------------------------------------------------
	 * CAUTION: The query is to contain an {@link EqualWithCriteria} addressing
	 * the partition in an unambiguous way; by being part of a root level
	 * {@link AndCriteria} or an unambiguously nested {@link AndCriteria}
	 * hierarchy. More than one partition gets detected when unambiguous
	 * {@link OrCriteria} are applied to the partition {@link Criteria}. In such
	 * cases, the query is addressed to the potential partitions.
	 * -------------------------------------------------------------------------
	 * 
	 * @param aCriteria The {@link Criteria} query from which to determine the
	 *        partitions.
	 * 
	 * @return A {@link Set} containing the detected partitions.
	 */
	private Set<P> getPartitions( Criteria aCriteria ) {
		Set<P> thePartitions = CriteriaUtility.getPartitions( aCriteria, getPartitionColumn().getKey(), (Class<P>) getPartitionColumn().getType() );

		if ( LOGGER.isLogDebug() ) {
			String theQuery;
			if ( aCriteria == null ) {
				theQuery = "(null)";
			}
			else {
				ExpressionQueryFactoryImpl theExpressionQueryFactory = new ExpressionQueryFactoryImpl();
				theQuery = theExpressionQueryFactory.fromCriteria( aCriteria );
			}
			LOGGER.debug( "Partitions " + new VerboseTextBuilder().withElements( thePartitions ).toString() + " determined for (raw) query is: \"" + theQuery + "\"" );
		}

		if ( thePartitions.size() > 1 ) {
			LOGGER.warn( "Determined <" + thePartitions.size() + "> partitions to be addresses by the query (criteria), the number of partitions to be targeted at is greater than one, probably you have a bad partitioning (partitioning) or query (criteria) strategy!" );

		}
		return thePartitions;
	}
}
