// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.logger;

import java.text.MessageFormat;

import org.refcodes.mixin.NameAccessor;
import org.refcodes.properties.PolyglotProperties;
import org.refcodes.properties.ext.runtime.RuntimeProperties;

/**
 * Plain simple interface for logging out runtime information generated by
 * software systems. The {@link org.refcodes.logger.RuntimeLogger} defines a
 * plain simple interface for logging out runtime information generated by
 * software systems. The {@link org.refcodes.logger.RuntimeLoggerImpl}
 * implementation takes care of logging out the class and the method generating
 * a log line. The {@link org.refcodes.logger.RuntimeLoggerImpl} actually takes
 * a {@link org.refcodes.logger.Logger} instance; which implementation to take
 * is up to you:
 * <ul>
 * <li>SystemLogger</li>
 * <li>AsyncLogger</li>
 * <li>ConsoleLogger (ConsoleLoggerImpl)</li>
 * <li>IoLogger</li>
 * <li>SimpleDbLoggerImpl</li>
 * <li>SimpleDbLoggerImpl</li>
 * <li>Slf4jLogger</li>
 * </ul>
 * (you find the various {@link org.refcodes.logger.Logger} implementations in
 * the refcodes-logger-alt artifact's modules) Use the factory (the factory is
 * implemented as a singleton)
 * {@link org.refcodes.logger.RuntimeLoggerFactorySingleton} retrieving
 * {@link org.refcodes.logger.RuntimeLogger} instances configured by a
 * "<code>runtimelogge.ini</code>" file using the {@link RuntimeProperties} as
 * well as the {@link PolyglotProperties}. This means you can use YAML, TOML,
 * INI, PROPERTIES, XML or JSON notations. A SLF4J binding exists in the
 * refcodes-logger-ext-slf4j" artifact which binds the refcodes-logger framework
 * to SLF4J enabling to log all your SLF4J logs out by the refcodes-logger
 * framework, e.g. to a SimpleDbLoggerImpl. The
 * {@link org.refcodes.logger.RuntimeLogger} may also be configured with one of
 * the below mentioned loggers.
 * 
 * @author steiner
 */
public interface RuntimeLogger extends MessageLogger<LogPriority>, LogPriorityAccessor, NameAccessor, LogDecorator {

	/**
	 * The filename of the file on the classpath containing the config for the
	 * runtime logger.
	 */
	String RUNTIME_LOGGER_CONFIG = "runtimelogger";

	/**
	 * The root logger is identified by this XML element path.
	 */
	String ROOT_LOGGER_ELEMENT_PATH = "root";

	// /////////////////////////////////////////////////////////////////////////
	// GENERIC METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Creates a new log message.
	 * 
	 * @param aPriority The level of the message.
	 * @param aMessage The message to be logged.
	 * @param aThrowable An exception that the message should contain.
	 */
	void log( LogPriority aPriority, String aMessage, Throwable aThrowable );

	/**
	 * Creates a new log message. The placeholders in the message {@link String}
	 * are being replaced by the provided arguments. Implementations of this
	 * interface are recommended to use Java's {@link MessageFormat}'s syntax,
	 * the placeholder being replaced by the first argument is identified by
	 * "{0}" (without the quotes), the second one by "{1}", the third one by
	 * "{3}" and so on: Given the message to be "{0} + {1} = {2}" and the three
	 * argument be x, y and x + y, then "{0}" is replaced by the value of x,
	 * "{1}" is replaced by the value of y and "{2}" is replaced by the value of
	 * x + y.
	 * 
	 * @param aPriority The level of the message.
	 * @param aMessage The message to be logged.
	 * @param aThrowable An exception that the message should contain.
	 * @param aArguments The arguments used when replacing the placeholders.
	 */
	void log( LogPriority aPriority, String aMessage, Throwable aThrowable, Object... aArguments );

	/**
	 * Determines whether the {@link RuntimeLogger} is being configured to log
	 * messages of the according {@link LogPriority}. This method is useful to
	 * prevent the generation of vast amounts of log message {@link String}
	 * objects (and according processing) for the according {@link LogPriority}
	 * in case them messages of that {@link LogPriority} are not logged at all.
	 * 
	 * @param aPriority The {@link LogPriority} to test whether the
	 *        {@link RuntimeLogger} does log messages of that
	 *        {@link LogPriority}.
	 * 
	 * @return True in case the {@link RuntimeLogger} does log messages of that
	 *         {@link LogPriority}.
	 */
	boolean isLog( LogPriority aPriority );

	// /////////////////////////////////////////////////////////////////////////
	// CONVENIANCE METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Logs a log message with log priority {@link LogPriority#TRACE}.
	 * 
	 * @param aMessage The message to be logged.
	 */
	void trace( String aMessage );

	/**
	 * Logs a log message with log priority {@link LogPriority#TRACE}. The
	 * placeholders in the message {@link String} are being replaced by the
	 * provided arguments. Implementations of this interface are recommended to
	 * use Java's {@link MessageFormat}'s syntax, the placeholder being replaced
	 * by the first argument is identified by "{0}" (without the quotes), the
	 * second one by "{1}", the third one by "{3}" and so on: Given the message
	 * to be "{0} + {1} = {2}" and the three argument be x, y and x + y, then
	 * "{0}" is replaced by the value of x, "{1}" is replaced by the value of y
	 * and "{2}" is replaced by the value of x + y.
	 * 
	 * @param aMessage The message to be logged.
	 * @param aArguments The arguments used when replacing the placeholders.
	 */
	void trace( String aMessage, Object... aArguments );

	/**
	 * Determines whether the {@link RuntimeLogger} is being configured to log
	 * comments ({@link LogPriority#TRACE}). This method is useful to prevent
	 * the generation of vast amounts of log message {@link String} objects (and
	 * according processing) for the according {@link LogPriority} in case them
	 * messages of that {@link LogPriority} are not logged at all.
	 * 
	 * @return True in case comments are logged.
	 */
	boolean isLogTrace();

	/**
	 * Logs a log message with log priority {@link LogPriority#DEBUG}. "Info
	 * useful to developers for debugging the application, not useful during
	 * operations." See also http://en.wikipedia.org/wiki/Syslog
	 *
	 * @param aMessage The message to be logged.
	 */
	void debug( String aMessage );

	/**
	 * Logs a log message with log priority {@link LogPriority#DEBUG}. "Info
	 * useful to developers for debugging the application, not useful during
	 * operations." The placeholders in the message {@link String} are being
	 * replaced by the provided arguments. Implementations of this interface are
	 * recommended to use Java's {@link MessageFormat}'s syntax, the placeholder
	 * being replaced by the first argument is identified by "{0}" (without the
	 * quotes), the second one by "{1}", the third one by "{3}" and so on: Given
	 * the message to be "{0} + {1} = {2}" and the three argument be x, y and x
	 * + y, then "{0}" is replaced by the value of x, "{1}" is replaced by the
	 * value of y and "{2}" is replaced by the value of x + y. See also
	 * http://en.wikipedia.org/wiki/Syslog
	 *
	 * @param aMessage The message to be logged.
	 * @param aArguments The arguments used when replacing the placeholders.
	 */
	void debug( String aMessage, Object... aArguments );

	/**
	 * Determines whether the {@link RuntimeLogger} is being configured to log
	 * debug messages ({@link LogPriority#DEBUG}). This method is useful to
	 * prevent the generation of vast amounts of log message {@link String}
	 * objects (and according processing) for the according {@link LogPriority}
	 * in case them messages of that {@link LogPriority} are not logged at all.
	 * 
	 * @return True in case debug messages are logged.
	 */
	boolean isLogDebug();

	/**
	 * Logs a log message with log priority {@link LogPriority#INFO}.
	 * "Additional information which might be useful for some stability period."
	 * See also http://en.wikipedia.org/wiki/Syslog
	 *
	 * @param aMessage The message to be logged.
	 */
	void info( String aMessage );

	/**
	 * Logs a log message with log priority {@link LogPriority#INFO}.
	 * "Additional information which might be useful for some stability period."
	 * The placeholders in the message {@link String} are being replaced by the
	 * provided arguments. Implementations of this interface are recommended to
	 * use Java's {@link MessageFormat}'s syntax, the placeholder being replaced
	 * by the first argument is identified by "{0}" (without the quotes), the
	 * second one by "{1}", the third one by "{3}" and so on: Given the message
	 * to be "{0} + {1} = {2}" and the three argument be x, y and x + y, then
	 * "{0}" is replaced by the value of x, "{1}" is replaced by the value of y
	 * and "{2}" is replaced by the value of x + y. See also
	 * http://en.wikipedia.org/wiki/Syslog
	 *
	 * @param aMessage The message to be logged.
	 * @param aArguments The arguments used when replacing the placeholders.
	 */
	void info( String aMessage, Object... aArguments );

	/**
	 * Determines whether the {@link RuntimeLogger} is being configured to log
	 * infos ({@link LogPriority#INFO}). This method is useful to prevent the
	 * generation of vast amounts of log message {@link String} objects (and
	 * according processing) for the according {@link LogPriority} in case them
	 * messages of that {@link LogPriority} are not logged at all.
	 * 
	 * @return True in case infos are logged.
	 */
	boolean isLogInfo();

	/**
	 * Logs a log message with log priority {@link LogPriority#NOTICE}. "Events
	 * that are unusual but not error conditions - might be summarized in an
	 * email to developers or admins to spot potential problems - no immediate
	 * action required." See also http://en.wikipedia.org/wiki/Syslog
	 *
	 * @param aMessage The message to be logged.
	 */
	void notice( String aMessage );

	/**
	 * Logs a log message with log priority {@link LogPriority#NOTICE}. "Events
	 * that are unusual but not error conditions - might be summarized in an
	 * email to developers or admins to spot potential problems - no immediate
	 * action required." The placeholders in the message {@link String} are
	 * being replaced by the provided arguments. Implementations of this
	 * interface are recommended to use Java's {@link MessageFormat}'s syntax,
	 * the placeholder being replaced by the first argument is identified by
	 * "{0}" (without the quotes), the second one by "{1}", the third one by
	 * "{3}" and so on: Given the message to be "{0} + {1} = {2}" and the three
	 * argument be x, y and x + y, then "{0}" is replaced by the value of x,
	 * "{1}" is replaced by the value of y and "{2}" is replaced by the value of
	 * x + y. See also http://en.wikipedia.org/wiki/Syslog
	 *
	 * @param aMessage The message to be logged.
	 * @param aArguments The arguments used when replacing the placeholders.
	 */
	void notice( String aMessage, Object... aArguments );

	/**
	 * Determines whether the {@link RuntimeLogger} is being configured to log
	 * notices ({@link LogPriority#NOTICE}). This method is useful to prevent
	 * the generation of vast amounts of log message {@link String} objects (and
	 * according processing) for the according {@link LogPriority} in case them
	 * messages of that {@link LogPriority} are not logged at all.
	 * 
	 * @return True in case infos are logged.
	 */
	boolean isLogNotice();

	/**
	 * Logs a log message with log priority {@link LogPriority#WARN}. "Warning
	 * messages, not an error, but indication that an error will occur if action
	 * is not taken, e.g. file system 85% full - each item must be resolved
	 * within a given time." See also http://en.wikipedia.org/wiki/Syslog
	 *
	 * @param aMessage The message to be logged.
	 */
	void warn( String aMessage );

	/**
	 * Logs a log message with log priority {@link LogPriority#WARN}. "Warning
	 * messages, not an error, but indication that an error will occur if action
	 * is not taken, e.g. file system 85% full - each item must be resolved
	 * within a given time." The placeholders in the message {@link String} are
	 * being replaced by the provided arguments. Implementations of this
	 * interface are recommended to use Java's {@link MessageFormat}'s syntax,
	 * the placeholder being replaced by the first argument is identified by
	 * "{0}" (without the quotes), the second one by "{1}", the third one by
	 * "{3}" and so on: Given the message to be "{0} + {1} = {2}" and the three
	 * argument be x, y and x + y, then "{0}" is replaced by the value of x,
	 * "{1}" is replaced by the value of y and "{2}" is replaced by the value of
	 * x + y. See also http://en.wikipedia.org/wiki/Syslog
	 *
	 * @param aMessage The message to be logged.
	 * @param aArguments The arguments used when replacing the placeholders.
	 */
	void warn( String aMessage, Object... aArguments );

	/**
	 * Logs a log message with log priority {@link LogPriority#WARN}. "Warning
	 * messages, not an error, but indication that an error will occur if action
	 * is not taken, e.g. file system 85% full - each item must be resolved
	 * within a given time." See also http://en.wikipedia.org/wiki/Syslog
	 *
	 * @param aMessage The message to be logged.
	 * @param aThrowable An exception that the message should contain.
	 */
	void warn( String aMessage, Throwable aThrowable );

	/**
	 * Logs a log message with log priority {@link LogPriority#WARN}. "Warning
	 * messages, not an error, but indication that an error will occur if action
	 * is not taken, e.g. file system 85% full - each item must be resolved
	 * within a given time." The placeholders in the message {@link String} are
	 * being replaced by the provided arguments. Implementations of this
	 * interface are recommended to use Java's {@link MessageFormat}'s syntax,
	 * the placeholder being replaced by the first argument is identified by
	 * "{0}" (without the quotes), the second one by "{1}", the third one by
	 * "{3}" and so on: Given the message to be "{0} + {1} = {2}" and the three
	 * argument be x, y and x + y, then "{0}" is replaced by the value of x,
	 * "{1}" is replaced by the value of y and "{2}" is replaced by the value of
	 * x + y. See also http://en.wikipedia.org/wiki/Syslog
	 *
	 * @param aMessage The message to be logged.
	 * @param aThrowable An exception that the message should contain.
	 * @param aArguments The arguments used when replacing the placeholders.
	 */
	void warn( String aMessage, Throwable aThrowable, Object... aArguments );

	/**
	 * Determines whether the {@link RuntimeLogger} is being configured to log
	 * warnings ({@link LogPriority#WARN}). This method is useful to prevent the
	 * generation of vast amounts of log message {@link String} objects (and
	 * according processing) for the according {@link LogPriority} in case them
	 * messages of that {@link LogPriority} are not logged at all.
	 * 
	 * @return True in case warnings are logged.
	 */
	boolean isLogWarn();

	/**
	 * Logs a log message with log priority {@link LogPriority#ERROR}.
	 * "Non-urgent failures, these should be relayed to developers or admins;
	 * each item must be resolved within a given time." See also
	 * http://en.wikipedia.org/wiki/Syslog
	 *
	 * @param aMessage The message to be logged.
	 */
	void error( String aMessage );

	/**
	 * Logs a log message with log priority {@link LogPriority#ERROR}.
	 * "Non-urgent failures, these should be relayed to developers or admins;
	 * each item must be resolved within a given time." The placeholders in the
	 * message {@link String} are being replaced by the provided arguments.
	 * Implementations of this interface are recommended to use Java's
	 * {@link MessageFormat}'s syntax, the placeholder being replaced by the
	 * first argument is identified by "{0}" (without the quotes), the second
	 * one by "{1}", the third one by "{3}" and so on: Given the message to be
	 * "{0} + {1} = {2}" and the three argument be x, y and x + y, then "{0}" is
	 * replaced by the value of x, "{1}" is replaced by the value of y and "{2}"
	 * is replaced by the value of x + y. See also
	 * http://en.wikipedia.org/wiki/Syslog
	 *
	 * @param aMessage The message to be logged.
	 * @param aArguments The arguments used when replacing the placeholders.
	 */
	void error( String aMessage, Object... aArguments );

	/**
	 * Logs a log message with log priority {@link LogPriority#ERROR}.
	 * "Non-urgent failures, these should be relayed to developers or admins;
	 * each item must be resolved within a given time." See also
	 * http://en.wikipedia.org/wiki/Syslog
	 *
	 * @param aMessage The message to be logged.
	 * @param aThrowable An exception that the message should contain.
	 */
	void error( String aMessage, Throwable aThrowable );

	/**
	 * Logs a log message with log priority {@link LogPriority#ERROR}.
	 * "Non-urgent failures, these should be relayed to developers or admins;
	 * each item must be resolved within a given time." The placeholders in the
	 * message {@link String} are being replaced by the provided arguments.
	 * Implementations of this interface are recommended to use Java's
	 * {@link MessageFormat}'s syntax, the placeholder being replaced by the
	 * first argument is identified by "{0}" (without the quotes), the second
	 * one by "{1}", the third one by "{3}" and so on: Given the message to be
	 * "{0} + {1} = {2}" and the three argument be x, y and x + y, then "{0}" is
	 * replaced by the value of x, "{1}" is replaced by the value of y and "{2}"
	 * is replaced by the value of x + y. See also
	 * http://en.wikipedia.org/wiki/Syslog
	 *
	 * @param aMessage The message to be logged.
	 * @param aThrowable An exception that the message should contain.
	 * @param aArguments The arguments used when replacing the placeholders.
	 */
	void error( String aMessage, Throwable aThrowable, Object... aArguments );

	/**
	 * Determines whether the {@link RuntimeLogger} is being configured to log
	 * errors ({@link LogPriority#ERROR}). This method is useful to prevent the
	 * generation of vast amounts of log message {@link String} objects (and
	 * according processing) for the according {@link LogPriority} in case them
	 * messages of that {@link LogPriority} are not logged at all.
	 * 
	 * @return True in case errors are logged.
	 */
	boolean isLogError();

	/**
	 * Logs a log message with log priority {@link LogPriority#CRITICAL}.
	 * "Should be corrected immediately, but indicates failure in a secondary
	 * system, an example is a loss of a backup ISP connection." See also
	 * http://en.wikipedia.org/wiki/Syslog
	 *
	 * @param aMessage The message to be logged.
	 */
	void critical( String aMessage );

	/**
	 * Logs a log message with log priority {@link LogPriority#CRITICAL}.
	 * "Should be corrected immediately, but indicates failure in a secondary
	 * system, an example is a loss of a backup ISP connection." The
	 * placeholders in the message {@link String} are being replaced by the
	 * provided arguments. Implementations of this interface are recommended to
	 * use Java's {@link MessageFormat}'s syntax, the placeholder being replaced
	 * by the first argument is identified by "{0}" (without the quotes), the
	 * second one by "{1}", the third one by "{3}" and so on: Given the message
	 * to be "{0} + {1} = {2}" and the three argument be x, y and x + y, then
	 * "{0}" is replaced by the value of x, "{1}" is replaced by the value of y
	 * and "{2}" is replaced by the value of x + y. See also
	 * http://en.wikipedia.org/wiki/Syslog
	 *
	 * @param aMessage The message to be logged.
	 * @param aArguments The arguments used when replacing the placeholders.
	 */
	void critical( String aMessage, Object... aArguments );

	/**
	 * Logs a log message with log priority {@link LogPriority#CRITICAL}.
	 * "Should be corrected immediately, but indicates failure in a secondary
	 * system, an example is a loss of a backup ISP connection." See also
	 * http://en.wikipedia.org/wiki/Syslog
	 *
	 * @param aMessage The message to be logged.
	 * @param aThrowable An exception that the message should contain.
	 */
	void critical( String aMessage, Throwable aThrowable );

	/**
	 * Logs a log message with log priority {@link LogPriority#CRITICAL}.
	 * "Should be corrected immediately, but indicates failure in a secondary
	 * system, an example is a loss of a backup ISP connection." See also
	 * http://en.wikipedia.org/wiki/Syslog
	 *
	 * @param aMessage The message to be logged.
	 * @param aThrowable An exception that the message should contain.
	 * @param aArguments The arguments used when replacing the placeholders.
	 */
	void critical( String aMessage, Throwable aThrowable, Object... aArguments );

	/**
	 * Determines whether the {@link RuntimeLogger} is being configured to log
	 * fatal messages ({@link LogPriority#CRITICAL}). This method is useful to
	 * prevent the generation of vast amounts of log message {@link String}
	 * objects (and according processing) for the according {@link LogPriority}
	 * in case them messages of that {@link LogPriority} are not logged at all.
	 * 
	 * @return True in case fatal messages are logged.
	 */
	boolean isLogCritical();

	/**
	 * Logs a log message with log priority {@link LogPriority#ALERT}. "Should
	 * be corrected immediately, therefore notify staff who can fix the problem.
	 * An example would be the loss of a primary ISP connection." See also
	 * http://en.wikipedia.org/wiki/Syslog
	 *
	 * @param aMessage The message to be logged.
	 */
	void alert( String aMessage );

	/**
	 * Logs a log message with log priority {@link LogPriority#ALERT}. "Should
	 * be corrected immediately, therefore notify staff who can fix the problem.
	 * An example would be the loss of a primary ISP connection." The
	 * placeholders in the message {@link String} are being replaced by the
	 * provided arguments. Implementations of this interface are recommended to
	 * use Java's {@link MessageFormat}'s syntax, the placeholder being replaced
	 * by the first argument is identified by "{0}" (without the quotes), the
	 * second one by "{1}", the third one by "{3}" and so on: Given the message
	 * to be "{0} + {1} = {2}" and the three argument be x, y and x + y, then
	 * "{0}" is replaced by the value of x, "{1}" is replaced by the value of y
	 * and "{2}" is replaced by the value of x + y. See also
	 * http://en.wikipedia.org/wiki/Syslog
	 *
	 * @param aMessage The message to be logged.
	 * @param aArguments The arguments used when replacing the placeholders.
	 */
	void alert( String aMessage, Object... aArguments );

	/**
	 * Logs a log message with log priority {@link LogPriority#ALERT}.
	 * 
	 * @param aMessage The message to be logged.
	 * @param aThrowable An exception that the message should contain.
	 */
	void alert( String aMessage, Throwable aThrowable );

	/**
	 * Logs a log message with log priority {@link LogPriority#ALERT}.
	 * 
	 * @param aMessage The message to be logged.
	 * @param aThrowable An exception that the message should contain.
	 * @param aArguments The arguments used when replacing the placeholders.
	 */
	void alert( String aMessage, Throwable aThrowable, Object... aArguments );

	/**
	 * Determines whether the {@link RuntimeLogger} is being configured to log
	 * fatal messages ({@link LogPriority#ALERT}). This method is useful to
	 * prevent the generation of vast amounts of log message {@link String}
	 * objects (and according processing) for the according {@link LogPriority}
	 * in case them messages of that {@link LogPriority} are not logged at all.
	 * 
	 * @return True in case fatal messages are logged.
	 */
	boolean isLogAlert();

	/**
	 * Logs a log message with log priority {@link LogPriority#PANIC}. "A
	 * "panic" condition usually affecting multiple apps/servers/sites. At this
	 * level it would usually notify all tech staff on call." ( See also
	 * http://en.wikipedia.org/wiki/Syslog
	 *
	 * @param aMessage The message to be logged.
	 */
	void panic( String aMessage );

	/**
	 * Logs a log message with log priority {@link LogPriority#PANIC}. "A
	 * "panic" condition usually affecting multiple apps/servers/sites. At this
	 * level it would usually notify all tech staff on call." The placeholders
	 * in the message {@link String} are being replaced by the provided
	 * arguments. Implementations of this interface are recommended to use
	 * Java's {@link MessageFormat}'s syntax, the placeholder being replaced by
	 * the first argument is identified by "{0}" (without the quotes), the
	 * second one by "{1}", the third one by "{3}" and so on: Given the message
	 * to be "{0} + {1} = {2}" and the three argument be x, y and x + y, then
	 * "{0}" is replaced by the value of x, "{1}" is replaced by the value of y
	 * and "{2}" is replaced by the value of x + y. See also
	 * http://en.wikipedia.org/wiki/Syslog
	 *
	 * @param aMessage The message to be logged.
	 * @param aArguments The arguments used when replacing the placeholders.
	 */
	void panic( String aMessage, Object... aArguments );

	/**
	 * Logs a log message with log priority {@link LogPriority#PANIC}. "A
	 * "panic" condition usually affecting multiple apps/servers/sites. At this
	 * level it would usually notify all tech staff on call." ( See also
	 * http://en.wikipedia.org/wiki/Syslog
	 *
	 * @param aMessage The message to be logged.
	 * @param aThrowable An exception that the message should contain.
	 */
	void panic( String aMessage, Throwable aThrowable );

	/**
	 * Logs a log message with log priority {@link LogPriority#PANIC}. "A
	 * "panic" condition usually affecting multiple apps/servers/sites. At this
	 * level it would usually notify all tech staff on call." The placeholders
	 * in the message {@link String} are being replaced by the provided
	 * arguments. Implementations of this interface are recommended to use
	 * Java's {@link MessageFormat}'s syntax, the placeholder being replaced by
	 * the first argument is identified by "{0}" (without the quotes), the
	 * second one by "{1}", the third one by "{3}" and so on: Given the message
	 * to be "{0} + {1} = {2}" and the three argument be x, y and x + y, then
	 * "{0}" is replaced by the value of x, "{1}" is replaced by the value of y
	 * and "{2}" is replaced by the value of x + y. See also
	 * http://en.wikipedia.org/wiki/Syslog
	 *
	 * @param aMessage The message to be logged.
	 * @param aThrowable An exception that the message should contain.
	 * @param aArguments The arguments used when replacing the placeholders.
	 */
	void panic( String aMessage, Throwable aThrowable, Object... aArguments );

	/**
	 * Determines whether the {@link RuntimeLogger} is being configured to log
	 * fatal messages ({@link LogPriority#PANIC}). This method is useful to
	 * prevent the generation of vast amounts of log message {@link String}
	 * objects (and according processing) for the according {@link LogPriority}
	 * in case them messages of that {@link LogPriority} are not logged at all.
	 * 
	 * @return True in case fatal messages are logged.
	 */
	boolean isLogPanic();

}
