// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.numerical;

import static org.junit.jupiter.api.Assertions.*;

import java.nio.charset.StandardCharsets;

import org.junit.jupiter.api.Test;
import org.refcodes.data.Text;

/**
 *
 */
public class CrcAlgorithmTest {

	private static boolean IS_LOG_TESTS = Boolean.getBoolean( "log.tests" );

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@Test
	public void testNoBitError() {
		String theValue1 = Text.ARECIBO_MESSAGE + "A";
		String theOtherValue1 = Text.ARECIBO_MESSAGE + "A";
		long eChecksum;
		long eOtherChecksum;
		byte[] eBytes;
		for ( CrcAlgorithmConfig eCrcChecksum : CrcAlgorithmConfig.values() ) {
			eChecksum = eCrcChecksum.toCrcChecksum( theValue1.getBytes( StandardCharsets.UTF_8 ) );
			eOtherChecksum = eCrcChecksum.toCrcChecksum( theOtherValue1.getBytes( StandardCharsets.UTF_8 ) );
			if ( IS_LOG_TESTS ) {
				System.out.println( eCrcChecksum.getName() + ": " + Long.toHexString( eChecksum ) + " -?-> " + Long.toHexString( eOtherChecksum ) );
			}
			assertEquals( eChecksum, eOtherChecksum );
			assertTrue( Long.toHexString( eChecksum ).length() <= (eCrcChecksum.getCrcSize().getCrcWidth() * 2) );
			assertTrue( Long.toHexString( eOtherChecksum ).length() <= (eCrcChecksum.getCrcSize().getCrcWidth() * 2) );
			eBytes = eCrcChecksum.toCrcBytes( theValue1.getBytes( StandardCharsets.UTF_8 ), Endianess.LITTLE_ENDIAN );
			assertEquals( eChecksum, Endianess.LITTLE_ENDIAN.toUnsignedLong( eBytes ) );
		}
	}

	@Test
	public void testOneBitError() {
		String theValue1 = Text.ARECIBO_MESSAGE + "D";
		String theOtherValue1 = Text.ARECIBO_MESSAGE + "E";
		long eChecksum;
		long eOtherChecksum;
		byte[] eBytes;
		for ( CrcAlgorithmConfig eCrcChecksum : CrcAlgorithmConfig.values() ) {
			eChecksum = eCrcChecksum.toCrcChecksum( theValue1.getBytes( StandardCharsets.UTF_8 ) );
			eOtherChecksum = eCrcChecksum.toCrcChecksum( theOtherValue1.getBytes( StandardCharsets.UTF_8 ) );
			if ( IS_LOG_TESTS ) {
				System.out.println( eCrcChecksum.getName() + ": " + Long.toHexString( eChecksum ) + " -?-> " + Long.toHexString( eOtherChecksum ) );
			}
			assertNotEquals( eChecksum, eOtherChecksum );
			assertTrue( Long.toHexString( eChecksum ).length() <= (eCrcChecksum.getCrcSize().getCrcWidth() * 2) );
			assertTrue( Long.toHexString( eOtherChecksum ).length() <= (eCrcChecksum.getCrcSize().getCrcWidth() * 2) );
			eBytes = eCrcChecksum.toCrcBytes( theValue1.getBytes( StandardCharsets.UTF_8 ), Endianess.LITTLE_ENDIAN );
			assertEquals( eChecksum, Endianess.LITTLE_ENDIAN.toUnsignedLong( eBytes ) );
		}
	}

	@Test
	public void testTwoBitError() {
		String theValue1 = Text.ARECIBO_MESSAGE + "C";
		String theOtherValue1 = Text.ARECIBO_MESSAGE + "F";
		long eChecksum;
		long eOtherChecksum;
		byte[] eBytes;
		for ( CrcAlgorithmConfig eCrcChecksum : CrcAlgorithmConfig.values() ) {
			eChecksum = eCrcChecksum.toCrcChecksum( theValue1.getBytes( StandardCharsets.UTF_8 ) );
			eOtherChecksum = eCrcChecksum.toCrcChecksum( theOtherValue1.getBytes( StandardCharsets.UTF_8 ) );
			if ( IS_LOG_TESTS ) {
				System.out.println( eCrcChecksum.getName() + ": " + Long.toHexString( eChecksum ) + " -?-> " + Long.toHexString( eOtherChecksum ) );
			}
			assertNotEquals( eChecksum, eOtherChecksum );
			assertTrue( Long.toHexString( eChecksum ).length() <= (eCrcChecksum.getCrcSize().getCrcWidth() * 2) );
			assertTrue( Long.toHexString( eOtherChecksum ).length() <= (eCrcChecksum.getCrcSize().getCrcWidth() * 2) );
			eBytes = eCrcChecksum.toCrcBytes( theValue1.getBytes( StandardCharsets.UTF_8 ), Endianess.LITTLE_ENDIAN );
			assertEquals( eChecksum, Endianess.LITTLE_ENDIAN.toUnsignedLong( eBytes ) );
		}
	}

	@Test
	public void testRandomLongChecksum() {
		long eChecksum;
		long eOtherChecksum;
		byte[] eBytes;
		for ( CrcAlgorithmConfig eCrcChecksum : CrcAlgorithmConfig.values() ) {
			long eValue;
			for ( int i = 0; i < 1024; i++ ) {
				eValue = (long) (Math.random() * Long.MAX_VALUE);
				eChecksum = eCrcChecksum.toCrcChecksum( NumericalUtility.toBytes( eValue ) );
				eOtherChecksum = eCrcChecksum.toCrcChecksum( NumericalUtility.toBytes( eValue ) );
				if ( IS_LOG_TESTS ) {
					System.out.print( eCrcChecksum.getName() + ": " + eValue + " = " + Long.toHexString( eChecksum ) + " -?-> " + Long.toHexString( eOtherChecksum ) );
				}
				assertEquals( eChecksum, eOtherChecksum );
				assertTrue( Long.toHexString( eChecksum ).length() <= (eCrcChecksum.getCrcSize().getCrcWidth() * 2) );
				eBytes = eCrcChecksum.toCrcBytes( NumericalUtility.toBytes( eValue ), Endianess.LITTLE_ENDIAN );
				if ( IS_LOG_TESTS ) {
					System.out.println( " {" + NumericalUtility.toHexString( ", ", eBytes ) + "}" );
				}
				assertEquals( eChecksum, Endianess.LITTLE_ENDIAN.toUnsignedLong( eBytes ) );
			}
		}
	}

	@Test
	public void testCrcCategories() {
		for ( CrcSize eCrcCategory : CrcSize.values() ) {
			for ( CrcAlgorithmConfig eCrcAlgorithm : eCrcCategory.getCrcAlgorithms() ) {
				if ( IS_LOG_TESTS ) {
					System.out.println( eCrcCategory + " <-- " + eCrcAlgorithm );
				}
				assertTrue( eCrcAlgorithm.name().startsWith( eCrcCategory.name() ) );
			}
		}
	}

	@Test
	public void testCrcAlgorithms() {
		for ( CrcAlgorithmConfig eCrcAlgorithm : CrcAlgorithmConfig.values() ) {
			CrcSize theCrcCategory = eCrcAlgorithm.getCrcSize();
			if ( IS_LOG_TESTS ) {
				System.out.println( eCrcAlgorithm + " --> " + theCrcCategory );
			}
			assertTrue( eCrcAlgorithm.name().startsWith( theCrcCategory.name() ) );
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
