// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.numerical;

import static org.junit.jupiter.api.Assertions.assertArrayEquals;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertTrue;
import static org.junit.jupiter.api.Assertions.fail;

import java.util.Arrays;
import java.util.UUID;

import org.junit.jupiter.api.Test;

public class NumericalUtilityTest {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static boolean IS_LOG_TESTS = Boolean.getBoolean( "log.tests" );

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@Test
	public void testSetBits1() {
		byte theValue = 0;
		assertFalse( NumericalUtility.isBitSetAt( theValue, 0 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 1 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 2 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 3 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 4 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 5 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 6 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 7 ) );
		theValue = NumericalUtility.setBitAt( theValue, 0, true );
		theValue = NumericalUtility.setBitAt( theValue, 1, true );
		theValue = NumericalUtility.setBitAt( theValue, 2, true );
		theValue = NumericalUtility.setBitAt( theValue, 3, true );
		theValue = NumericalUtility.setBitAt( theValue, 4, true );
		theValue = NumericalUtility.setBitAt( theValue, 5, true );
		theValue = NumericalUtility.setBitAt( theValue, 6, true );
		theValue = NumericalUtility.setBitAt( theValue, 7, true );
		if ( IS_LOG_TESTS ) System.out.println( theValue );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 0 ) );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 1 ) );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 2 ) );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 3 ) );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 4 ) );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 5 ) );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 6 ) );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 7 ) );
		assertEquals( (byte) 0xFF, theValue );
	}

	@Test
	public void testSetBits2() {
		byte theValue = (byte) 0xFF;
		assertTrue( NumericalUtility.isBitSetAt( theValue, 0 ) );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 1 ) );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 2 ) );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 3 ) );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 4 ) );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 5 ) );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 6 ) );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 7 ) );
		theValue = NumericalUtility.setBitAt( theValue, 0, false );
		theValue = NumericalUtility.setBitAt( theValue, 1, false );
		theValue = NumericalUtility.setBitAt( theValue, 2, false );
		theValue = NumericalUtility.setBitAt( theValue, 3, false );
		theValue = NumericalUtility.setBitAt( theValue, 4, false );
		theValue = NumericalUtility.setBitAt( theValue, 5, false );
		theValue = NumericalUtility.setBitAt( theValue, 6, false );
		theValue = NumericalUtility.setBitAt( theValue, 7, false );
		if ( IS_LOG_TESTS ) System.out.println( theValue );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 0 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 1 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 2 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 3 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 4 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 5 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 6 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 7 ) );
		assertEquals( 0, theValue );
	}

	@Test
	public void testSetBits3() {
		byte theValue = (byte) 0xFF;
		assertTrue( NumericalUtility.isBitSetAt( theValue, 0 ) );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 1 ) );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 2 ) );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 3 ) );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 4 ) );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 5 ) );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 6 ) );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 7 ) );
		theValue = NumericalUtility.setBitAt( theValue, 0, true );
		theValue = NumericalUtility.setBitAt( theValue, 1, false );
		theValue = NumericalUtility.setBitAt( theValue, 2, true );
		theValue = NumericalUtility.setBitAt( theValue, 3, false );
		theValue = NumericalUtility.setBitAt( theValue, 4, true );
		theValue = NumericalUtility.setBitAt( theValue, 5, false );
		theValue = NumericalUtility.setBitAt( theValue, 6, true );
		theValue = NumericalUtility.setBitAt( theValue, 7, false );
		if ( IS_LOG_TESTS ) System.out.println( theValue );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 0 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 1 ) );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 2 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 3 ) );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 4 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 5 ) );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 6 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 7 ) );
		assertEquals( 85, theValue );
	}

	@Test
	public void testSetBits4() {
		byte theValue = 0;
		assertFalse( NumericalUtility.isBitSetAt( theValue, 0 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 1 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 2 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 3 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 4 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 5 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 6 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 7 ) );
		theValue = NumericalUtility.setBitAt( theValue, 0, true );
		theValue = NumericalUtility.setBitAt( theValue, 1, false );
		theValue = NumericalUtility.setBitAt( theValue, 2, true );
		theValue = NumericalUtility.setBitAt( theValue, 3, false );
		theValue = NumericalUtility.setBitAt( theValue, 4, true );
		theValue = NumericalUtility.setBitAt( theValue, 5, false );
		theValue = NumericalUtility.setBitAt( theValue, 6, true );
		theValue = NumericalUtility.setBitAt( theValue, 7, false );
		if ( IS_LOG_TESTS ) System.out.println( theValue );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 0 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 1 ) );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 2 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 3 ) );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 4 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 5 ) );
		assertTrue( NumericalUtility.isBitSetAt( theValue, 6 ) );
		assertFalse( NumericalUtility.isBitSetAt( theValue, 7 ) );
		assertEquals( 85, theValue );
	}

	@Test
	public void testToBytes1() {
		byte[] theBytes = {
				0, 1, 2, 3, 127, (byte) 128, (byte) 129, (byte) 253, (byte) 254, (byte) 255
		};
		if ( IS_LOG_TESTS ) {
			System.out.println( "[✓] " + Arrays.toString( theBytes ) );
		}
		// @formatter:off
		String[] theStrings = {
			"{ 0x00 0x01 0x02 0x03 0x7F 0x80 0x81 0xFD 0xFE 0xFF }", 
			"[   0,   1,   2,   3, 127, 128, 129, 253, 254, 255 ]",
			"[   0,   1,   2,   3, 127, 128, 129, -3, -2, -1 ]",
			"( 0    0x01,   2,0x03, 127,0x80, 129 0xFD, 254,0xFF )",
			"0,0x01,2,0x03,127,0x80,129,0xFD,254,0xFF"
		};
		// @formatter:on
		String eString;
		byte[] eBytes;
		for ( int i = 0; i < theStrings.length; i++ ) {
			eString = theStrings[i];
			eBytes = NumericalUtility.toBytes( eString );
			if ( IS_LOG_TESTS ) {
				System.out.println( "[" + i + "] " + Arrays.toString( eBytes ) );
			}
			assertArrayEquals( theBytes, eBytes );
		}
	}

	@Test
	public void testToBytes2() {
		// @formatter:off
		String[] theStrings = {
			"{ 0x00 0x01 0x02 0x03 0x7F 0x80 0x81 0xFD 0xFE 0x100 }", 
			"[   0,   1,   X,   3, 127, 128, 129, 253, 254, 255 ]",
			"[   0,   1,   2,   3, 127, 128, 129, -3, -2, -129 ]",
			"( !0    0x01,   2,0x03, 127,0x80, 129 0xFD, 254,0xFF )",
			"0,0x01,2,0x03,127,0x80,129,0xFD,254,-256"
		};
		// @formatter:on
		String eString;
		byte[] eBytes;
		for ( int i = 0; i < theStrings.length; i++ ) {
			eString = theStrings[i];
			try {
				eBytes = NumericalUtility.toBytes( eString );
				if ( IS_LOG_TESTS ) {
					System.out.println( "[" + i + "] " + eString );
					System.out.println( "[x] " + Arrays.toString( eBytes ) );
				}
				fail( "Expected an <IllegalArgumentException> (<NumberFormatException>!" );
			}
			catch ( IllegalArgumentException expected ) {
				if ( IS_LOG_TESTS ) {
					System.out.println( "[" + i + "] " + expected.getMessage() );
				}
			}
		}
	}

	@Test
	public void testEdgeCase() {
		long theEdgeCase = 0xff9833db2bcc861dL;
		byte[] theBytes = NumericalUtility.toUnsignedBigEndianBytes( theEdgeCase, 8 );
		long theConversion = NumericalUtility.toLongFromBigEndianBytes( theBytes );
		System.out.println( "   theEdgeCase = " + theEdgeCase );
		System.out.println( "theEdgeCaseHex = " + Long.toHexString( theEdgeCase ) );
		System.out.println( "      theBytes = " + NumericalUtility.toHexString( theBytes ) );
		System.out.println( " theConversion = " + Long.toHexString( theConversion ) );

	}

	@Test
	public void testToLittleEndian() {
		// 16535 = 01000000 10010111 in big endian 
		byte[] theLittleEndian = new byte[2];
		theLittleEndian[0] = (byte) Integer.parseInt( "10010111", 2 );
		theLittleEndian[1] = (byte) Integer.parseInt( "01000000", 2 );
		byte[] theResult = NumericalUtility.toLittleEndianBytes( 16535, 2 );
		if ( IS_LOG_TESTS ) {
			System.out.println( "Expected = " + Arrays.toString( theLittleEndian ) );
			System.out.println( "  Result = " + Arrays.toString( theResult ) );
		}
		assertArrayEquals( theLittleEndian, theResult );
	}

	@Test
	public void testToBigEndian() {
		// 16535 = 01000000 10010111 in big endian 
		byte[] theLittleEndian = new byte[2];
		theLittleEndian[0] = (byte) Integer.parseInt( "01000000", 2 );
		theLittleEndian[1] = (byte) Integer.parseInt( "10010111", 2 );
		byte[] theResult = NumericalUtility.toBigEndianBytes( 16535, 2 );
		if ( IS_LOG_TESTS ) {
			System.out.println( "Expected = " + Arrays.toString( theLittleEndian ) );
			System.out.println( "  Result = " + Arrays.toString( theResult ) );
		}
		assertArrayEquals( theLittleEndian, theResult );
	}

	/**
	 * Test to double.
	 */
	@Test
	public void testToDouble() {
		Double eDouble;
		String eText;
		double theMin = -1;
		double theMax = 0;
		for ( int i = 0; i < 15000; i++ ) {
			for ( int j = 1; j < 30; j++ ) {
				// eText = RandomStringUtils.random( j );
				eText = toRandomText( j );
				eDouble = NumericalUtility.toDouble( eText );
				if ( IS_LOG_TESTS ) System.out.println( eDouble + " := " + eText );
				assertTrue( eDouble >= 0 );
				assertTrue( eDouble <= 1 );
				if ( theMin == -1 || theMin > eDouble ) theMin = eDouble;
				if ( theMax < eDouble ) theMax = eDouble;
			}
		}
		if ( IS_LOG_TESTS ) {
			System.out.println( "Min := " + theMin );
			System.out.println( "Max := " + theMax );
		}
	}

	/**
	 * Test to double min.
	 */
	@Test
	public void testToDoubleMin() {
		byte[] theBytes = new byte[] {
				0, 0, 0, 0, 0, 0, 0, 0
		};
		String theText = new String( theBytes );
		double theDouble = NumericalUtility.toDouble( theText );
		if ( IS_LOG_TESTS ) System.out.println( theDouble + " := " + theText );
	}

	/**
	 * Test to double max.
	 */
	@Test
	public void testToDoubleMax() {
		byte[] theBytes = new byte[] {
				(byte) 0xff
		};
		String theText = new String( theBytes );
		double theDouble = NumericalUtility.toDouble( theText );
		if ( IS_LOG_TESTS ) System.out.println( theDouble + " := " + theText );
	}

	/**
	 * Test to doubles 1.
	 */
	@Test
	public void testToDoubles1() {
		String eString = "";
		for ( int l = 1; l < 100; l++ ) {
			for ( int i = 1; i < 100; i++ ) {
				try {
					NumericalUtility.toDoubles( eString, i );
				}
				catch ( Exception e ) {
					fail( "Expecting not to fail at length <" + i + ">! Message = " + e.getMessage() );
				}
			}
			// eString = RandomStringUtils.randomAlphabetic( l );
			eString = toRandomText( l );
		}
	}

	/**
	 * Test to doubles 2.
	 */
	@Test
	public void testToDoubles2() {
		double[] theDoubles = NumericalUtility.toDoubles( "Hallo Welt, Chais is here to stay!", 3 );
		if ( IS_LOG_TESTS ) {
			for ( double eDouble : theDoubles ) {
				System.out.println( eDouble );
			}
		}
	}

	/**
	 * Test to doubles 3.
	 */
	@Test
	public void testToDoubles3() {
		double[] theDoubles = NumericalUtility.toDoubles( "08021972", 3 );
		if ( IS_LOG_TESTS ) {
			for ( double eDouble : theDoubles ) {
				System.out.println( eDouble );
			}
		}
	}

	/**
	 * Test long to bytes.
	 */
	@Test
	public void testLongToBytes() {
		long theLong = 1234567890;
		byte[] theBytes = NumericalUtility.toBytes( theLong );
		long theResult = NumericalUtility.toLong( theBytes );
		assertEquals( theLong, theResult );
	}

	/**
	 * Test long to bytes zero.
	 */
	@Test
	public void testLongToBytesZero() {
		long theLong = 0;
		byte[] theBytes = NumericalUtility.toBytes( theLong );
		long theResult = NumericalUtility.toLong( theBytes );
		assertEquals( theLong, theResult );
	}

	/**
	 * Test long to bytes min.
	 */
	@Test
	public void testLongToBytesMin() {
		long theLong = Long.MIN_VALUE;
		byte[] theBytes = NumericalUtility.toBytes( theLong );
		long theResult = NumericalUtility.toLong( theBytes );
		assertEquals( theLong, theResult );
	}

	/**
	 * Test long to bytes max.
	 */
	@Test
	public void testLongToBytesMax() {
		long theLong = Long.MAX_VALUE;
		byte[] theBytes = NumericalUtility.toBytes( theLong );
		long theResult = NumericalUtility.toLong( theBytes );
		assertEquals( theLong, theResult );
	}

	/**
	 * Test integer to bytes.
	 */
	@Test
	public void testIntegerToBytes() {
		int theInteger = 1234567890;
		byte[] theBytes = NumericalUtility.toBytes( theInteger );
		int theResult = NumericalUtility.toInt( theBytes );
		assertEquals( theInteger, theResult );
	}

	/**
	 * Test integer to bytes zero.
	 */
	@Test
	public void testIntegerToBytesZero() {
		int theInteger = 0;
		byte[] theBytes = NumericalUtility.toBytes( theInteger );
		int theResult = NumericalUtility.toInt( theBytes );
		assertEquals( theInteger, theResult );
	}

	/**
	 * Test integer to bytes min.
	 */
	@Test
	public void testIntegerToBytesMin() {
		int theInteger = Integer.MIN_VALUE;
		byte[] theBytes = NumericalUtility.toBytes( theInteger );
		int theResult = NumericalUtility.toInt( theBytes );
		assertEquals( theInteger, theResult );
	}

	/**
	 * Test integer to bytes max.
	 */
	@Test
	public void testIntegerToBytesMax() {
		int theInteger = Integer.MAX_VALUE;
		byte[] theBytes = NumericalUtility.toBytes( theInteger );
		int theResult = NumericalUtility.toInt( theBytes );
		assertEquals( theInteger, theResult );
	}

	@Test
	public void testToFromBigEndianBytes() {
		byte[] eBytes;
		int eValue;
		for ( int i = -1024; i < 1024; i++ ) {
			eBytes = NumericalUtility.toBigEndianBytes( i, 8 );
			eValue = NumericalUtility.toIntFromBigEndianBytes( eBytes );
			if ( IS_LOG_TESTS ) {
				System.out.println( NumericalUtility.toHexString( " ", eBytes ) + " = " + i + "-?->" + eValue );
			}
			assertEquals( i, eValue );
		}

		// Integer MAX & MIN |-->

		int theValue = Integer.MAX_VALUE;
		eBytes = NumericalUtility.toBigEndianBytes( theValue, 8 );
		eValue = NumericalUtility.toIntFromBigEndianBytes( eBytes );
		if ( IS_LOG_TESTS ) {
			System.out.println( NumericalUtility.toHexString( " ", eBytes ) + " = " + theValue + "-?->" + eValue );
		}
		assertEquals( theValue, eValue );

		theValue = Integer.MIN_VALUE;
		eBytes = NumericalUtility.toBigEndianBytes( theValue, 8 );
		eValue = NumericalUtility.toIntFromBigEndianBytes( eBytes );
		if ( IS_LOG_TESTS ) {
			System.out.println( NumericalUtility.toHexString( " ", eBytes ) + " = " + theValue + "-?->" + eValue );
		}
		assertEquals( theValue, eValue );

		// Long MAX & MIN |-->

		long theLong = Long.MAX_VALUE;
		eBytes = NumericalUtility.toBigEndianBytes( theLong, 8 );
		long eLong = NumericalUtility.toLongFromBigEndianBytes( eBytes );
		if ( IS_LOG_TESTS ) {
			System.out.println( NumericalUtility.toHexString( " ", eBytes ) + " = " + theLong + "-?->" + eLong );
		}
		assertEquals( theLong, eLong );

		theLong = Long.MIN_VALUE;
		eBytes = NumericalUtility.toBigEndianBytes( theLong, 8 );
		eLong = NumericalUtility.toLongFromBigEndianBytes( eBytes );
		if ( IS_LOG_TESTS ) {
			System.out.println( NumericalUtility.toHexString( " ", eBytes ) + " = " + theLong + "-?->" + eLong );
		}
		assertEquals( theLong, eLong );
	}

	@Test
	public void testToFromLittleEndianBytes() {
		byte[] eBytes;
		int eValue;
		for ( int i = -1024; i < 1024; i++ ) {
			eBytes = NumericalUtility.toLittleEndianBytes( i, 8 );
			eValue = NumericalUtility.toIntFromLittleEndianBytes( eBytes );
			if ( IS_LOG_TESTS ) {
				System.out.println( NumericalUtility.toHexString( " ", eBytes ) + " = " + i + "-?->" + eValue );
			}
			assertEquals( i, eValue );
		}

		// Integer MAX & MIN |-->

		int theValue = Integer.MAX_VALUE;
		eBytes = NumericalUtility.toLittleEndianBytes( theValue, 8 );
		eValue = NumericalUtility.toIntFromLittleEndianBytes( eBytes );
		if ( IS_LOG_TESTS ) {
			System.out.println( NumericalUtility.toHexString( " ", eBytes ) + " = " + theValue + "-?->" + eValue );
		}
		assertEquals( theValue, eValue );

		theValue = Integer.MIN_VALUE;
		eBytes = NumericalUtility.toLittleEndianBytes( theValue, 8 );
		eValue = NumericalUtility.toIntFromLittleEndianBytes( eBytes );
		if ( IS_LOG_TESTS ) {
			System.out.println( NumericalUtility.toHexString( " ", eBytes ) + " = " + theValue + "-?->" + eValue );
		}
		assertEquals( theValue, eValue );

		// Long MAX & MIN |-->

		long theLong = Long.MAX_VALUE;
		eBytes = NumericalUtility.toLittleEndianBytes( theLong, 8 );
		long eLong = NumericalUtility.toLongFromLittleEndianBytes( eBytes );
		if ( IS_LOG_TESTS ) {
			System.out.println( NumericalUtility.toHexString( " ", eBytes ) + " = " + theLong + "-?->" + eLong );
		}
		assertEquals( theLong, eLong );

		theLong = Long.MIN_VALUE;
		eBytes = NumericalUtility.toLittleEndianBytes( theLong, 8 );
		eLong = NumericalUtility.toLongFromLittleEndianBytes( eBytes );
		if ( IS_LOG_TESTS ) {
			System.out.println( NumericalUtility.toHexString( " ", eBytes ) + " = " + theLong + "-?->" + eLong );
		}
		assertEquals( theLong, eLong );
	}

	@Test
	public void testFloatToFromBigEndianBytes() {
		byte[] eBytes;
		float eValue;
		for ( float i = 0; i < 1024; i += 1.3337 ) {
			eBytes = NumericalUtility.toBigEndianBytes( i );
			eValue = NumericalUtility.toFloatFromBigEndianBytes( eBytes );
			if ( IS_LOG_TESTS ) {
				System.out.println( NumericalUtility.toHexString( " ", eBytes ) + " = " + i + "-?->" + eValue );
			}
			assertEquals( i, eValue );
		}

		// Float MAX & MIN |-->

		float theValue = Float.MAX_VALUE;
		eBytes = NumericalUtility.toBigEndianBytes( theValue );
		eValue = NumericalUtility.toFloatFromBigEndianBytes( eBytes );
		if ( IS_LOG_TESTS ) {
			System.out.println( NumericalUtility.toHexString( " ", eBytes ) + " = " + theValue + "-?->" + eValue );
		}
		assertEquals( theValue, eValue );

		theValue = Float.MIN_VALUE;
		eBytes = NumericalUtility.toBigEndianBytes( theValue );
		eValue = NumericalUtility.toFloatFromBigEndianBytes( eBytes );
		if ( IS_LOG_TESTS ) {
			System.out.println( NumericalUtility.toHexString( " ", eBytes ) + " = " + theValue + "-?->" + eValue );
		}
		assertEquals( theValue, eValue );

	}

	@Test
	public void testFloatToFromLittleEndianBytes() {
		byte[] eBytes;
		float eValue;
		for ( float i = 0; i < 1024; i += 1.3337 ) {
			eBytes = NumericalUtility.toLittleEndianBytes( i );
			eValue = NumericalUtility.toFloatFromLittleEndianBytes( eBytes );
			if ( IS_LOG_TESTS ) {
				System.out.println( NumericalUtility.toHexString( " ", eBytes ) + " = " + i + "-?->" + eValue );
			}
			assertEquals( i, eValue );
		}

		// Float MAX & MIN |-->

		float theValue = Float.MAX_VALUE;
		eBytes = NumericalUtility.toLittleEndianBytes( theValue );
		eValue = NumericalUtility.toFloatFromLittleEndianBytes( eBytes );
		if ( IS_LOG_TESTS ) {
			System.out.println( NumericalUtility.toHexString( " ", eBytes ) + " = " + theValue + "-?->" + eValue );
		}
		assertEquals( theValue, eValue );

		theValue = Float.MIN_VALUE;
		eBytes = NumericalUtility.toLittleEndianBytes( theValue );
		eValue = NumericalUtility.toFloatFromLittleEndianBytes( eBytes );
		if ( IS_LOG_TESTS ) {
			System.out.println( NumericalUtility.toHexString( " ", eBytes ) + " = " + theValue + "-?->" + eValue );
		}
		assertEquals( theValue, eValue );

	}

	@Test
	public void testDoubleToFromBigEndianBytes() {
		byte[] eBytes;
		double eValue;
		for ( double i = 0; i < 1024; i += 1.3337 ) {
			eBytes = NumericalUtility.toBigEndianBytes( i );
			eValue = NumericalUtility.toDoubleFromBigEndianBytes( eBytes );
			if ( IS_LOG_TESTS ) {
				System.out.println( NumericalUtility.toHexString( " ", eBytes ) + " = " + i + "-?->" + eValue );
			}
			assertEquals( i, eValue );
		}

		// Double MAX & MIN |-->

		double theValue = Double.MAX_VALUE;
		eBytes = NumericalUtility.toBigEndianBytes( theValue );
		eValue = NumericalUtility.toDoubleFromBigEndianBytes( eBytes );
		if ( IS_LOG_TESTS ) {
			System.out.println( NumericalUtility.toHexString( " ", eBytes ) + " = " + theValue + "-?->" + eValue );
		}
		assertEquals( theValue, eValue );

		theValue = Double.MIN_VALUE;
		eBytes = NumericalUtility.toBigEndianBytes( theValue );
		eValue = NumericalUtility.toDoubleFromBigEndianBytes( eBytes );
		if ( IS_LOG_TESTS ) {
			System.out.println( NumericalUtility.toHexString( " ", eBytes ) + " = " + theValue + "-?->" + eValue );
		}
		assertEquals( theValue, eValue );

	}

	@Test
	public void testDoubleToFromLittleEndianBytes() {
		byte[] eBytes;
		double eValue;
		for ( double i = 0; i < 1024; i += 1.3337 ) {
			eBytes = NumericalUtility.toLittleEndianBytes( i );
			eValue = NumericalUtility.toDoubleFromLittleEndianBytes( eBytes );
			if ( IS_LOG_TESTS ) {
				System.out.println( NumericalUtility.toHexString( " ", eBytes ) + " = " + i + "-?->" + eValue );
			}
			assertEquals( i, eValue );
		}

		// Double MAX & MIN |-->

		double theValue = Double.MAX_VALUE;
		eBytes = NumericalUtility.toLittleEndianBytes( theValue );
		eValue = NumericalUtility.toDoubleFromLittleEndianBytes( eBytes );
		if ( IS_LOG_TESTS ) {
			System.out.println( NumericalUtility.toHexString( " ", eBytes ) + " = " + theValue + "-?->" + eValue );
		}
		assertEquals( theValue, eValue );

		theValue = Double.MIN_VALUE;
		eBytes = NumericalUtility.toLittleEndianBytes( theValue );
		eValue = NumericalUtility.toDoubleFromLittleEndianBytes( eBytes );
		if ( IS_LOG_TESTS ) {
			System.out.println( NumericalUtility.toHexString( " ", eBytes ) + " = " + theValue + "-?->" + eValue );
		}
		assertEquals( theValue, eValue );

	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	private String toRandomText( int aLength ) {
		String eText = UUID.randomUUID().toString();
		while ( eText.length() < aLength ) {
			eText = eText + UUID.randomUUID().toString();
		}
		return eText.substring( 0, aLength );
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
