// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.numerical;

import org.refcodes.exception.BugException;

/**
 * An enumeration of valid endianess.
 */
public enum Endianess {

	LITTLE, BIG;

	/**
	 * Converts the given value to a byte array in the according endian
	 * notation.
	 * 
	 * @param aValue The value for which to get the byte array.
	 * @param aLength The number of bytes to use.
	 * 
	 * @return The according array.
	 */
	public byte[] toBytes( long aValue, int aLength ) {
		switch ( this ) {
		case BIG:
			return NumericalUtility.toBigEndianBytes( aValue, aLength );
		case LITTLE:
			return NumericalUtility.toLittleEndianBytes( aValue, aLength );
		default:
			throw new BugException( "The enumeration value <" + this + "> has not been implemented yet!" );
		}
	}

	/**
	 * Converts the given value to a byte array in the according endian
	 * notation.
	 * 
	 * @param aValue The value for which to get the byte array.
	 * 
	 * @return The according array.
	 */
	public byte[] toBytes( long aValue ) {
		switch ( this ) {
		case BIG:
			return NumericalUtility.toBigEndianBytes( aValue );
		case LITTLE:
			return NumericalUtility.toLittleEndianBytes( aValue );
		default:
			throw new BugException( "The enumeration value <" + this + "> has not been implemented yet!" );
		}
	}

	/**
	 * Converts the given value to a byte array in the according endian
	 * notation.
	 * 
	 * @param aValue The value for which to get the byte array.
	 * @param aLength The number of bytes to use.
	 * 
	 * @return The according array.
	 */
	public byte[] toBytes( int aValue, int aLength ) {
		switch ( this ) {
		case BIG:
			return NumericalUtility.toBigEndianBytes( aValue, aLength );
		case LITTLE:
			return NumericalUtility.toLittleEndianBytes( aValue, aLength );
		default:
			throw new BugException( "The enumeration value <" + this + "> has not been implemented yet!" );
		}
	}

	/**
	 * Converts the given value to a byte array in the according endian
	 * notation.
	 * 
	 * @param aValue The value for which to get the byte array.
	 * 
	 * @return The according array.
	 */
	public byte[] toBytes( int aValue ) {
		switch ( this ) {
		case BIG:
			return NumericalUtility.toBigEndianBytes( aValue );
		case LITTLE:
			return NumericalUtility.toLittleEndianBytes( aValue );
		default:
			throw new BugException( "The enumeration value <" + this + "> has not been implemented yet!" );
		}
	}

	/**
	 * Converts the given value to a byte array in the according endian
	 * notation.
	 * 
	 * @param aValue The value for which to get the byte array.
	 * @param aLength The number of bytes to use.
	 * 
	 * @return The according array.
	 */
	public byte[] toBytes( short aValue, int aLength ) {
		switch ( this ) {
		case BIG:
			return NumericalUtility.toBigEndianBytes( aValue, aLength );
		case LITTLE:
			return NumericalUtility.toLittleEndianBytes( aValue, aLength );
		default:
			throw new BugException( "The enumeration value <" + this + "> has not been implemented yet!" );
		}
	}

	/**
	 * Converts the given value to a byte array in the according endian
	 * notation.
	 * 
	 * @param aValue The value for which to get the byte array.
	 * 
	 * @return The according array.
	 */
	public byte[] toBytes( short aValue ) {
		switch ( this ) {
		case BIG:
			return NumericalUtility.toBigEndianBytes( aValue );
		case LITTLE:
			return NumericalUtility.toLittleEndianBytes( aValue );
		default:
			throw new BugException( "The enumeration value <" + this + "> has not been implemented yet!" );
		}
	}

	/**
	 * Converts the given value to a byte array in the according endian
	 * notation. As the float type is 32 bits in java, an array of length 4 will
	 * be returned.
	 * 
	 * @param aValue The value for which to get the byte array.
	 * 
	 * @return The according array.
	 */
	public byte[] toBytes( float aValue ) {
		switch ( this ) {
		case BIG:
			return NumericalUtility.toBigEndianBytes( aValue );
		case LITTLE:
			return NumericalUtility.toLittleEndianBytes( aValue );
		default:
			throw new BugException( "The enumeration value <" + this + "> has not been implemented yet!" );
		}
	}

	/**
	 * Converts the given value to a byte array in the according endian
	 * notation. As the float type is 32 bits in java, an array of length 4 will
	 * be returned.
	 * 
	 * @param aValue The value for which to get the byte array.
	 * 
	 * @return The according array.
	 */
	public byte[] toBytes( double aValue ) {
		switch ( this ) {
		case BIG:
			return NumericalUtility.toBigEndianBytes( aValue );
		case LITTLE:
			return NumericalUtility.toLittleEndianBytes( aValue );
		default:
			throw new BugException( "The enumeration value <" + this + "> has not been implemented yet!" );
		}
	}

	/**
	 * Converts the given value to a byte array in the according endian
	 * notation.
	 * 
	 * @param aValue The value for which to get the byte array.
	 * @param aLength The number of bytes to use.
	 * 
	 * @return The according array.
	 */
	public byte[] toUnsignedBytes( long aValue, int aLength ) {
		switch ( this ) {
		case BIG:
			return NumericalUtility.toUnsignedBigEndianBytes( aValue, aLength );
		case LITTLE:
			return NumericalUtility.toUnsignedLittleEndianBytes( aValue, aLength );
		default:
			throw new BugException( "The enumeration value <" + this + "> has not been implemented yet!" );
		}
	}

	/**
	 * Converts the given value to a byte array in the according endian
	 * notation.
	 * 
	 * @param aValue The value for which to get the byte array.
	 * 
	 * @return The according array.
	 */
	public byte[] toUnsignedBytes( long aValue ) {
		switch ( this ) {
		case BIG:
			return NumericalUtility.toUnsignedBigEndianBytes( aValue );
		case LITTLE:
			return NumericalUtility.toUnsignedLittleEndianBytes( aValue );
		default:
			throw new BugException( "The enumeration value <" + this + "> has not been implemented yet!" );
		}
	}

	/**
	 * Converts the given value to a byte array in the according endian
	 * notation.
	 * 
	 * @param aValue The value for which to get the byte array.
	 * 
	 * @return The according array.
	 */
	public byte[] toUnsignedBytes( int aValue ) {
		switch ( this ) {
		case BIG:
			return NumericalUtility.toUnsignedBigEndianBytes( aValue );
		case LITTLE:
			return NumericalUtility.toUnsignedLittleEndianBytes( aValue );
		default:
			throw new BugException( "The enumeration value <" + this + "> has not been implemented yet!" );
		}
	}

	/**
	 * Converts the given value to a byte array in the according endian
	 * notation.
	 * 
	 * @param aValue The value for which to get the byte array.
	 * 
	 * @return The according array.
	 */
	public byte[] toUnsignedBytes( short aValue ) {
		switch ( this ) {
		case BIG:
			return NumericalUtility.toUnsignedBigEndianBytes( aValue );
		case LITTLE:
			return NumericalUtility.toUnsignedLittleEndianBytes( aValue );
		default:
			throw new BugException( "The enumeration value <" + this + "> has not been implemented yet!" );
		}
	}

	/**
	 * Converts a byte array in according endianess to a long value.
	 * 
	 * @param aBytes The byte array to be converted.
	 * 
	 * @return The resulting long value.
	 */
	public long toLong( byte[] aBytes ) {
		switch ( this ) {
		case BIG:
			return NumericalUtility.toLongFromBigEndianBytes( aBytes );
		case LITTLE:
			return NumericalUtility.toLongFromLittleEndianBytes( aBytes );
		default:
			throw new BugException( "The enumeration value <" + this + "> has not been implemented yet!" );
		}
	}

	/**
	 * Converts a byte array in according endianess to an unsigned long value.
	 * 
	 * @param aBytes The byte array to be converted.
	 * 
	 * @return The resulting long value.
	 */
	public long toUnsignedLong( byte[] aBytes ) {
		switch ( this ) {
		case BIG:
			return NumericalUtility.toUnsignedLongFromBigEndianBytes( aBytes );
		case LITTLE:
			return NumericalUtility.toUnsignedLongFromLittleEndianBytes( aBytes );
		default:
			throw new BugException( "The enumeration value <" + this + "> has not been implemented yet!" );
		}
	}

	/**
	 * Converts a byte array in according endianess to an integer value.
	 * 
	 * @param aBytes The byte array to be converted.
	 * 
	 * @return The resulting integer value.
	 */
	public int toInteger( byte[] aBytes ) {
		switch ( this ) {
		case BIG:
			return NumericalUtility.toIntFromBigEndianBytes( aBytes );
		case LITTLE:
			return NumericalUtility.toIntFromLittleEndianBytes( aBytes );
		default:
			throw new BugException( "The enumeration value <" + this + "> has not been implemented yet!" );
		}
	}

	/**
	 * Converts a byte array in according endianess to an unsigned integer
	 * value.
	 * 
	 * @param aBytes The byte array to be converted.
	 * 
	 * @return The resulting integer value.
	 */
	public int toUnsignedInteger( byte[] aBytes ) {
		switch ( this ) {
		case BIG:
			return NumericalUtility.toUnsignedIntFromBigEndianBytes( aBytes );
		case LITTLE:
			return NumericalUtility.toUnsignedIntFromLittleEndianBytes( aBytes );
		default:
			throw new BugException( "The enumeration value <" + this + "> has not been implemented yet!" );
		}
	}

	/**
	 * Converts a byte array in according endianess to an short value.
	 * 
	 * @param aBytes The byte array to be converted.
	 * 
	 * @return The resulting short value.
	 */
	public short toShort( byte[] aBytes ) {
		switch ( this ) {
		case BIG:
			return NumericalUtility.toShortFromBigEndianBytes( aBytes );
		case LITTLE:
			return NumericalUtility.toShortFromLittleEndianBytes( aBytes );
		default:
			throw new BugException( "The enumeration value <" + this + "> has not been implemented yet!" );
		}
	}

	/**
	 * Converts a byte array in according endianess to an unsigned short value.
	 * 
	 * @param aBytes The byte array to be converted.
	 * 
	 * @return The resulting short value.
	 */
	public short toUnsignedShort( byte[] aBytes ) {
		switch ( this ) {
		case BIG:
			return NumericalUtility.toUnsignedShortFromBigEndianBytes( aBytes );
		case LITTLE:
			return NumericalUtility.toUnsignedShortFromLittleEndianBytes( aBytes );
		default:
			throw new BugException( "The enumeration value <" + this + "> has not been implemented yet!" );
		}
	}

	/**
	 * Converts a byte array in according endianess to an float value.
	 * 
	 * @param aBytes The byte array to be converted.
	 * 
	 * @return The resulting float value.
	 */
	public float toFloat( byte[] aBytes ) {
		switch ( this ) {
		case BIG:
			return NumericalUtility.toFloatFromBigEndianBytes( aBytes );
		case LITTLE:
			return NumericalUtility.toFloatFromLittleEndianBytes( aBytes );
		default:
			throw new BugException( "The enumeration value <" + this + "> has not been implemented yet!" );
		}
	}

	/**
	 * Converts a byte array in according endianess to an double value.
	 * 
	 * @param aBytes The byte array to be converted.
	 * 
	 * @return The resulting double value.
	 */
	public double toDouble( byte[] aBytes ) {
		switch ( this ) {
		case BIG:
			return NumericalUtility.toDoubleFromBigEndianBytes( aBytes );
		case LITTLE:
			return NumericalUtility.toDoubleFromLittleEndianBytes( aBytes );
		default:
			throw new BugException( "The enumeration value <" + this + "> has not been implemented yet!" );
		}
	}

}
