// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.observer;

import org.refcodes.mixin.EventMetaData;

/**
 * Basic implementation of the {@link GenericActionPayloadMetaDataEvent}.
 *
 * @param <A> The type of the action stored in the event.
 * @param <P> The type of the payload to be carried.
 * @param <EM> The type of the EventMetaData
 * @param <SRC> The type of the source in question.
 */
public abstract class AbstractPayloadMetaDataActionEvent<A, P, EM extends EventMetaData, SRC> extends AbstractMetaDataActionEvent<A, EM, SRC> implements GenericActionPayloadMetaDataEvent<A, P, EM, SRC> {

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private P _payload;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs an event with the given Meta-Data.
	 * 
	 * @param aAction The action which the event represents.
	 * @param aPayload The payload to be carried by the event.
	 * @param aEventMetaData The Meta-Data to by supplied by the event.
	 * @param aSource The source from which this event originated.
	 */
	public AbstractPayloadMetaDataActionEvent( A aAction, P aPayload, EM aEventMetaData, SRC aSource ) {
		super( aAction, aEventMetaData, aSource );
		_payload = aPayload;
	}

	/**
	 * Constructs an event with the given Meta-Data.
	 * 
	 * @param aAction The action which the event represents.
	 * @param aPayload The payload to be carried by the event.
	 * @param aSource The source from which this event originated.
	 */
	public AbstractPayloadMetaDataActionEvent( A aAction, P aPayload, SRC aSource ) {
		super( aAction, aSource );
		_payload = aPayload;
	}

	/**
	 * Constructs an event with the given Meta-Data.
	 * 
	 * @param aPayload The payload to be carried by the event.
	 * @param aEventMetaData The Meta-Data to by supplied by the event.
	 * @param aSource The source from which this event originated.
	 */
	public AbstractPayloadMetaDataActionEvent( P aPayload, EM aEventMetaData, SRC aSource ) {
		super( aEventMetaData, aSource );
		_payload = aPayload;
	}

	/**
	 * Constructs an event with the given Meta-Data.
	 *
	 * @param aPayload The payload to be carried by the event.
	 * @param aSource The source from which this event originated.
	 */
	public AbstractPayloadMetaDataActionEvent( P aPayload, SRC aSource ) {
		super( aSource );
		_payload = aPayload;
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public P getPayload() {
		return _payload;
	}

	/**
	 * Basic implementation of the
	 * {@link GenericActionPayloadMetaDataEventBuilder}.
	 * 
	 * @param <A> The type of the action stored in the event.
	 * @param <P> The type of the payload to be carried.
	 * @param <EM> The type of the EventMetaData
	 * @param <SRC> The type of the source in question.
	 * @param <B> The type of the builder to be returned by the builder methods.
	 */
	public static abstract class AbstractActionPayloadMetaDataEventBuilder<A, P, EM extends EventMetaData, SRC, B extends GenericActionPayloadMetaDataEventBuilder<A, P, EM, SRC, B>> extends AbstractActionMetaDataEventBuilder<A, EM, SRC, B> implements GenericActionPayloadMetaDataEventBuilder<A, P, EM, SRC, B> {

		// /////////////////////////////////////////////////////////////////////////
		// VARIABLES:
		// /////////////////////////////////////////////////////////////////////////

		private P _payload;

		// /////////////////////////////////////////////////////////////////////////
		// CONSTRUCTORS:
		// /////////////////////////////////////////////////////////////////////////

		/**
		 * Constructs an event with the given Meta-Data.
		 * 
		 * @param aAction The action which the event represents.
		 * @param aPayload The payload to be carried by the event.
		 * @param aEventMetaData The Meta-Data to by supplied by the event.
		 * @param aSource The source from which this event originated.
		 */
		public AbstractActionPayloadMetaDataEventBuilder( A aAction, P aPayload, EM aEventMetaData, SRC aSource ) {
			super( aAction, aEventMetaData, aSource );
			_payload = aPayload;
		}

		/**
		 * Constructs an event with the given Meta-Data.
		 * 
		 * @param aAction The action which the event represents.
		 * @param aPayload The payload to be carried by the event.
		 * @param aSource The source from which this event originated.
		 */
		public AbstractActionPayloadMetaDataEventBuilder( A aAction, P aPayload, SRC aSource ) {
			super( aAction, aSource );
			_payload = aPayload;
		}

		/**
		 * Constructs an event with the given Meta-Data.
		 * 
		 * @param aPayload The payload to be carried by the event.
		 * @param aEventMetaData The Meta-Data to by supplied by the event.
		 * @param aSource The source from which this event originated.
		 */
		public AbstractActionPayloadMetaDataEventBuilder( P aPayload, EM aEventMetaData, SRC aSource ) {
			super( aEventMetaData, aSource );
			_payload = aPayload;
		}

		/**
		 * Constructs an event with the given Meta-Data.
		 * 
		 * @param aEventMetaData The Meta-Data to by supplied by the event.
		 * @param aSource The source from which this event originated.
		 */
		public AbstractActionPayloadMetaDataEventBuilder( EM aEventMetaData, SRC aSource ) {
			super( aEventMetaData, aSource );
		}

		/**
		 * Constructs an event with the given Meta-Data.
		 *
		 * @param aPayload The payload to be carried by the event.
		 * @param aSource The source from which this event originated.
		 */
		public AbstractActionPayloadMetaDataEventBuilder( P aPayload, SRC aSource ) {
			super( aSource );
			_payload = aPayload;
		}

		// /////////////////////////////////////////////////////////////////////////
		// METHODS:
		// /////////////////////////////////////////////////////////////////////////

		/**
		 * {@inheritDoc}
		 */
		@Override
		public P getPayload() {
			return _payload;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public void setPayload( P aPayload ) {
			_payload = aPayload;
		}
	}
}
