// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.observer;

import org.refcodes.mixin.EventMetaData;
import org.refcodes.mixin.EventMetaDataBuilderImpl;
import org.refcodes.mixin.EventMetaDataImpl;
import org.refcodes.observer.AbstractMetaDataActionEvent.AbstractActionMetaDataEventBuilder;
import org.refcodes.observer.MetaDataActionEvent.ActionMetaDataEventBuilder;

/**
 * Implementation of the {@link ActionMetaDataEventBuilder} interface for easily
 * creating {@link MetaDataActionEvent} instances.
 */
public class MetaDataActionEventBuilderImpl extends AbstractActionMetaDataEventBuilder<Enum<?>, EventMetaData, Object, ActionMetaDataEventBuilder> implements ActionMetaDataEventBuilder {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs an event with the given Meta-Data.
	 * 
	 * @param aEventMetaData The Meta-Data to by supplied by the event.
	 * @param aSource The source from which this event originated.
	 */
	public MetaDataActionEventBuilderImpl( EventMetaData aEventMetaData, Object aSource ) {
		super( aEventMetaData instanceof EventMetaDataBuilder ? aEventMetaData : new EventMetaDataBuilderImpl( aEventMetaData ), aSource );
	}

	/**
	 * Constructs an event with the given Meta-Data.
	 * 
	 * @param aAction The action which the {@link GenericActionEvent}
	 *        represents.
	 * @param aEventMetaData The Meta-Data to by supplied by the event.
	 * @param aSource The source from which this event originated.
	 */
	public MetaDataActionEventBuilderImpl( Enum<?> aAction, EventMetaData aEventMetaData, Object aSource ) {
		super( aAction, aEventMetaData instanceof EventMetaData.EventMetaDataBuilder ? aEventMetaData : new EventMetaDataBuilderImpl( aEventMetaData ), aSource );
	}

	/**
	 * Constructs an event with the given Meta-Data.
	 *
	 * @param aAction The action which the {@link GenericActionEvent}
	 *        represents.
	 * @param aSource The source from which this event originated.
	 */
	public MetaDataActionEventBuilderImpl( Enum<?> aAction, Object aSource ) {
		super( aAction, aSource );
	}

	/**
	 * Constructs an event with the given Meta-Data.
	 *
	 * @param aSource The source from which this event originated.
	 */
	public MetaDataActionEventBuilderImpl( Object aSource ) {
		super( aSource );
	}

	/**
	 * Constructs an event with predefined values for the according properties
	 * retrieved from the caller's class.
	 * 
	 * @param aPublisherType The type of the event publisher.
	 * @param aSource The source from which this event originated.
	 */
	public MetaDataActionEventBuilderImpl( Class<?> aPublisherType, Object aSource ) {
		super( new EventMetaDataImpl( aPublisherType ), aSource );
	}

	/**
	 * Constructs an event with predefined values for the according properties
	 * retrieved from the caller's class.
	 * 
	 * @param aChannel The channel name on which the event is receivable.
	 * @param aSource The source from which this event originated.
	 */
	public MetaDataActionEventBuilderImpl( String aChannel, Object aSource ) {
		super( new EventMetaDataImpl( aChannel ), aSource );
	}

	/**
	 * Constructs an event with the given values for the according properties.
	 * 
	 * @param aAlias The alias property.
	 * @param aGroup The group property.
	 * @param aChannel The channel property.
	 * @param aUid The UID (Universal-TID) property.
	 * @param aPublisherType The type of the event publisher.
	 * @param aSource The source from which this event originated.
	 */
	public MetaDataActionEventBuilderImpl( String aAlias, String aGroup, String aChannel, String aUid, Class<?> aPublisherType, Object aSource ) {
		super( new EventMetaDataImpl( aAlias, aGroup, aChannel, aUid, aPublisherType ), aSource );
	}

	/**
	 * Constructs an event with predefined values for the according properties
	 * retrieved from the caller's class.
	 * 
	 * @param aAction The action which this represents.
	 * @param aPublisherType The type of the event publisher.
	 * @param aSource The source from which this event originated.
	 */
	public MetaDataActionEventBuilderImpl( Enum<?> aAction, Class<?> aPublisherType, Object aSource ) {
		super( aAction, new EventMetaDataImpl( aPublisherType ), aSource );
	}

	/**
	 * Constructs an event with predefined values for the according properties
	 * retrieved from the caller's class.
	 * 
	 * @param aAction The action which this represents.
	 * @param aChannel The channel name on which the event is receivable.
	 * @param aSource The source from which this event originated.
	 */
	public MetaDataActionEventBuilderImpl( Enum<?> aAction, String aChannel, Object aSource ) {
		super( aAction, new EventMetaDataImpl( aChannel ), aSource );
	}

	/**
	 * Constructs an event with the given values for the according properties.
	 * 
	 * @param aAction The action which this represents.
	 * @param aAlias The alias property.
	 * @param aGroup The group property.
	 * @param aChannel The channel property.
	 * @param aUid The UID (Universal-TID) property.
	 * @param aPublisherType The type of the event publisher.
	 * @param aSource The source from which this event originated.
	 */
	public MetaDataActionEventBuilderImpl( Enum<?> aAction, String aAlias, String aGroup, String aChannel, String aUid, Class<?> aPublisherType, Object aSource ) {
		super( aAction, new EventMetaDataImpl( aAlias, aGroup, aChannel, aUid, aPublisherType ), aSource );
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPublisherType( Class<?> aPublisherType ) {
		if ( _eventMetaData == null ) {
			_eventMetaData = new EventMetaDataBuilderImpl();
		}
		((EventMetaDataBuilder) _eventMetaData).setPublisherType( aPublisherType );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setUniversalId( String aUid ) {
		if ( _eventMetaData == null ) {
			_eventMetaData = new EventMetaDataBuilderImpl();
		}
		((EventMetaDataBuilder) _eventMetaData).setUniversalId( aUid );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAlias( String aAlias ) {
		if ( _eventMetaData == null ) {
			_eventMetaData = new EventMetaDataBuilderImpl();
		}
		((EventMetaDataBuilder) _eventMetaData).setAlias( aAlias );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setGroup( String aGroup ) {
		if ( _eventMetaData == null ) {
			_eventMetaData = new EventMetaDataBuilderImpl();
		}
		((EventMetaDataBuilder) _eventMetaData).setGroup( aGroup );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setChannel( String aChannel ) {
		if ( _eventMetaData == null ) {
			_eventMetaData = new EventMetaDataBuilderImpl();
		}
		((EventMetaDataBuilder) _eventMetaData).setChannel( aChannel );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMetaData( EventMetaData aMetaData ) {
		if ( !(aMetaData instanceof EventMetaDataBuilder) ) {
			aMetaData = new EventMetaDataBuilderImpl( aMetaData );
		}
		_eventMetaData = aMetaData;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public EventMetaDataBuilder getMetaData() {
		return (EventMetaDataBuilder) _eventMetaData;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
