// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.observer;

/**
 * As a descriptor describes something fixed, this class is immutable in the
 * sense that there are no setters for the attributes. The attributes must be
 * provided via the constructor.
 *
 * @param <E> The event type.
 * @param <O> the generic type
 * @param <EM> the generic type
 */
public class ObserverDescriptorImpl<E extends GenericEvent<?>, O extends Observer<E>, EM extends EventMatcher<E>> implements ObserverDescriptor<E, O, EM> {

	private O _observer;

	private EM _eventMatcher;

	/**
	 * Constructs the event listener descriptor with the given event listener
	 * and the given event matcher.
	 * 
	 * @param aEventListener The listener to be stored in the event listener
	 *        descriptor.
	 * @param aEventMatcher The matcher to be stored in the event listener
	 *        descriptor.
	 */
	public ObserverDescriptorImpl( O aEventListener, EM aEventMatcher ) {
		_observer = aEventListener;
		_eventMatcher = aEventMatcher;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onEvent( E aEvent ) {
		if ( _eventMatcher == null || _eventMatcher.isMatching( aEvent ) ) {
			_observer.onEvent( aEvent );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public O getObserver() {
		return _observer;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public EM getEventMatcher() {
		return _eventMatcher;
	}
}
