// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.properties;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.refcodes.data.Delimiter;
import org.refcodes.properties.JavaProperties.JavaPropertiesFactory;
import org.refcodes.properties.JsonProperties.JsonPropertiesFactory;
import org.refcodes.properties.TomlProperties.TomlPropertiesFactory;
import org.refcodes.properties.XmlProperties.XmlPropertiesFactory;
import org.refcodes.properties.YamlProperties.YamlPropertiesFactory;
import org.refcodes.runtime.ConfigLocator;

/**
 * Implementation of the {@link ResourceProperties} interface with support of so
 * called "{@link PolyglotProperties}" (or just "properties"). For
 * {@link PolyglotProperties}, see "https://en.wikipedia.org/wiki/.properties".
 */
public class PolyglotProperties extends AbstractResourcePropertiesDecorator<ResourceProperties> implements ResourceProperties {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Create a {@link PolyglotProperties} instance containing the elements as
	 * of {@link MutablePathMap#insert(Object)} using the default path delimiter
	 * "/" ({@link Delimiter#PATH}) for the path declarations: "Inspects the
	 * given object and adds all elements found in the given object. Elements of
	 * type {@link Map}, {@link Collection} and arrays are identified and
	 * handled as of their type: The path for each value in a {@link Map} is
	 * appended with its according key. The path for each value in a
	 * {@link Collection} or array is appended with its according index of
	 * occurrence (in case of a {@link List} or an array, its actual index). In
	 * case of reflection, the path for each member is appended with its
	 * according mamber's name. All elements (e.g. the members and values) are
	 * inspected recursively which results in the according paths of the
	 * terminating values."
	 *
	 * @param aObj The object from which the elements are to be added.
	 */
	public PolyglotProperties( Object aObj ) {
		this( new PolyglotPropertiesFactory(), aObj );
	}

	/**
	 * Create a {@link PolyglotProperties} instance containing the elements of
	 * the provided {@link Properties} instance using the default path delimiter
	 * "/" ({@link Delimiter#PATH}) for the path declarations
	 *
	 * @param aProperties the properties to be added.
	 */
	public PolyglotProperties( Properties aProperties ) {
		this( new PolyglotPropertiesFactory(), aProperties );
	}

	/**
	 * Create a {@link PolyglotProperties} instance containing the elements of
	 * the provided {@link PropertiesBuilder} instance using the default path
	 * delimiter "/" ({@link Delimiter#PATH}) for the path declarations
	 *
	 * @param aProperties the properties to be added.
	 */
	public PolyglotProperties( PropertiesBuilder aProperties ) {
		this( new PolyglotPropertiesFactory(), aProperties );
	}

	/**
	 * Create a {@link PolyglotProperties} instance containing the elements of
	 * the provided {@link Map} instance using the default path delimiter "/"
	 * ({@link Delimiter#PATH}) for the path declarations
	 *
	 * @param aProperties the properties to be added.
	 */
	public PolyglotProperties( Map<?, ?> aProperties ) {
		this( new PolyglotPropertiesFactory(), aProperties );
	}

	/**
	 * Loads the {@link PolyglotProperties} from the given {@link File}.
	 *
	 * @param aFile The {@link File} from which to load the properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( File aFile ) throws IOException, ParseException {
		this( new PolyglotPropertiesFactory(), aFile );
	}

	/**
	 * Loads the {@link PolyglotProperties} from the given {@link File}.
	 *
	 * @param aFile The {@link File} from which to load the properties.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( File aFile, char... aDelimiters ) throws IOException, ParseException {
		this( new PolyglotPropertiesFactory(), aFile, aDelimiters );
	}

	/**
	 * Loads or seeks the {@link PolyglotProperties} from the given
	 * {@link File}. A provided {@link ConfigLocator} describes the locations to
	 * additional crawl for the desired file.
	 * 
	 * @param aFile The {@link File} from which to load the properties.
	 * @param aConfigLocator The {@link ConfigLocator} describes the locations
	 *        to additional crawl for the desired file.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( File aFile, ConfigLocator aConfigLocator ) throws IOException, ParseException {
		this( new PolyglotPropertiesFactory(), aFile, aConfigLocator );
	}

	/**
	 * Loads or seeks the {@link PolyglotProperties} from the given
	 * {@link File}. A provided {@link ConfigLocator} describes the locations to
	 * additional crawl for the desired file.
	 * 
	 * @param aFile The {@link File} from which to load the properties.
	 * @param aConfigLocator The {@link ConfigLocator} describes the locations
	 *        to additional crawl for the desired file.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( File aFile, ConfigLocator aConfigLocator, char... aDelimiters ) throws IOException, ParseException {
		this( new PolyglotPropertiesFactory(), aFile, aConfigLocator, aDelimiters );
	}

	/**
	 * Loads the {@link PolyglotProperties} from the given file's path.
	 *
	 * @param aFilePath The path to the file from which to load the properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( String aFilePath ) throws IOException, ParseException {
		this( new PolyglotPropertiesFactory(), aFilePath );
	}

	/**
	 * Loads the {@link PolyglotProperties} from the given file's path.
	 *
	 * @param aFilePath The path to the file from which to load the properties.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( String aFilePath, char... aDelimiters ) throws IOException, ParseException {
		this( new PolyglotPropertiesFactory(), aFilePath, aDelimiters );
	}

	/**
	 * Loads the {@link PolyglotProperties} from the given file's path.
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( Class<?> aResourceClass, String aFilePath ) throws IOException, ParseException {
		this( new PolyglotPropertiesFactory(), aResourceClass, aFilePath );
	}

	/**
	 * Loads the {@link PolyglotProperties} from the given file's path.
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( Class<?> aResourceClass, String aFilePath, char... aDelimiters ) throws IOException, ParseException {
		this( new PolyglotPropertiesFactory(), aResourceClass, aFilePath, aDelimiters );
	}

	/**
	 * Loads the {@link PolyglotProperties} from the given file's path. A
	 * provided {@link ConfigLocator} describes the locations to additional
	 * crawl for the desired file.
	 * 
	 * @param aFilePath The path to the file from which to load the properties.
	 * @param aConfigLocator The {@link ConfigLocator} describes the locations
	 *        to additional crawl for the desired file.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( String aFilePath, ConfigLocator aConfigLocator ) throws IOException, ParseException {
		this( new PolyglotPropertiesFactory(), aFilePath, aConfigLocator );
	}

	/**
	 * Loads the {@link PolyglotProperties} from the given file's path. A
	 * provided {@link ConfigLocator} describes the locations to additional
	 * crawl for the desired file.
	 * 
	 * @param aFilePath The path to the file from which to load the properties.
	 * @param aConfigLocator The {@link ConfigLocator} describes the locations
	 *        to additional crawl for the desired file.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( String aFilePath, ConfigLocator aConfigLocator, char... aDelimiters ) throws IOException, ParseException {
		this( new PolyglotPropertiesFactory(), aFilePath, aConfigLocator, aDelimiters );
	}

	/**
	 * Loads the {@link PolyglotProperties} from the given file's path. A
	 * provided {@link ConfigLocator} describes the locations to additional
	 * crawl for the desired file. Finally (if nothing else succeeds) the
	 * properties are loaded by the provided class's class loader which takes
	 * care of loading the properties (in case the file path is a relative path,
	 * also the absolute path with a prefixed path delimiter "/" is probed).
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * @param aConfigLocator The {@link ConfigLocator} describes the locations
	 *        to additional crawl for the desired file.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( Class<?> aResourceClass, String aFilePath, ConfigLocator aConfigLocator ) throws IOException, ParseException {
		this( new PolyglotPropertiesFactory(), aFilePath, aConfigLocator );
	}

	/**
	 * Loads the {@link PolyglotProperties} from the given file's path. A
	 * provided {@link ConfigLocator} describes the locations to additional
	 * crawl for the desired file. Finally (if nothing else succeeds) the
	 * properties are loaded by the provided class's class loader which takes
	 * care of loading the properties (in case the file path is a relative path,
	 * also the absolute path with a prefixed path delimiter "/" is probed).
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * @param aConfigLocator The {@link ConfigLocator} describes the locations
	 *        to additional crawl for the desired file.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( Class<?> aResourceClass, String aFilePath, ConfigLocator aConfigLocator, char... aDelimiters ) throws IOException, ParseException {
		this( new PolyglotPropertiesFactory(), aResourceClass, aFilePath, aConfigLocator, aDelimiters );
	}

	/**
	 * Loads the {@link PolyglotProperties} from the given {@link URL}.
	 *
	 * @param aUrl The {@link URL} from which to read the properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( URL aUrl ) throws IOException, ParseException {
		this( new PolyglotPropertiesFactory(), aUrl );
	}

	/**
	 * Loads the {@link PolyglotProperties} from the given {@link URL}.
	 *
	 * @param aUrl The {@link URL} from which to read the properties.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( URL aUrl, char... aDelimiters ) throws IOException, ParseException {
		this( new PolyglotPropertiesFactory(), aUrl, aDelimiters );
	}

	/**
	 * Reads the {@link PolyglotProperties} from the given {@link InputStream}.
	 *
	 * @param aInputStream The {@link InputStream} from which to read the
	 *        properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( InputStream aInputStream ) throws IOException, ParseException {
		this( new PolyglotPropertiesFactory(), aInputStream );
	}

	/**
	 * Reads the {@link PolyglotProperties} from the given {@link InputStream}.
	 *
	 * @param aInputStream The {@link InputStream} from which to read the
	 *        properties.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( InputStream aInputStream, char... aDelimiters ) throws IOException, ParseException {
		this( new PolyglotPropertiesFactory(), aInputStream, aDelimiters );
	}

	// -------------------------------------------------------------------------

	/**
	 * Create a {@link PolyglotProperties} instance containing the elements as
	 * of {@link MutablePathMap#insert(Object)} using the default path delimiter
	 * "/" ({@link Delimiter#PATH}) for the path declarations: "Inspects the
	 * given object and adds all elements found in the given object. Elements of
	 * type {@link Map}, {@link Collection} and arrays are identified and
	 * handled as of their type: The path for each value in a {@link Map} is
	 * appended with its according key. The path for each value in a
	 * {@link Collection} or array is appended with its according index of
	 * occurrence (in case of a {@link List} or an array, its actual index). In
	 * case of reflection, the path for each member is appended with its
	 * according mamber's name. All elements (e.g. the members and values) are
	 * inspected recursively which results in the according paths of the
	 * terminating values."
	 *
	 * @param aPropertiesFactory The factory to use when constructing
	 *        {@link ResourceProperties} instances.
	 * @param aObj The object from which the elements are to be added.
	 */
	public PolyglotProperties( PolyglotPropertiesFactory aPropertiesFactory, Object aObj ) {
		super( aPropertiesFactory.toProperties( aObj ) );
	}

	/**
	 * Create a {@link PolyglotProperties} instance containing the elements of
	 * the provided {@link Properties} instance using the default path delimiter
	 * "/" ({@link Delimiter#PATH}) for the path declarations
	 *
	 * @param aProperties the properties to be added.
	 * @param aPropertiesFactory The factory to use when constructing
	 *        {@link ResourceProperties} instances.
	 */
	public PolyglotProperties( PolyglotPropertiesFactory aPropertiesFactory, Properties aProperties ) {
		super( aPropertiesFactory.toProperties( aProperties ) );
	}

	/**
	 * Create a {@link PolyglotProperties} instance containing the elements of
	 * the provided {@link PropertiesBuilder} instance using the default path
	 * delimiter "/" ({@link Delimiter#PATH}) for the path declarations
	 *
	 * @param aProperties the properties to be added.
	 * @param aPropertiesFactory The factory to use when constructing
	 *        {@link ResourceProperties} instances.
	 */
	public PolyglotProperties( PolyglotPropertiesFactory aPropertiesFactory, PropertiesBuilder aProperties ) {
		super( aPropertiesFactory.toProperties( aProperties ) );
	}

	/**
	 * Create a {@link PolyglotProperties} instance containing the elements of
	 * the provided {@link Map} instance using the default path delimiter "/"
	 * ({@link Delimiter#PATH}) for the path declarations
	 * 
	 * @param aPropertiesFactory The factory to use when constructing
	 *        {@link ResourceProperties} instances.
	 * @param aProperties the properties to be added.
	 */
	public PolyglotProperties( PolyglotPropertiesFactory aPropertiesFactory, Map<?, ?> aProperties ) {
		super( aPropertiesFactory.toProperties( aProperties ) );
	}

	/**
	 * Loads the {@link PolyglotProperties} from the given {@link File}.
	 *
	 * @param aFile The {@link File} from which to load the properties.
	 * @param aPropertiesFactory The factory to use when constructing
	 *        {@link ResourceProperties} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( PolyglotPropertiesFactory aPropertiesFactory, File aFile ) throws IOException, ParseException {
		super( aPropertiesFactory.toProperties( aFile ) );
	}

	/**
	 * Loads the {@link PolyglotProperties} from the given {@link File}.
	 *
	 * @param aFile The {@link File} from which to load the properties.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * @param aPropertiesFactory The factory to use when constructing
	 *        {@link ResourceProperties} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( PolyglotPropertiesFactory aPropertiesFactory, File aFile, char... aDelimiters ) throws IOException, ParseException {
		super( aPropertiesFactory.toProperties( aFile, aDelimiters ) );
	}

	/**
	 * Loads or seeks the {@link PolyglotProperties} from the given
	 * {@link File}. A provided {@link ConfigLocator} describes the locations to
	 * additional crawl for the desired file.
	 * 
	 * @param aFile The {@link File} from which to load the properties.
	 * @param aConfigLocator The {@link ConfigLocator} describes the locations
	 *        to additional crawl for the desired file.
	 * @param aPropertiesFactory The factory to use when constructing
	 *        {@link ResourceProperties} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( PolyglotPropertiesFactory aPropertiesFactory, File aFile, ConfigLocator aConfigLocator ) throws IOException, ParseException {
		super( aPropertiesFactory.toProperties( aFile, aConfigLocator ) );
	}

	/**
	 * Loads or seeks the {@link PolyglotProperties} from the given
	 * {@link File}. A provided {@link ConfigLocator} describes the locations to
	 * additional crawl for the desired file.
	 * 
	 * @param aFile The {@link File} from which to load the properties.
	 * @param aConfigLocator The {@link ConfigLocator} describes the locations
	 *        to additional crawl for the desired file.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * @param aPropertiesFactory The factory to use when constructing
	 *        {@link ResourceProperties} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( PolyglotPropertiesFactory aPropertiesFactory, File aFile, ConfigLocator aConfigLocator, char... aDelimiters ) throws IOException, ParseException {
		super( aPropertiesFactory.toProperties( aFile, aConfigLocator, aDelimiters ) );
	}

	/**
	 * Loads the {@link PolyglotProperties} from the given file's path.
	 *
	 * @param aFilePath The path to the file from which to load the properties.
	 * @param aPropertiesFactory The factory to use when constructing
	 *        {@link ResourceProperties} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( PolyglotPropertiesFactory aPropertiesFactory, String aFilePath ) throws IOException, ParseException {
		super( aPropertiesFactory.toProperties( aFilePath ) );
	}

	/**
	 * Loads the {@link PolyglotProperties} from the given file's path.
	 *
	 * @param aFilePath The path to the file from which to load the properties.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * @param aPropertiesFactory The factory to use when constructing
	 *        {@link ResourceProperties} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( PolyglotPropertiesFactory aPropertiesFactory, String aFilePath, char... aDelimiters ) throws IOException, ParseException {
		super( aPropertiesFactory.toProperties( aFilePath, aDelimiters ) );
	}

	/**
	 * Loads the {@link PolyglotProperties} from the given file's path.
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * @param aPropertiesFactory The factory to use when constructing
	 *        {@link ResourceProperties} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( PolyglotPropertiesFactory aPropertiesFactory, Class<?> aResourceClass, String aFilePath ) throws IOException, ParseException {
		super( aPropertiesFactory.toProperties( aResourceClass, aFilePath ) );
	}

	/**
	 * Loads the {@link PolyglotProperties} from the given file's path.
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * @param aPropertiesFactory The factory to use when constructing
	 *        {@link ResourceProperties} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( PolyglotPropertiesFactory aPropertiesFactory, Class<?> aResourceClass, String aFilePath, char... aDelimiters ) throws IOException, ParseException {
		super( aPropertiesFactory.toProperties( aResourceClass, aFilePath, aDelimiters ) );
	}

	/**
	 * Loads the {@link PolyglotProperties} from the given file's path. A
	 * provided {@link ConfigLocator} describes the locations to additional
	 * crawl for the desired file.
	 * 
	 * @param aFilePath The path to the file from which to load the properties.
	 * @param aConfigLocator The {@link ConfigLocator} describes the locations
	 *        to additional crawl for the desired file.
	 * @param aPropertiesFactory The factory to use when constructing
	 *        {@link ResourceProperties} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( PolyglotPropertiesFactory aPropertiesFactory, String aFilePath, ConfigLocator aConfigLocator ) throws IOException, ParseException {
		super( aPropertiesFactory.toProperties( aFilePath, aConfigLocator ) );
	}

	/**
	 * Loads the {@link PolyglotProperties} from the given file's path. A
	 * provided {@link ConfigLocator} describes the locations to additional
	 * crawl for the desired file.
	 * 
	 * @param aFilePath The path to the file from which to load the properties.
	 * @param aConfigLocator The {@link ConfigLocator} describes the locations
	 *        to additional crawl for the desired file.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * @param aPropertiesFactory The factory to use when constructing
	 *        {@link ResourceProperties} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( PolyglotPropertiesFactory aPropertiesFactory, String aFilePath, ConfigLocator aConfigLocator, char... aDelimiters ) throws IOException, ParseException {
		super( aPropertiesFactory.toProperties( aFilePath, aConfigLocator, aDelimiters ) );
	}

	/**
	 * Loads the {@link PolyglotProperties} from the given file's path. A
	 * provided {@link ConfigLocator} describes the locations to additional
	 * crawl for the desired file. Finally (if nothing else succeeds) the
	 * properties are loaded by the provided class's class loader which takes
	 * care of loading the properties (in case the file path is a relative path,
	 * also the absolute path with a prefixed path delimiter "/" is probed).
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * @param aConfigLocator The {@link ConfigLocator} describes the locations
	 *        to additional crawl for the desired file.
	 * @param aPropertiesFactory The factory to use when constructing
	 *        {@link ResourceProperties} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( PolyglotPropertiesFactory aPropertiesFactory, Class<?> aResourceClass, String aFilePath, ConfigLocator aConfigLocator ) throws IOException, ParseException {
		super( aPropertiesFactory.toProperties( aResourceClass, aFilePath, aConfigLocator ) );
	}

	/**
	 * Loads the {@link PolyglotProperties} from the given file's path. A
	 * provided {@link ConfigLocator} describes the locations to additional
	 * crawl for the desired file. Finally (if nothing else succeeds) the
	 * properties are loaded by the provided class's class loader which takes
	 * care of loading the properties (in case the file path is a relative path,
	 * also the absolute path with a prefixed path delimiter "/" is probed).
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * @param aConfigLocator The {@link ConfigLocator} describes the locations
	 *        to additional crawl for the desired file.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * @param aPropertiesFactory The factory to use when constructing
	 *        {@link ResourceProperties} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( PolyglotPropertiesFactory aPropertiesFactory, Class<?> aResourceClass, String aFilePath, ConfigLocator aConfigLocator, char... aDelimiters ) throws IOException, ParseException {
		super( aPropertiesFactory.toProperties( aResourceClass, aFilePath, aConfigLocator, aDelimiters ) );
	}

	/**
	 * Loads the {@link PolyglotProperties} from the given {@link URL}.
	 *
	 * @param aUrl The {@link URL} from which to read the properties.
	 * @param aPropertiesFactory The factory to use when constructing
	 *        {@link ResourceProperties} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( PolyglotPropertiesFactory aPropertiesFactory, URL aUrl ) throws IOException, ParseException {
		super( aPropertiesFactory.toProperties( aUrl ) );
	}

	/**
	 * Loads the {@link PolyglotProperties} from the given {@link URL}.
	 *
	 * @param aUrl The {@link URL} from which to read the properties.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * @param aPropertiesFactory The factory to use when constructing
	 *        {@link ResourceProperties} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( PolyglotPropertiesFactory aPropertiesFactory, URL aUrl, char... aDelimiters ) throws IOException, ParseException {
		super( aPropertiesFactory.toProperties( aUrl, aDelimiters ) );
	}

	/**
	 * Reads the {@link PolyglotProperties} from the given {@link InputStream}.
	 *
	 * @param aInputStream The {@link InputStream} from which to read the
	 *        properties.
	 * @param aPropertiesFactory The factory to use when constructing
	 *        {@link ResourceProperties} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( PolyglotPropertiesFactory aPropertiesFactory, InputStream aInputStream ) throws IOException, ParseException {
		super( aPropertiesFactory.toProperties( aInputStream ) );
	}

	/**
	 * Reads the {@link PolyglotProperties} from the given {@link InputStream}.
	 *
	 * @param aInputStream The {@link InputStream} from which to read the
	 *        properties.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * @param aPropertiesFactory The factory to use when constructing
	 *        {@link ResourceProperties} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotProperties( PolyglotPropertiesFactory aPropertiesFactory, InputStream aInputStream, char... aDelimiters ) throws IOException, ParseException {
		super( aPropertiesFactory.toProperties( aInputStream, aDelimiters ) );
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * The {@link PolyglotPropertiesFactory} is a meta factory using a
	 * collection of {@link ResourcePropertiesFactory} instances to deliver
	 * {@link ResourceProperties} instances. In case a properties file for a
	 * filename was not found, then the according factories filename extension
	 * ({@link ResourcePropertiesFactory#getFilenameSuffixes()}) is append to
	 * the filename and probing is repeated. Any factory method such as
	 * {@link #toProperties(Map)}, {@link #toProperties(Object)},
	 * {@link #toProperties(Properties)} or
	 * {@link #toProperties(PropertiesBuilder)} will return
	 * {@link ResourceProperties} created by the first added
	 * {@link ResourcePropertiesFactory} instance.
	 */
	public static class PolyglotPropertiesFactory implements ResourcePropertiesFactory {

		// /////////////////////////////////////////////////////////////////////
		// VARIABLES:
		// /////////////////////////////////////////////////////////////////////

		private List<ResourcePropertiesFactory> _factories = new ArrayList<>();

		// /////////////////////////////////////////////////////////////////////
		// CONSTRUCTORS:
		// /////////////////////////////////////////////////////////////////////

		/**
		 * Initializes the {@link PolyglotPropertiesFactory} with a predefined
		 * set of {@link ResourcePropertiesFactory} instances.
		 */
		public PolyglotPropertiesFactory() {
			_factories.add( new TomlPropertiesFactory() );
			_factories.add( new YamlPropertiesFactory() );
			_factories.add( new XmlPropertiesFactory() );
			_factories.add( new JsonPropertiesFactory() );
			// Last as them Java properties eat all notations |-->
			_factories.add( new JavaPropertiesFactory() );
			// Last as them Java properties eat all notations <--|
		}

		/**
		 * Initializes the {@link PolyglotPropertiesFactory} with the given
		 * {@link ResourcePropertiesFactory} instances. Them factories will be
		 * queried in the order being provided.
		 * 
		 * @param aFactories The factories to be added.
		 */
		public PolyglotPropertiesFactory( ResourcePropertiesFactory... aFactories ) {
			for ( ResourcePropertiesFactory eFactory : aFactories )
				_factories.add( eFactory );
		}

		// /////////////////////////////////////////////////////////////////////
		// METHODS:
		// /////////////////////////////////////////////////////////////////////

		/**
		 * Returns the filename extensions of the ResourcePropertiesFactory in
		 * the given order. {@inheritDoc}
		 */
		@Override
		public String[] getFilenameSuffixes() {
			List<String> theExtensions = new ArrayList<>();
			String[] eExtensions;
			for ( int i = 0; i < _factories.size(); i++ ) {
				eExtensions = _factories.get( i ).getFilenameSuffixes();
				if ( eExtensions != null ) {
					for ( int j = 0; j < eExtensions.length; j++ ) {
						theExtensions.add( eExtensions[j] );
					}
				}
			}
			return theExtensions.toArray( new String[theExtensions.size()] );
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public ResourceProperties toProperties( Class<?> aResourceClass, String aFilePath, ConfigLocator aConfigLocator, char... aDelimiters ) throws IOException, ParseException {
			Exception theCause = null;

			// Keep the precedence |-->
			try {
				return toPropertiesFromFilePath( aFilePath, aConfigLocator, aDelimiters );
			}
			catch ( Exception e ) {
				if ( theCause == null ) theCause = e;
			}
			// Keep the precedence <--|

			for ( ResourcePropertiesFactory eFactory : _factories ) {
				try {
					return eFactory.toProperties( aResourceClass, aFilePath, aConfigLocator, aDelimiters );
				}
				catch ( Exception e ) {
					if ( theCause == null ) theCause = e;
				}
			}
			// Test with the factorie's filename extension |-->
			String[] eExtensions;
			for ( ResourcePropertiesFactory eFactory : _factories ) {
				eExtensions = eFactory.getFilenameSuffixes();
				if ( eExtensions != null ) {
					for ( String eExtension : eExtensions ) {
						try {
							return eFactory.toProperties( aResourceClass, aFilePath + eExtension, aConfigLocator, aDelimiters );
						}
						catch ( Exception e ) {
							if ( theCause == null ) theCause = e;
						}
					}
				}
			}
			// Test with the factorie's filename extension <--|
			if ( theCause instanceof IOException ) throw (IOException) theCause;
			if ( theCause instanceof ParseException ) throw (ParseException) theCause;
			throw new IOException( "No configuration file with name (scheme) <" + aFilePath + "> found, where a suffix (if applicable) might by one of the following: " + Arrays.toString( getFilenameSuffixes() ) );
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public ResourceProperties toProperties( File aFile, ConfigLocator aConfigLocator, char... aDelimiters ) throws IOException, ParseException {
			Exception theCause = null;

			// Keep the precedence |-->
			try {
				return toPropertiesFromFilePath( aFile.getPath(), aConfigLocator, aDelimiters );
			}
			catch ( Exception e ) {
				if ( theCause == null ) theCause = e;
			}
			// Keep the precedence <--|

			for ( ResourcePropertiesFactory eFactory : _factories ) {
				try {
					return eFactory.toProperties( aFile, aConfigLocator, aDelimiters );
				}
				catch ( Exception e ) {
					if ( theCause == null ) theCause = e;
				}
			}
			// Test with the factorie's filename extension |-->
			File eFile;
			String[] eExtensions;
			for ( ResourcePropertiesFactory eFactory : _factories ) {
				eExtensions = eFactory.getFilenameSuffixes();
				if ( eExtensions != null ) {
					for ( String eExtension : eExtensions ) {
						eFile = new File( aFile.getAbsolutePath() + eExtension );
						try {
							return eFactory.toProperties( eFile, aConfigLocator, aDelimiters );
						}
						catch ( Exception e ) {
							if ( theCause == null ) theCause = e;
						}
					}
				}
			}
			// Test with the factorie's filename extension <--|
			if ( theCause instanceof IOException ) throw (IOException) theCause;
			if ( theCause instanceof ParseException ) throw (ParseException) theCause;
			throw new IOException( "No configuration file with name (scheme) <" + aFile.toPath() + "> found, where a suffix (if applicable) might by one of the following: " + Arrays.toString( getFilenameSuffixes() ) );
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public ResourceProperties toProperties( InputStream aInputStream, char... aDelimiters ) throws IOException, ParseException {
			Exception theCause = null;
			byte[] theBytes = toByteArray( aInputStream );
			for ( ResourcePropertiesFactory eFactory : _factories ) {
				try {
					return eFactory.toProperties( new ByteArrayInputStream( theBytes ), aDelimiters );
				}
				catch ( Exception e ) {
					if ( theCause == null ) theCause = e;
				}
			}
			if ( theCause instanceof IOException ) throw (IOException) theCause;
			if ( theCause instanceof ParseException ) throw (ParseException) theCause;
			throw new IOException( "No configuration file for the given InputStream found!" );
		}

		/**
		 * Will return {@link ResourceProperties} created by the first added
		 * {@link ResourcePropertiesFactory} instance. {@inheritDoc}
		 */
		@Override
		public ResourceProperties toProperties( Map<?, ?> aProperties ) {
			return _factories.get( 0 ).toProperties( aProperties );
		}

		/**
		 * Will return {@link ResourceProperties} created by the first added
		 * {@link ResourcePropertiesFactory} instance. {@inheritDoc}
		 */
		@Override
		public ResourceProperties toProperties( Object aObj ) {
			return _factories.get( 0 ).toProperties( aObj );
		}

		/**
		 * Will return {@link ResourceProperties} created by the first added
		 * {@link ResourcePropertiesFactory} instance. {@inheritDoc}
		 */
		@Override
		public ResourceProperties toProperties( Properties aProperties ) {
			return _factories.get( 0 ).toProperties( aProperties );
		}

		/**
		 * Will return {@link ResourceProperties} created by the first added
		 * {@link ResourcePropertiesFactory} instance. {@inheritDoc}
		 */
		@Override
		public ResourceProperties toProperties( PropertiesBuilder aProperties ) {
			return _factories.get( 0 ).toProperties( aProperties );
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public ResourceProperties toProperties( String aFilePath, ConfigLocator aConfigLocator, char... aDelimiters ) throws IOException, ParseException {
			return toProperties( null, aFilePath, aConfigLocator, aDelimiters );
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public ResourceProperties toProperties( URL aUrl, char... aDelimiters ) throws IOException, ParseException {
			Exception theCause = null;

			// Try file extension's factory first |-->
			try {
				ResourcePropertiesFactory theFactory = fromFilenameExtension( aUrl.toExternalForm() );
				if ( theFactory != null ) return theFactory.toProperties( aUrl, aDelimiters );
			}
			catch ( Exception e ) {
				if ( theCause == null ) theCause = e;
			}
			// Try file extension's factory first <--|

			for ( ResourcePropertiesFactory eFactory : _factories ) {
				try {
					return eFactory.toProperties( aUrl, aDelimiters );
				}
				catch ( Exception e ) {
					if ( theCause == null ) theCause = e;
				}
			}
			// Test with the factorie's filename extension |-->
			URL eUrl;
			String[] eExtensions;
			for ( ResourcePropertiesFactory eFactory : _factories ) {
				eExtensions = eFactory.getFilenameSuffixes();
				if ( eExtensions != null ) {
					for ( String eExtension : eExtensions ) {
						eUrl = new URL( aUrl.toExternalForm() + eExtension );
						try {
							return eFactory.toProperties( eUrl, aDelimiters );
						}
						catch ( Exception e ) {
							if ( theCause == null ) theCause = e;
						}
					}
				}
			}
			// Test with the factorie's filename extension <--|
			if ( theCause instanceof IOException ) throw (IOException) theCause;
			if ( theCause instanceof ParseException ) throw (ParseException) theCause;
			throw new IOException( "No configuration file with (base) URL <" + aUrl.toExternalForm() + "> found!" );
		}

		// /////////////////////////////////////////////////////////////////////
		// HELPER:
		// /////////////////////////////////////////////////////////////////////

		private ResourcePropertiesFactory fromFilenameExtension( String aFilePath ) {
			String[] eExtensions;
			for ( ResourcePropertiesFactory eFactory : _factories ) {
				eExtensions = eFactory.getFilenameSuffixes();
				if ( eExtensions != null ) {
					for ( String eExtension : eExtensions ) {
						if ( aFilePath.endsWith( eExtension ) ) {
							return eFactory;
						}
					}
				}
			}
			return null;
		}

		private ResourceProperties toPropertiesFromFilePath( String aFilePath, ConfigLocator aConfigLocator, char... aDelimiters ) throws IOException, ParseException {
			Exception theCause = null;
			File theFile = new File( aFilePath );
			File eFile;
			if ( theFile.exists() ) {
				try {
					ResourcePropertiesFactory theFactory = fromFilenameExtension( theFile.getAbsolutePath() );
					if ( theFactory != null ) {
						return theFactory.toProperties( theFile, ConfigLocator.ABSOLUTE, aDelimiters );
					}
				}
				catch ( IOException | ParseException e ) {
					if ( theCause == null ) theCause = e;
				}
			}
			else {
				for ( ResourcePropertiesFactory eFactory : _factories ) {
					for ( String eExtension : eFactory.getFilenameSuffixes() ) {
						eFile = new File( aFilePath + eExtension );
						if ( eFile.exists() ) {
							try {
								return eFactory.toProperties( eFile, ConfigLocator.ABSOLUTE, aDelimiters );
							}
							catch ( IOException | ParseException e ) {
								if ( theCause == null ) theCause = e;
							}
						}
					}
				}
				if ( aConfigLocator != null && aConfigLocator != ConfigLocator.ABSOLUTE ) {
					for ( File eDir : aConfigLocator.getFolders() ) {
						for ( ResourcePropertiesFactory eFactory : _factories ) {
							for ( String eExtension : eFactory.getFilenameSuffixes() ) {
								eFile = new File( eDir, aFilePath + eExtension );
								if ( eFile.exists() ) {
									try {
										return eFactory.toProperties( eFile, ConfigLocator.ABSOLUTE, aDelimiters );
									}
									catch ( IOException | ParseException e ) {
										if ( theCause == null ) theCause = e;
									}
								}
							}
						}
					}
				}
			}
			if ( theCause instanceof IOException ) throw (IOException) theCause;
			if ( theCause instanceof ParseException ) throw (ParseException) theCause;
			throw new IOException( "No configuration file with name (scheme) <" + aFilePath + "> found, where the suffix may by one of the following: " + Arrays.toString( getFilenameSuffixes() ) );
		}

		// /////////////////////////////////////////////////////////////////////
		// HOOKS:
		// /////////////////////////////////////////////////////////////////////

		/**
		 * Converts a byte array from an {@link InputStream}.
		 * 
		 * @param aInputStream The {@link InputStream} to be converted.
		 * 
		 * @return The byte array from the {@link InputStream}.
		 * 
		 * @throws IOException Thrown in case there were problems reading the
		 *         {@link InputStream}.
		 */
		public static byte[] toByteArray( InputStream aInputStream ) throws IOException {
			ByteArrayOutputStream theBuffer = new ByteArrayOutputStream();
			int eRead;
			byte[] theData = new byte[16384];
			while ( (eRead = aInputStream.read( theData, 0, theData.length )) != -1 ) {
				theBuffer.write( theData, 0, eRead );
			}
			theBuffer.flush();
			return theBuffer.toByteArray();
		}
	}
}
