// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

/**
 * <h2>What is this package for?</h2> This artifact provides means to read
 * configuration data from various different locations such as properties from
 * JAR files, file system files or remote HTTP addresses or GIT repositories.
 * <h2>Jump start</h2>
 * 
 * <pre>
 * import static org.refcodes.properties.PropertiesSugar.*;
 * 
 * // ...
 * public class Foo {
 * 	public void bar() throws IOException, ParseException {
 * 		// ...
 * 		PropertiesBuilder theProperties = fromProperties( toProperty( "message", "Hello world!" ), toProperty( "foo", "bar" ) );
 * 		fileToTomlProperties( theProperties, "application.toml" );
 * 		// ...
 * 		Properties thePrecedence = toPrecedence( fromSystemProperties(), fromEnvironmentVariables(), seekFromTomlProperties( "application.toml" ) );
 * 		FooConfig theConfig = thePrecedence.toType( FooConfig.class );
 * 		// ...
 * 	}
 * 
 * 	// ...
 * 	public static class FooConfig {
 * 		String message;
 * 		int frequency;
 * 		TemperatureUnit unit;
 * 		int port;
 * 	}
 * 	// ...
 * }
 * </pre>
 * 
 * See also the blog post <a target="_blank" href=
 * "http://www.refcodes.org/blog/dead_simple_java_application_configuration">Dead
 * simple Java application configuration</a>.
 * <h2>How do I get set up?</h2> To get up and running, include the following
 * dependency (without the three dots "...") in your <code>pom.xml</code>:
 * 
 * <pre>
 * {@code
 * <dependencies>
 * 	...
 * 	<dependency>
 * 		<artifactId>refcodes-properties</artifactId>
 * 		<groupId>org.refcodes</groupId>
 * 		<version>1.1.9</version>
 * 	</dependency>
 * 	...
 * </dependencies>
 * }
 * </pre>
 * 
 * The artifact is hosted directly at
 * <a target="_blank" href="http://search.maven.org">Maven Central</a>. Jump
 * straight to the source codes at <a target="_blank" href=
 * "https://bitbucket.org/refcodes/refcodes-properties">Bitbucket</a>. Read the
 * artifact's javadoc at <a target="_blank" href=
 * "http://www.javadoc.io/doc/org.refcodes/refcodes-properties">javadoc.io</a>.
 * If you also want to observe your properties (e.g. listen for
 * <code>create</code> , <code>update</code> or <code>delete</code> operations),
 * you may instead add the following dependency to your <code>pom.xml</code>:
 * 
 * <pre>
 * {@code
 * <dependencies>
 * 	...
 * 	<dependency>
 * 		<artifactId>refcodes-properties-ext-observer</artifactId>
 * 		<groupId>org.refcodes</groupId>
 * 		<version>1.1.9</version>
 * 	</dependency>
 * 	...
 * </dependencies>
 * }
 * </pre>
 * 
 * The artifact is hosted directly at
 * <a target="_blank" href="http://search.maven.org">Maven Central</a>. Jump
 * straight to the source codes at <a target="_blank" href=
 * "https://bitbucket.org/refcodes/refcodes-properties-ext">Bitbucket</a>. Read
 * the artifact's javadoc at <a target="_blank" href=
 * "http://www.javadoc.io/doc/org.refcodes/refcodes-properties-ext-observer">javadoc.io</a>.
 * <h2>How do I get started?</h2> Use the *static import* syntactic sugar to
 * easily harness the <a target="_blank" href=
 * "http://www.javadoc.io/doc/org.refcodes/refcodes-properties"><code>refcoces-configuration</code></a>
 * features.
 * 
 * <pre>
 * import static org.refcodes.properties.PropertiesSugar.*;
 * // ...
 * </pre>
 * 
 * Create some <code>properties</code> and play around with them:
 * 
 * <pre>
 * // ...
 * PropertiesBuilder theProperties = fromProperties( toProperty( "/user/firstName", "Nolan" ), toProperty( "/user/lastName", "Bushnell" ), toProperty( "/commodore/user/firstName", "Jack" ), toProperty( "/commodore/user/lastName", "Tramiel" ) );
 * // ...
 * </pre>
 * 
 * The content of your <code>properties</code> looks as follows:
 * 
 * <pre>
 * /user/firstName=Nolan
 * /user/lastName=Bushnell
 * /commodore/user/lastName=Tramiel
 * /commodore/user/firstName=Jack
 * </pre>
 * 
 * The {@link org.refcodes.properties.Properties.PropertiesBuilder} inherits
 * from the {@link java.util.Map} interface, it is fully compatible with the
 * <a target="_blank" href=
 * "https://en.wikipedia.org/wiki/Java_collections_framework"><code>Java collections framework</code></a>.
 * You may have noted that the <code>keys</code> in your <code>properties</code>
 * look like <code>path</code> declarations. This is no coincidence, you can now
 * manipulate your <code>properties</code>using (sub-)<code>paths</code>. See
 * more in the article <a target="_blank" href=
 * "http://www.refcodes.org/blog/the_canonical_model_an_ace_upon_your_sleeve">The
 * canonical model, an ace upon your sleeve</a>. E.g. you may now retrieve the
 * <code>commodore</code> specific properties (the <code>properties</code> below
 * the <code>/commodore</code> path):
 * 
 * <pre>
 * // ...
 * Properties theCommodoreProperties = theProperties.retrieveFrom( "/commodore" );
 * // ...
 * </pre>
 * 
 * This results in your <code>commodore</code> specific <code>properties</code>
 * to look as such:
 * 
 * <pre>
 * /user/lastName=Tramiel
 * /user/firstName=Jack
 * </pre>
 * 
 * There are many more features hidden in the
 * {@link org.refcodes.properties.Properties} type, just browse the Javadoc. The
 * {@link org.refcodes.properties.Properties} type is the <code>read-only</code>
 * super-type of the
 * {@link org.refcodes.properties.Properties.PropertiesBuilder} type. Common
 * functionality produces {@link org.refcodes.properties.Properties} instances
 * which easily can be converted into a mutable
 * {@link org.refcodes.properties.Properties.PropertiesBuilder} instance as
 * follows:
 * 
 * <pre>
 * // ...
 * PropertiesBuilder theBuilder = toPropertiesBuilder( theCommodoreProperties );
 * // ...
 * </pre>
 * 
 * <h3>Storing properties</h3> Considering the example from above, storing
 * <code>properties</code> is as easy as this:
 * 
 * <pre>
 * // ...
 * ResourcePropertiesBuilder theResourceProperties = saveToTomlProperties( theProperties, "/some/path/to/my/properties.toml" );
 * // ...
 * </pre>
 * 
 * You can make the <code>library</code> choose a suitable place for you where
 * to save your <code>properties</code> to; being near the launcher
 * (<code>JAR</code>) of your application:
 * 
 * <pre>
 * // ...
 * ResourcePropertiesBuilder theResourceProperties = fileToTomlProperties( theProperties, "properties.toml" );
 * // ...
 * </pre>
 * 
 * See the {@link org.refcodes.runtime.RuntimeUtility} on how a suitable
 * location is determines by the <code>library</code>.
 * <h3>Loading properties</h3> Considering the example from above, loading
 * <code>properties</code> back again is as easy as this:
 * 
 * <pre>
 * // ...
 * ResourcePropertiesBuilder theResourceProperties = loadFromTomlProperties( "/some/path/to/my/properties.toml" ):
 * // ...
 * </pre>
 * 
 * You can make the <code>library</code> seek for a suitable
 * <code>properties</code> for you to load; being near the launcher
 * (<code>JAR</code>) of your application:
 * 
 * <pre>
 * // ...
 * ResourcePropertiesBuilder theResourceProperties = seekFromTomlProperties( "properties.toml" );
 * // ...
 * </pre>
 * 
 * See the {@link org.refcodes.runtime.RuntimeUtility} on how a suitable
 * location is determines by the <code>library</code>.
 * <h3>Properties parsers</h3> There are some notations being supported by the
 * <a target="_blank" href=
 * "http://www.javadoc.io/doc/org.refcodes/refcodes-properties"><code>refcodes-properties</code></a>
 * artifact:
 * <ul>
 * <li><a target="_blank" href=
 * "https://en.wikipedia.org/wiki/.properties"><code>Java</code></a> based
 * properties
 * <li><a target="_blank" href=
 * "https://en.wikipedia.org/wiki/TOML"><code>TOML</code></a> based properties
 * <li><a target="_blank" href=
 * "https://en.wikipedia.org/wiki/XML"><code>XML</code></a> based properties
 * <li><a target="_blank" href=
 * "https://en.wikipedia.org/wiki/YAML"><code>YAML</code></a>based properties
 * <li><a target="_blank" href=
 * "https://en.wikipedia.org/wiki/JSON"><code>JSON</code></a> based properties
 * </ul>
 * <h3>Profiles from properties</h3> A <code>profile</code> is identified by the
 * first level path hierarchy in your originating <code>properties</code>. In
 * the example above, <code>commodore</code> represents a profile specific
 * configuration which we can use to get our <code>profile</code> view:
 * 
 * <pre>
 * // ...
 * Properties theProfile = fromProfile( theProperties, "commodore" );
 * // ...
 * </pre>
 * 
 * We can also specify a <code>property</code> in our <code>properties</code>
 * for the path <code>/runtime/profiles</code> identifying the
 * <code>profiles</code> to be considered (comma separated). See also the
 * {@link org.refcodes.properties.ProfilePropertiesProjection} type as well as
 * the {@link org.refcodes.properties.ProfilePropertiesDecorator} type on more
 * usages.
 * <h3>Schedule reloading of properties</h3> Using the
 * {@link org.refcodes.properties.ResourceProperties.ResourcePropertiesBuilder}
 * form above which we attached to a file, we now can schedule a *reload* of the
 * <code>properties</code>:
 * 
 * <pre>
 * // ...
 * ResourceProperties theScheduled = schedule( theResourceProperties, 5000, ReloadMode.ORPHAN_REMOVAL );
 * // ...
 * </pre>
 * 
 * We actually encapsulate the <code>properties</code> with a *schedule*
 * <a target="_blank" href=
 * "https://en.wikipedia.org/wiki/Decorator_pattern"><code>decorator</code></a>
 * which reloads the encapsulated <code>properties</code> accordingly: Reload
 * them each <code>5000</code> milliseconds and remove any
 * <code>properties</code> not found in the attached resource (e.g.
 * {@link java.io.File} ) from your <code>properties</code> instance.
 * <h3>Observe properties</h3> You may also listen to any <code>create</code>,
 * <code>update</code> or <code>delete</code> changes applied to your
 * properties. To do so, you must encapsulate your
 * {@link org.refcodes.properties.ResourceProperties.ResourcePropertiesBuilder}
 * instance with an <a target="_blank" href=
 * "https://en.wikipedia.org/wiki/Observer_pattern">observable</a>
 * <a target="_blank" href=
 * "https://en.wikipedia.org/wiki/Decorator_pattern"><code>decorator</code></a>
 * of type <a href=
 * "https://static.javadoc.io/org.refcodes/refcodes-properties-ext-observer/2.1.2/org/refcodes/configuration/ext/observer/ObservableResouceProperties.ObservableResourcePropertiesBuilder.html"><code>ObservableResourcePropertiesBuilder</code></a>
 * This <a target="_blank" href=
 * "https://en.wikipedia.org/wiki/Observer_pattern"><code>observable</code></a>
 * fires a sub-type of the <a href=
 * "https://static.javadoc.io/org.refcodes/refcodes-properties-ext-observer/2.1.2/org/refcodes/configuration/ext/observer/PropertyEvent.html"><code>PropertyEvent</code></a>
 * upon updates applied to the <code>properties</code> to each subscriber of
 * those <code>events</code>:
 * 
 * <pre>
 * {@code
 * import static org.refcodes.properties.ext.observer.ObservablePropertiesSugar.*;
 * 	// ...
 * 	PropertiesBuilder theProperties = toPropertiesBuilder();
 * 	ObservablePropertiesBuilder theObservable = observe( theProperties );
 * 	theObservable.subscribeObserver( aEvent -> {
 * 		if ( aEvent.getAction() == PropertyAction.PROPERTY_UPDATED ) {
 * 			System.out.println( aEvent.getClass().getSimpleName() + " (" + aEvent.getAction() + ") --> " + aEvent.getKey() + " := " + aEvent.getValue() );
 * 		}
 * 	} );
 * 	theObservable.put( "/user/firstName", "Nolan" );
 * 	theObservable.put( "/user/lastName", "Bushnell" );
 * 	theObservable.put( "/user/firstName", "Jack" );
 * 	theObservable.put( "/user/lastName", "Tramiel" );
 * 	theObservable.remove( "/user/firstName" );
 * 	theObservable.remove( "/user/lastName" );
 * 	// ...
 * }
 * </pre>
 * 
 * For the example above to work, make sure to include the
 * <a target="_blank" href=
 * "https://www.javadoc.io/doc/org.refcodes/refcodes-properties-ext-observer/1.1.9"><code>refcodes-properties-ext-observer</code></a>
 * dependency (see at the beginning of this article) in your project's
 * <code>pom.xml</code>. The <code>lambda</code> being subscribed acts as a
 * <code>listener</code> which is a {@link java.lang.FunctionalInterface} of
 * type <a href=
 * "https://static.javadoc.io/org.refcodes/refcodes-properties-ext-observer/2.1.2/org/refcodes/configuration/ext/observer/PropertiesObserver.html"><code>PropertiesObserver</code></a>
 * with which you explicitly may listen to the three event types <a href=
 * "https://static.javadoc.io/org.refcodes/refcodes-properties-ext-observer/2.1.2/org/refcodes/configuration/ext/observer/PropertyCreatedEvent.html"><code>PropertyCreatedEvent</code></a>,
 * <a href=
 * "https://static.javadoc.io/org.refcodes/refcodes-properties-ext-observer/2.1.2/org/refcodes/configuration/ext/observer/PropertyUpdatedEvent.html"><code>PropertyUpdatedEvent</code></a>
 * and <a href=
 * "https://static.javadoc.io/org.refcodes/refcodes-properties-ext-observer/2.1.2/org/refcodes/configuration/ext/observer/PropertyDeletedEvent.html"><code>PropertyDeletedEvent</code></a>
 * as well as to the super-type <a href=
 * "https://static.javadoc.io/org.refcodes/refcodes-properties-ext-observer/2.1.2/org/refcodes/configuration/ext/observer/PropertyEvent.html">[<code>PropertyEvent</code>]</a>.
 * Observing just the <a href=
 * "https://static.javadoc.io/org.refcodes/refcodes-properties-ext-observer/2.1.2/org/refcodes/configuration/ext/observer/PropertyEvent.html"><code>PropertyEvent</code></a>
 * type *catches* all of its sub-types (as we did in the example above).
 * <h3>Multiple properties precedence</h3> We can combine multiple
 * {@link org.refcodes.properties.Properties} instances behind a
 * <a target="_blank" href=
 * "https://en.wikipedia.org/wiki/Composite_pattern">composite</a> behaving just
 * like a {@link org.refcodes.properties.Properties} type:
 * 
 * <pre>
 * // ...
 * Properties theProperties = toPrecedence( fromSystemProperties(), fromEnvironmentVariables(), seekFromJavaProperties( "application.config" ) );
 * // ...
 * </pre>
 * 
 * Above we created a <a target="_blank" href=
 * "https://en.wikipedia.org/wiki/Composite_pattern">composite</a> containing
 * various various <code>properties</code> instances, the first one overrules
 * the second one and the second one overrules the third one when accessing the
 * resulting {@link org.refcodes.properties.Properties} <a target="_blank" href=
 * "https://en.wikipedia.org/wiki/Composite_pattern">composite</a>. Take a look
 * at the {@link org.refcodes.properties.SystemProperties} as well as at the
 * {@link org.refcodes.properties.EnvironmentProperties} instances we added:
 * They provide means to access <code>Java</code> 's <a target="_blank" href=
 * "https://docs.oracle.com/javase/tutorial/essential/environment/sysprop.html"><code>system properties</code></a>
 * or the <code>operating system</code>'s <a target="_blank" href=
 * "https://en.wikipedia.org/wiki/Environment_variable"><code>environment variables</code></a>
 * within your <code>properties</code>..
 * <h2>Under the hood</h2> The <a target="_blank" href=
 * "https://en.wikipedia.org/wiki/Canonical_model"><code>canonical model</code></a>
 * pattern is an ace up your sleeve in order to open your libraries for
 * functionality otherwise to be implemented in a tiresome, error prone and
 * redundant way. As you settle upon a <a target="_blank" href=
 * "https://en.wikipedia.org/wiki/Canonical_model"><code>canonical model</code></a>
 * within your library, your library's will be able to interact with any
 * existing and yet to be implemented functionality on top of your
 * <a target="_blank" href=
 * "https://en.wikipedia.org/wiki/Canonical_model"><code>canonical model</code></a>,
 * making your bits and pieces work together magically. The
 * {@link org.refcodes.struct.CanonicalMap} is the super-type of the
 * {@link org.refcodes.properties.Properties} related types. Read more in the
 * blog post <a target="_blank" href=
 * "http://www.refcodes.org/blog/the_canonical_model_an_ace_upon_your_sleeve">The
 * canonical model, an ace upon your sleeve</a>.
 * <h2>Examples</h2> For examples and usage, please take a look at the according
 * <a target="_blank" href=
 * "https://bitbucket.org/refcodes/refcodes-properties/src/master/src/test/java/org/refcodes/configuration/?at=master"><code>Unit-Tests</code></a>.
 * For examples and usage on the observable extensions, please take a look at
 * the according <a target="_blank" href=
 * "https://bitbucket.org/refcodes/refcodes-properties-ext/src/master/refcodes-properties-ext-observer/src/test/java/org/refcodes/configuration/ext/observer/?at=master"><code>Unit-Tests</code></a>.
 */
package org.refcodes.properties;