// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.properties;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.refcodes.data.Delimiter;
import org.refcodes.properties.JavaPropertiesBuilder.JavaPropertiesBuilderFactory;
import org.refcodes.properties.JsonPropertiesBuilder.JsonPropertiesBuilderFactory;
import org.refcodes.properties.ResourceProperties.ResourcePropertiesBuilder;
import org.refcodes.properties.ResourcePropertiesFactory.ResourcePropertiesBuilderFactory;
import org.refcodes.properties.TomlPropertiesBuilder.TomlPropertiesBuilderFactory;
import org.refcodes.properties.XmlPropertiesBuilder.XmlPropertiesBuilderFactory;
import org.refcodes.properties.YamlPropertiesBuilder.YamlPropertiesBuilderFactory;
import org.refcodes.runtime.ConfigLocator;

/**
 * Implementation of the {@link ResourcePropertiesBuilder} interface with
 * support of so called "{@link PolyglotPropertiesBuilder}" (or just
 * "properties"). For {@link PolyglotPropertiesBuilder}, see
 * "https://en.wikipedia.org/wiki/.properties".
 */
public class PolyglotPropertiesBuilder extends AbstractResourcePropertiesBuilderDecorator<ResourcePropertiesBuilder> implements ResourcePropertiesBuilder {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Create a {@link PolyglotPropertiesBuilder} instance containing the
	 * elements as of {@link MutablePathMap#insert(Object)} using the default
	 * path delimiter "/" ({@link Delimiter#PATH}) for the path declarations:
	 * "Inspects the given object and adds all elements found in the given
	 * object. Elements of type {@link Map}, {@link Collection} and arrays are
	 * identified and handled as of their type: The path for each value in a
	 * {@link Map} is appended with its according key. The path for each value
	 * in a {@link Collection} or array is appended with its according index of
	 * occurrence (in case of a {@link List} or an array, its actual index). In
	 * case of reflection, the path for each member is appended with its
	 * according mamber's name. All elements (e.g. the members and values) are
	 * inspected recursively which results in the according paths of the
	 * terminating values."
	 *
	 * @param aObj The object from which the elements are to be added.
	 */
	public PolyglotPropertiesBuilder( Object aObj ) {
		this( new PolyglotPropertiesBuilderFactory(), aObj );
	}

	/**
	 * Create a {@link PolyglotPropertiesBuilder} instance containing the
	 * elements of the provided {@link Properties} instance using the default
	 * path delimiter "/" ({@link Delimiter#PATH}) for the path declarations
	 *
	 * @param aPropertiesBuilder the properties to be added.
	 */
	public PolyglotPropertiesBuilder( Properties aPropertiesBuilder ) {
		this( new PolyglotPropertiesBuilderFactory(), aPropertiesBuilder );
	}

	/**
	 * Create a {@link PolyglotPropertiesBuilder} instance containing the
	 * elements of the provided {@link PropertiesBuilder} instance using the
	 * default path delimiter "/" ({@link Delimiter#PATH}) for the path
	 * declarations
	 *
	 * @param aPropertiesBuilder the properties to be added.
	 */
	public PolyglotPropertiesBuilder( PropertiesBuilder aPropertiesBuilder ) {
		this( new PolyglotPropertiesBuilderFactory(), aPropertiesBuilder );
	}

	/**
	 * Create a {@link PolyglotPropertiesBuilder} instance containing the
	 * elements of the provided {@link Map} instance using the default path
	 * delimiter "/" ({@link Delimiter#PATH}) for the path declarations
	 *
	 * @param aPropertiesBuilder the properties to be added.
	 */
	public PolyglotPropertiesBuilder( Map<?, ?> aPropertiesBuilder ) {
		this( new PolyglotPropertiesBuilderFactory(), aPropertiesBuilder );
	}

	/**
	 * Loads the {@link PolyglotPropertiesBuilder} from the given {@link File}.
	 *
	 * @param aFile The {@link File} from which to load the properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( File aFile ) throws IOException, ParseException {
		this( new PolyglotPropertiesBuilderFactory(), aFile );
	}

	/**
	 * Loads the {@link PolyglotPropertiesBuilder} from the given {@link File}.
	 *
	 * @param aFile The {@link File} from which to load the properties.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( File aFile, char... aDelimiters ) throws IOException, ParseException {
		this( new PolyglotPropertiesBuilderFactory(), aFile, aDelimiters );
	}

	/**
	 * Loads or seeks the {@link PolyglotPropertiesBuilder} from the given
	 * {@link File}. A provided {@link ConfigLocator} describes the locations to
	 * additional crawl for the desired file.
	 * 
	 * @param aFile The {@link File} from which to load the properties.
	 * @param aConfigLocator The {@link ConfigLocator} describes the locations
	 *        to additional crawl for the desired file.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( File aFile, ConfigLocator aConfigLocator ) throws IOException, ParseException {
		this( new PolyglotPropertiesBuilderFactory(), aFile, aConfigLocator );
	}

	/**
	 * Loads or seeks the {@link PolyglotPropertiesBuilder} from the given
	 * {@link File}. A provided {@link ConfigLocator} describes the locations to
	 * additional crawl for the desired file.
	 * 
	 * @param aFile The {@link File} from which to load the properties.
	 * @param aConfigLocator The {@link ConfigLocator} describes the locations
	 *        to additional crawl for the desired file.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( File aFile, ConfigLocator aConfigLocator, char... aDelimiters ) throws IOException, ParseException {
		this( new PolyglotPropertiesBuilderFactory(), aFile, aConfigLocator, aDelimiters );
	}

	/**
	 * Loads the {@link PolyglotPropertiesBuilder} from the given file's path.
	 *
	 * @param aFilePath The path to the file from which to load the properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( String aFilePath ) throws IOException, ParseException {
		this( new PolyglotPropertiesBuilderFactory(), aFilePath );
	}

	/**
	 * Loads the {@link PolyglotPropertiesBuilder} from the given file's path.
	 *
	 * @param aFilePath The path to the file from which to load the properties.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( String aFilePath, char... aDelimiters ) throws IOException, ParseException {
		this( new PolyglotPropertiesBuilderFactory(), aFilePath, aDelimiters );
	}

	/**
	 * Loads the {@link PolyglotPropertiesBuilder} from the given file's path.
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( Class<?> aResourceClass, String aFilePath ) throws IOException, ParseException {
		this( new PolyglotPropertiesBuilderFactory(), aResourceClass, aFilePath );
	}

	/**
	 * Loads the {@link PolyglotPropertiesBuilder} from the given file's path.
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( Class<?> aResourceClass, String aFilePath, char... aDelimiters ) throws IOException, ParseException {
		this( new PolyglotPropertiesBuilderFactory(), aResourceClass, aFilePath, aDelimiters );
	}

	/**
	 * Loads the {@link PolyglotPropertiesBuilder} from the given file's path. A
	 * provided {@link ConfigLocator} describes the locations to additional
	 * crawl for the desired file.
	 * 
	 * @param aFilePath The path to the file from which to load the properties.
	 * @param aConfigLocator The {@link ConfigLocator} describes the locations
	 *        to additional crawl for the desired file.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( String aFilePath, ConfigLocator aConfigLocator ) throws IOException, ParseException {
		this( new PolyglotPropertiesBuilderFactory(), aFilePath, aConfigLocator );
	}

	/**
	 * Loads the {@link PolyglotPropertiesBuilder} from the given file's path. A
	 * provided {@link ConfigLocator} describes the locations to additional
	 * crawl for the desired file.
	 * 
	 * @param aFilePath The path to the file from which to load the properties.
	 * @param aConfigLocator The {@link ConfigLocator} describes the locations
	 *        to additional crawl for the desired file.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( String aFilePath, ConfigLocator aConfigLocator, char... aDelimiters ) throws IOException, ParseException {
		this( new PolyglotPropertiesBuilderFactory(), aFilePath, aConfigLocator, aDelimiters );
	}

	/**
	 * Loads the {@link PolyglotPropertiesBuilder} from the given file's path. A
	 * provided {@link ConfigLocator} describes the locations to additional
	 * crawl for the desired file. Finally (if nothing else succeeds) the
	 * properties are loaded by the provided class's class loader which takes
	 * care of loading the properties (in case the file path is a relative path,
	 * also the absolute path with a prefixed path delimiter "/" is probed).
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * @param aConfigLocator The {@link ConfigLocator} describes the locations
	 *        to additional crawl for the desired file.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( Class<?> aResourceClass, String aFilePath, ConfigLocator aConfigLocator ) throws IOException, ParseException {
		this( new PolyglotPropertiesBuilderFactory(), aFilePath, aConfigLocator );
	}

	/**
	 * Loads the {@link PolyglotPropertiesBuilder} from the given file's path. A
	 * provided {@link ConfigLocator} describes the locations to additional
	 * crawl for the desired file. Finally (if nothing else succeeds) the
	 * properties are loaded by the provided class's class loader which takes
	 * care of loading the properties (in case the file path is a relative path,
	 * also the absolute path with a prefixed path delimiter "/" is probed).
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * @param aConfigLocator The {@link ConfigLocator} describes the locations
	 *        to additional crawl for the desired file.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( Class<?> aResourceClass, String aFilePath, ConfigLocator aConfigLocator, char... aDelimiters ) throws IOException, ParseException {
		this( new PolyglotPropertiesBuilderFactory(), aResourceClass, aFilePath, aConfigLocator, aDelimiters );
	}

	/**
	 * Loads the {@link PolyglotPropertiesBuilder} from the given {@link URL}.
	 *
	 * @param aUrl The {@link URL} from which to read the properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( URL aUrl ) throws IOException, ParseException {
		this( new PolyglotPropertiesBuilderFactory(), aUrl );
	}

	/**
	 * Loads the {@link PolyglotPropertiesBuilder} from the given {@link URL}.
	 *
	 * @param aUrl The {@link URL} from which to read the properties.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( URL aUrl, char... aDelimiters ) throws IOException, ParseException {
		this( new PolyglotPropertiesBuilderFactory(), aUrl, aDelimiters );
	}

	/**
	 * Reads the {@link PolyglotPropertiesBuilder} from the given
	 * {@link InputStream}.
	 *
	 * @param aInputStream The {@link InputStream} from which to read the
	 *        properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( InputStream aInputStream ) throws IOException, ParseException {
		this( new PolyglotPropertiesBuilderFactory(), aInputStream );
	}

	/**
	 * Reads the {@link PolyglotPropertiesBuilder} from the given
	 * {@link InputStream}.
	 *
	 * @param aInputStream The {@link InputStream} from which to read the
	 *        properties.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( InputStream aInputStream, char... aDelimiters ) throws IOException, ParseException {
		this( new PolyglotPropertiesBuilderFactory(), aInputStream, aDelimiters );
	}

	// -------------------------------------------------------------------------

	/**
	 * Create a {@link PolyglotPropertiesBuilder} instance containing the
	 * elements as of {@link MutablePathMap#insert(Object)} using the default
	 * path delimiter "/" ({@link Delimiter#PATH}) for the path declarations:
	 * "Inspects the given object and adds all elements found in the given
	 * object. Elements of type {@link Map}, {@link Collection} and arrays are
	 * identified and handled as of their type: The path for each value in a
	 * {@link Map} is appended with its according key. The path for each value
	 * in a {@link Collection} or array is appended with its according index of
	 * occurrence (in case of a {@link List} or an array, its actual index). In
	 * case of reflection, the path for each member is appended with its
	 * according mamber's name. All elements (e.g. the members and values) are
	 * inspected recursively which results in the according paths of the
	 * terminating values."
	 *
	 * @param aPropertiesBuilderFactory The factory to use when constructing
	 *        {@link ResourcePropertiesBuilder} instances.
	 * @param aObj The object from which the elements are to be added.
	 */
	public PolyglotPropertiesBuilder( PolyglotPropertiesBuilderFactory aPropertiesBuilderFactory, Object aObj ) {
		super( aPropertiesBuilderFactory.toProperties( aObj ) );
	}

	/**
	 * Create a {@link PolyglotPropertiesBuilder} instance containing the
	 * elements of the provided {@link Properties} instance using the default
	 * path delimiter "/" ({@link Delimiter#PATH}) for the path declarations
	 *
	 * @param aProperties the properties to be added.
	 * @param aPropertiesBuilderFactory The factory to use when constructing
	 *        {@link ResourcePropertiesBuilder} instances.
	 */
	public PolyglotPropertiesBuilder( PolyglotPropertiesBuilderFactory aPropertiesBuilderFactory, Properties aProperties ) {
		super( aPropertiesBuilderFactory.toProperties( aProperties ) );
	}

	/**
	 * Create a {@link PolyglotPropertiesBuilder} instance containing the
	 * elements of the provided {@link PropertiesBuilder} instance using the
	 * default path delimiter "/" ({@link Delimiter#PATH}) for the path
	 * declarations
	 *
	 * @param aPropertiesBuilder the properties to be added.
	 * @param aPropertiesBuilderFactory The factory to use when constructing
	 *        {@link ResourcePropertiesBuilder} instances.
	 */
	public PolyglotPropertiesBuilder( PolyglotPropertiesBuilderFactory aPropertiesBuilderFactory, PropertiesBuilder aPropertiesBuilder ) {
		super( aPropertiesBuilderFactory.toProperties( aPropertiesBuilder ) );
	}

	/**
	 * Create a {@link PolyglotPropertiesBuilder} instance containing the
	 * elements of the provided {@link Map} instance using the default path
	 * delimiter "/" ({@link Delimiter#PATH}) for the path declarations
	 * 
	 * @param aPropertiesBuilderFactory The factory to use when constructing
	 *        {@link ResourcePropertiesBuilder} instances.
	 * @param aPropertiesBuilder the properties to be added.
	 */
	public PolyglotPropertiesBuilder( PolyglotPropertiesBuilderFactory aPropertiesBuilderFactory, Map<?, ?> aPropertiesBuilder ) {
		super( aPropertiesBuilderFactory.toProperties( aPropertiesBuilder ) );
	}

	/**
	 * Loads the {@link PolyglotPropertiesBuilder} from the given {@link File}.
	 *
	 * @param aFile The {@link File} from which to load the properties.
	 * @param aPropertiesBuilderFactory The factory to use when constructing
	 *        {@link ResourcePropertiesBuilder} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( PolyglotPropertiesBuilderFactory aPropertiesBuilderFactory, File aFile ) throws IOException, ParseException {
		super( aPropertiesBuilderFactory.toProperties( aFile ) );
	}

	/**
	 * Loads the {@link PolyglotPropertiesBuilder} from the given {@link File}.
	 *
	 * @param aFile The {@link File} from which to load the properties.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * @param aPropertiesBuilderFactory The factory to use when constructing
	 *        {@link ResourcePropertiesBuilder} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( PolyglotPropertiesBuilderFactory aPropertiesBuilderFactory, File aFile, char... aDelimiters ) throws IOException, ParseException {
		super( aPropertiesBuilderFactory.toProperties( aFile, aDelimiters ) );
	}

	/**
	 * Loads or seeks the {@link PolyglotPropertiesBuilder} from the given
	 * {@link File}. A provided {@link ConfigLocator} describes the locations to
	 * additional crawl for the desired file.
	 * 
	 * @param aFile The {@link File} from which to load the properties.
	 * @param aConfigLocator The {@link ConfigLocator} describes the locations
	 *        to additional crawl for the desired file.
	 * @param aPropertiesBuilderFactory The factory to use when constructing
	 *        {@link ResourcePropertiesBuilder} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( PolyglotPropertiesBuilderFactory aPropertiesBuilderFactory, File aFile, ConfigLocator aConfigLocator ) throws IOException, ParseException {
		super( aPropertiesBuilderFactory.toProperties( aFile, aConfigLocator ) );
	}

	/**
	 * Loads or seeks the {@link PolyglotPropertiesBuilder} from the given
	 * {@link File}. A provided {@link ConfigLocator} describes the locations to
	 * additional crawl for the desired file.
	 * 
	 * @param aFile The {@link File} from which to load the properties.
	 * @param aConfigLocator The {@link ConfigLocator} describes the locations
	 *        to additional crawl for the desired file.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * @param aPropertiesBuilderFactory The factory to use when constructing
	 *        {@link ResourcePropertiesBuilder} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( PolyglotPropertiesBuilderFactory aPropertiesBuilderFactory, File aFile, ConfigLocator aConfigLocator, char... aDelimiters ) throws IOException, ParseException {
		super( aPropertiesBuilderFactory.toProperties( aFile, aConfigLocator, aDelimiters ) );
	}

	/**
	 * Loads the {@link PolyglotPropertiesBuilder} from the given file's path.
	 *
	 * @param aFilePath The path to the file from which to load the properties.
	 * @param aPropertiesBuilderFactory The factory to use when constructing
	 *        {@link ResourcePropertiesBuilder} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( PolyglotPropertiesBuilderFactory aPropertiesBuilderFactory, String aFilePath ) throws IOException, ParseException {
		super( aPropertiesBuilderFactory.toProperties( aFilePath ) );
	}

	/**
	 * Loads the {@link PolyglotPropertiesBuilder} from the given file's path.
	 *
	 * @param aFilePath The path to the file from which to load the properties.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * @param aPropertiesBuilderFactory The factory to use when constructing
	 *        {@link ResourcePropertiesBuilder} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( PolyglotPropertiesBuilderFactory aPropertiesBuilderFactory, String aFilePath, char... aDelimiters ) throws IOException, ParseException {
		super( aPropertiesBuilderFactory.toProperties( aFilePath, aDelimiters ) );
	}

	/**
	 * Loads the {@link PolyglotPropertiesBuilder} from the given file's path.
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * @param aPropertiesBuilderFactory The factory to use when constructing
	 *        {@link ResourcePropertiesBuilder} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( PolyglotPropertiesBuilderFactory aPropertiesBuilderFactory, Class<?> aResourceClass, String aFilePath ) throws IOException, ParseException {
		super( aPropertiesBuilderFactory.toProperties( aResourceClass, aFilePath ) );
	}

	/**
	 * Loads the {@link PolyglotPropertiesBuilder} from the given file's path.
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * @param aPropertiesBuilderFactory The factory to use when constructing
	 *        {@link ResourcePropertiesBuilder} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( PolyglotPropertiesBuilderFactory aPropertiesBuilderFactory, Class<?> aResourceClass, String aFilePath, char... aDelimiters ) throws IOException, ParseException {
		super( aPropertiesBuilderFactory.toProperties( aResourceClass, aFilePath, aDelimiters ) );
	}

	/**
	 * Loads the {@link PolyglotPropertiesBuilder} from the given file's path. A
	 * provided {@link ConfigLocator} describes the locations to additional
	 * crawl for the desired file.
	 * 
	 * @param aFilePath The path to the file from which to load the properties.
	 * @param aConfigLocator The {@link ConfigLocator} describes the locations
	 *        to additional crawl for the desired file.
	 * @param aPropertiesBuilderFactory The factory to use when constructing
	 *        {@link ResourcePropertiesBuilder} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( PolyglotPropertiesBuilderFactory aPropertiesBuilderFactory, String aFilePath, ConfigLocator aConfigLocator ) throws IOException, ParseException {
		super( aPropertiesBuilderFactory.toProperties( aFilePath, aConfigLocator ) );
	}

	/**
	 * Loads the {@link PolyglotPropertiesBuilder} from the given file's path. A
	 * provided {@link ConfigLocator} describes the locations to additional
	 * crawl for the desired file.
	 * 
	 * @param aFilePath The path to the file from which to load the properties.
	 * @param aConfigLocator The {@link ConfigLocator} describes the locations
	 *        to additional crawl for the desired file.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * @param aPropertiesBuilderFactory The factory to use when constructing
	 *        {@link ResourcePropertiesBuilder} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( PolyglotPropertiesBuilderFactory aPropertiesBuilderFactory, String aFilePath, ConfigLocator aConfigLocator, char... aDelimiters ) throws IOException, ParseException {
		super( aPropertiesBuilderFactory.toProperties( aFilePath, aConfigLocator, aDelimiters ) );
	}

	/**
	 * Loads the {@link PolyglotPropertiesBuilder} from the given file's path. A
	 * provided {@link ConfigLocator} describes the locations to additional
	 * crawl for the desired file. Finally (if nothing else succeeds) the
	 * properties are loaded by the provided class's class loader which takes
	 * care of loading the properties (in case the file path is a relative path,
	 * also the absolute path with a prefixed path delimiter "/" is probed).
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * @param aConfigLocator The {@link ConfigLocator} describes the locations
	 *        to additional crawl for the desired file.
	 * @param aPropertiesBuilderFactory The factory to use when constructing
	 *        {@link ResourcePropertiesBuilder} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( PolyglotPropertiesBuilderFactory aPropertiesBuilderFactory, Class<?> aResourceClass, String aFilePath, ConfigLocator aConfigLocator ) throws IOException, ParseException {
		super( aPropertiesBuilderFactory.toProperties( aResourceClass, aFilePath, aConfigLocator ) );
	}

	/**
	 * Loads the {@link PolyglotPropertiesBuilder} from the given file's path. A
	 * provided {@link ConfigLocator} describes the locations to additional
	 * crawl for the desired file. Finally (if nothing else succeeds) the
	 * properties are loaded by the provided class's class loader which takes
	 * care of loading the properties (in case the file path is a relative path,
	 * also the absolute path with a prefixed path delimiter "/" is probed).
	 *
	 * @param aResourceClass The class which's class loader is to take care of
	 *        loading the properties (from inside a JAR).
	 * @param aFilePath The file path of the class's resources from which to
	 *        load the properties.
	 * @param aConfigLocator The {@link ConfigLocator} describes the locations
	 *        to additional crawl for the desired file.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * @param aPropertiesBuilderFactory The factory to use when constructing
	 *        {@link ResourcePropertiesBuilder} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( PolyglotPropertiesBuilderFactory aPropertiesBuilderFactory, Class<?> aResourceClass, String aFilePath, ConfigLocator aConfigLocator, char... aDelimiters ) throws IOException, ParseException {
		super( aPropertiesBuilderFactory.toProperties( aResourceClass, aFilePath, aConfigLocator, aDelimiters ) );
	}

	/**
	 * Loads the {@link PolyglotPropertiesBuilder} from the given {@link URL}.
	 *
	 * @param aUrl The {@link URL} from which to read the properties.
	 * @param aPropertiesBuilderFactory The factory to use when constructing
	 *        {@link ResourcePropertiesBuilder} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( PolyglotPropertiesBuilderFactory aPropertiesBuilderFactory, URL aUrl ) throws IOException, ParseException {
		super( aPropertiesBuilderFactory.toProperties( aUrl ) );
	}

	/**
	 * Loads the {@link PolyglotPropertiesBuilder} from the given {@link URL}.
	 *
	 * @param aUrl The {@link URL} from which to read the properties.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * @param aPropertiesBuilderFactory The factory to use when constructing
	 *        {@link ResourcePropertiesBuilder} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( PolyglotPropertiesBuilderFactory aPropertiesBuilderFactory, URL aUrl, char... aDelimiters ) throws IOException, ParseException {
		super( aPropertiesBuilderFactory.toProperties( aUrl, aDelimiters ) );
	}

	/**
	 * Reads the {@link PolyglotPropertiesBuilder} from the given
	 * {@link InputStream}.
	 *
	 * @param aInputStream The {@link InputStream} from which to read the
	 *        properties.
	 * @param aPropertiesBuilderFactory The factory to use when constructing
	 *        {@link ResourcePropertiesBuilder} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( PolyglotPropertiesBuilderFactory aPropertiesBuilderFactory, InputStream aInputStream ) throws IOException, ParseException {
		super( aPropertiesBuilderFactory.toProperties( aInputStream ) );
	}

	/**
	 * Reads the {@link PolyglotPropertiesBuilder} from the given
	 * {@link InputStream}.
	 *
	 * @param aInputStream The {@link InputStream} from which to read the
	 *        properties.
	 * @param aDelimiters The path delimiters to be used when parsing the
	 *        source's properties.
	 * @param aPropertiesBuilderFactory The factory to use when constructing
	 *        {@link ResourcePropertiesBuilder} instances.
	 * 
	 * @throws IOException thrown in case accessing or processing the properties
	 *         file failed.
	 * @throws ParseException Signals that an error has been reached
	 *         unexpectedly while parsing the data to be loaded.
	 */
	public PolyglotPropertiesBuilder( PolyglotPropertiesBuilderFactory aPropertiesBuilderFactory, InputStream aInputStream, char... aDelimiters ) throws IOException, ParseException {
		super( aPropertiesBuilderFactory.toProperties( aInputStream, aDelimiters ) );
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * The {@link PolyglotPropertiesBuilderFactory} is a meta factory using a
	 * collection of {@link ResourcePropertiesBuilderFactory} instances to
	 * deliver {@link ResourcePropertiesBuilder} instances. In case a properties
	 * file for a filename was not found, then the according factories filename
	 * extension
	 * ({@link ResourcePropertiesBuilderFactory#getFilenameSuffixes()}) is
	 * append to the filename and probing is repeated. Any factory method such
	 * as {@link #toProperties(Map)}, {@link #toProperties(Object)},
	 * {@link #toProperties(PropertiesBuilder)} or
	 * {@link #toProperties(PropertiesBuilder)} will return
	 * {@link ResourcePropertiesBuilder} created by the first added
	 * {@link ResourcePropertiesBuilderFactory} instance.
	 */
	public static class PolyglotPropertiesBuilderFactory implements ResourcePropertiesBuilderFactory {

		// /////////////////////////////////////////////////////////////////////
		// VARIABLES:
		// /////////////////////////////////////////////////////////////////////

		private List<ResourcePropertiesBuilderFactory> _factories = new ArrayList<>();

		// /////////////////////////////////////////////////////////////////////
		// CONSTRUCTORS:
		// /////////////////////////////////////////////////////////////////////

		/**
		 * Initializes the {@link PolyglotPropertiesBuilderFactory} with a
		 * predefined set of {@link ResourcePropertiesBuilderFactory} instances.
		 */
		public PolyglotPropertiesBuilderFactory() {
			_factories.add( new TomlPropertiesBuilderFactory() );
			_factories.add( new YamlPropertiesBuilderFactory() );
			_factories.add( new XmlPropertiesBuilderFactory() );
			_factories.add( new JsonPropertiesBuilderFactory() );
			// Last as them Java properties eat all notations |-->
			_factories.add( new JavaPropertiesBuilderFactory() );
			// Last as them Java properties eat all notations <--|
		}

		/**
		 * Initializes the {@link PolyglotPropertiesBuilderFactory} with the
		 * given {@link ResourcePropertiesBuilderFactory} instances. Them
		 * factories will be queried in the order being provided.
		 * 
		 * @param aFactories The factories to be added.
		 */
		public PolyglotPropertiesBuilderFactory( ResourcePropertiesBuilderFactory... aFactories ) {
			for ( ResourcePropertiesBuilderFactory eFactory : aFactories ) _factories.add( eFactory );
		}

		// /////////////////////////////////////////////////////////////////////
		// METHODS:
		// /////////////////////////////////////////////////////////////////////

		/**
		 * Returns the filename extensions of the ResourcePropertiesFactory in
		 * the given order. {@inheritDoc}
		 */
		@Override
		public String[] getFilenameSuffixes() {
			List<String> theExtensions = new ArrayList<>();
			String[] eExtensions;
			for ( int i = 0; i < _factories.size(); i++ ) {
				eExtensions = _factories.get( i ).getFilenameSuffixes();
				if ( eExtensions != null ) {
					for ( int j = 0; j < eExtensions.length; j++ ) {
						theExtensions.add( eExtensions[j] );
					}
				}
			}
			return theExtensions.toArray( new String[theExtensions.size()] );
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public ResourcePropertiesBuilder toProperties( Class<?> aResourceClass, String aFilePath, ConfigLocator aConfigLocator, char... aDelimiters ) throws IOException, ParseException {
			Exception theCause = null;

			// Keep the precedence |-->
			try {
				return toPropertiesFromFilePath( aFilePath, aConfigLocator, aDelimiters );
			}
			catch ( Exception e ) {
				if ( theCause == null ) theCause = e;
			}
			// Keep the precedence <--|

			for ( ResourcePropertiesBuilderFactory eFactory : _factories ) {
				try {
					return eFactory.toProperties( aResourceClass, aFilePath, aConfigLocator, aDelimiters );
				}
				catch ( ParseException e ) {
					if ( eFactory.hasFilenameSuffix( aFilePath ) ) throw new ParseException( "Parsing of file <" + aFilePath + "> failed: " + e.getMessage(), e.getErrorOffset() );
					if ( theCause == null ) theCause = e;
				}
				catch ( Exception e ) {
					if ( theCause == null ) theCause = e;
				}
			}

			// Test with the factorie's filename extension |-->
			String[] eExtensions;
			String eFilePath;
			for ( ResourcePropertiesBuilderFactory eFactory : _factories ) {
				eExtensions = eFactory.getFilenameSuffixes();
				if ( eExtensions != null ) {
					for ( String eExtension : eExtensions ) {
						eFilePath = aFilePath + eExtension;
						try {
							return eFactory.toProperties( eFilePath, aConfigLocator, aDelimiters );
						}
						catch ( ParseException e ) {
							if ( eFactory.hasFilenameSuffix( eFilePath ) ) throw new ParseException( "Parsing of file <" + eFilePath + "> failed: " + e.getMessage(), e.getErrorOffset() );
							if ( theCause == null ) theCause = e;
						}
						catch ( Exception e ) {
							if ( theCause == null ) theCause = e;
						}
					}
				}
			}
			// Test with the factorie's filename extension <--|
			if ( theCause instanceof IOException ) throw (IOException) theCause;
			if ( theCause instanceof ParseException ) throw (ParseException) theCause;
			throw new IOException( "No configuration file with name scheme <" + aFilePath + ".*> found, where a suffix (if applicable) might by one of the following: " + Arrays.toString( getFilenameSuffixes() ) );
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public ResourcePropertiesBuilder toProperties( File aFile, ConfigLocator aConfigLocator, char... aDelimiters ) throws IOException, ParseException {
			Exception theCause = null;

			// Keep the precedence |-->
			try {
				return toPropertiesFromFilePath( aFile.getPath(), aConfigLocator, aDelimiters );
			}
			catch ( Exception e ) {
				if ( theCause == null ) theCause = e;
			}
			// Keep the precedence <--|

			for ( ResourcePropertiesBuilderFactory eFactory : _factories ) {
				try {
					return eFactory.toProperties( aFile, aConfigLocator, aDelimiters );
				}
				catch ( Exception e ) {
					if ( theCause == null ) theCause = e;
				}
			}
			// Test with the factorie's filename extension |-->
			File eFile;
			String[] eExtensions;
			for ( ResourcePropertiesBuilderFactory eFactory : _factories ) {
				eExtensions = eFactory.getFilenameSuffixes();
				if ( eExtensions != null ) {
					for ( String eExtension : eExtensions ) {
						eFile = new File( aFile.getAbsolutePath() + eExtension );
						try {
							return eFactory.toProperties( eFile, aConfigLocator, aDelimiters );
						}
						catch ( Exception e ) {
							if ( theCause == null ) theCause = e;
						}
					}
				}
			}
			// Test with the factorie's filename extension <--|
			if ( theCause instanceof IOException ) throw (IOException) theCause;
			if ( theCause instanceof ParseException ) throw (ParseException) theCause;
			throw new IOException( "No configuration file with (base) file <" + aFile.getPath() + "> found!" );
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public ResourcePropertiesBuilder toProperties( InputStream aInputStream, char... aDelimiters ) throws IOException, ParseException {
			Exception theCause = null;
			byte[] theBytes = toByteArray( aInputStream );
			for ( ResourcePropertiesBuilderFactory eFactory : _factories ) {
				try {
					return eFactory.toProperties( new ByteArrayInputStream( theBytes ), aDelimiters );
				}
				catch ( Exception e ) {
					if ( theCause == null ) theCause = e;
				}
			}
			if ( theCause instanceof IOException ) throw (IOException) theCause;
			if ( theCause instanceof ParseException ) throw (ParseException) theCause;
			throw new IOException( "No configuration file for the given InputStream found!" );
		}

		/**
		 * Will return {@link ResourcePropertiesBuilder} created by the first
		 * added {@link ResourcePropertiesBuilderFactory} instance.
		 * {@inheritDoc}
		 */
		@Override
		public ResourcePropertiesBuilder toProperties( Map<?, ?> aPropertiesBuilder ) {
			return _factories.get( 0 ).toProperties( aPropertiesBuilder );
		}

		/**
		 * Will return {@link ResourcePropertiesBuilder} created by the first
		 * added {@link ResourcePropertiesBuilderFactory} instance.
		 * {@inheritDoc}
		 */
		@Override
		public ResourcePropertiesBuilder toProperties( Object aObj ) {
			return _factories.get( 0 ).toProperties( aObj );
		}

		/**
		 * Will return {@link ResourcePropertiesBuilder} created by the first
		 * added {@link ResourcePropertiesBuilderFactory} instance.
		 * {@inheritDoc}
		 */
		@Override
		public ResourcePropertiesBuilder toProperties( Properties aProperties ) {
			return _factories.get( 0 ).toProperties( aProperties );
		}

		/**
		 * Will return {@link ResourcePropertiesBuilder} created by the first
		 * added {@link ResourcePropertiesBuilderFactory} instance.
		 * {@inheritDoc}
		 */
		@Override
		public ResourcePropertiesBuilder toProperties( PropertiesBuilder aPropertiesBuilder ) {
			return _factories.get( 0 ).toProperties( aPropertiesBuilder );
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public ResourcePropertiesBuilder toProperties( String aFilePath, ConfigLocator aConfigLocator, char... aDelimiters ) throws IOException, ParseException {
			return toProperties( null, aFilePath, aConfigLocator, aDelimiters );
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public ResourcePropertiesBuilder toProperties( URL aUrl, char... aDelimiters ) throws IOException, ParseException {
			Exception theCause = null;

			// Try file extension's factory first |-->
			try {
				ResourcePropertiesBuilderFactory theFactory = fromFilenameExtension( aUrl.toExternalForm() );
				if ( theFactory != null ) return theFactory.toProperties( aUrl, aDelimiters );
			}
			catch ( Exception e ) {
				if ( theCause == null ) theCause = e;
			}
			// Try file extension's factory first <--|

			for ( ResourcePropertiesBuilderFactory eFactory : _factories ) {
				try {
					return eFactory.toProperties( aUrl, aDelimiters );
				}
				catch ( Exception e ) {
					if ( theCause == null ) theCause = e;
				}
			}
			// Test with the factorie's filename extension |-->
			URL eUrl;
			String[] eExtensions;
			for ( ResourcePropertiesBuilderFactory eFactory : _factories ) {
				eExtensions = eFactory.getFilenameSuffixes();
				if ( eExtensions != null ) {
					for ( String eExtension : eExtensions ) {
						eUrl = new URL( aUrl.toExternalForm() + eExtension );
						try {
							return eFactory.toProperties( eUrl, aDelimiters );
						}
						catch ( Exception e ) {
							if ( theCause == null ) theCause = e;
						}
					}
				}
			}
			// Test with the factorie's filename extension <--|
			if ( theCause instanceof IOException ) throw (IOException) theCause;
			if ( theCause instanceof ParseException ) throw (ParseException) theCause;
			throw new IOException( "No configuration file with (base) URL <" + aUrl.toExternalForm() + "> found!" );
		}

		// /////////////////////////////////////////////////////////////////////
		// HELPER:
		// /////////////////////////////////////////////////////////////////////

		private ResourcePropertiesBuilderFactory fromFilenameExtension( String aFilePath ) {
			String[] eExtensions;
			for ( ResourcePropertiesBuilderFactory eFactory : _factories ) {
				eExtensions = eFactory.getFilenameSuffixes();
				if ( eExtensions != null ) {
					for ( String eExtension : eExtensions ) {
						if ( aFilePath.toLowerCase().endsWith( eExtension.toLowerCase() ) ) {
							return eFactory;
						}
					}
				}
			}
			return null;
		}

		private ResourcePropertiesBuilder toPropertiesFromFilePath( String aFilePath, ConfigLocator aConfigLocator, char... aDelimiters ) throws IOException, ParseException {
			Exception theCause = null;
			File theFile = new File( aFilePath );
			File eFile;
			if ( theFile.exists() ) {
				ResourcePropertiesBuilderFactory theFactory = fromFilenameExtension( theFile.getAbsolutePath() );
				try {
					if ( theFactory != null ) {
						return theFactory.toProperties( theFile, ConfigLocator.ABSOLUTE, aDelimiters );
					}
				}
				catch ( ParseException e ) {
					if ( theFactory.hasFilenameSuffix( theFile ) ) throw new ParseException( "Parsing of file <" + theFile.getAbsolutePath() + "> failed: " + e.getMessage(), e.getErrorOffset() );
					if ( theCause == null ) theCause = e;
				}
				catch ( Exception e ) {
					if ( theCause == null ) theCause = e;
				}
			}
			else {
				for ( ResourcePropertiesBuilderFactory eFactory : _factories ) {
					for ( String eExtension : eFactory.getFilenameSuffixes() ) {
						eFile = new File( aFilePath + eExtension );
						if ( eFile.exists() ) {
							try {
								return eFactory.toProperties( eFile, ConfigLocator.ABSOLUTE, aDelimiters );
							}
							catch ( ParseException e ) {
								if ( eFactory.hasFilenameSuffix( eFile ) ) throw new ParseException( "Parsing of file <" + eFile.getAbsolutePath() + "> failed: " + e.getMessage(), e.getErrorOffset() );
								if ( theCause == null ) theCause = e;
							}
							catch ( Exception e ) {
								if ( theCause == null ) theCause = e;
							}
						}
					}
				}
				if ( aConfigLocator != null && aConfigLocator != ConfigLocator.ABSOLUTE ) {
					File[] theFolders = aConfigLocator.getFolders();
					for ( File eDir : theFolders ) {
						for ( ResourcePropertiesBuilderFactory eFactory : _factories ) {
							for ( String eExtension : eFactory.getFilenameSuffixes() ) {
								eFile = new File( eDir, aFilePath + eExtension );
								if ( eFile.exists() ) {
									try {
										return eFactory.toProperties( eFile, ConfigLocator.ABSOLUTE, aDelimiters );
									}
									catch ( ParseException e ) {
										if ( eFactory.hasFilenameSuffix( eFile ) ) throw new ParseException( "Parsing of file <" + eFile.getAbsolutePath() + "> failed: " + e.getMessage(), e.getErrorOffset() );
										if ( theCause == null ) theCause = e;
									}
									catch ( Exception e ) {
										if ( theCause == null ) theCause = e;
									}
								}
							}
						}
					}
				}
			}
			if ( theCause instanceof IOException ) throw (IOException) theCause;
			if ( theCause instanceof ParseException ) throw (ParseException) theCause;
			throw new IOException( "No configuration file with name (scheme) <" + aFilePath + "> found, where a suffix (if applicable) might by one of the following: " + Arrays.toString( getFilenameSuffixes() ) );
		}

		// /////////////////////////////////////////////////////////////////////
		// HOOKS:
		// /////////////////////////////////////////////////////////////////////

		/**
		 * Converts a byte array from an {@link InputStream}.
		 * 
		 * @param aInputStream The {@link InputStream} to be converted.
		 * 
		 * @return The byte array from the {@link InputStream}.
		 * 
		 * @throws IOException Thrown in case there were problems reading the
		 *         {@link InputStream}.
		 */
		public static byte[] toByteArray( InputStream aInputStream ) throws IOException {
			ByteArrayOutputStream theBuffer = new ByteArrayOutputStream();
			int eRead;
			byte[] theData = new byte[16384];
			while ( (eRead = aInputStream.read( theData, 0, theData.length )) != -1 ) {
				theBuffer.write( theData, 0, eRead );
			}
			theBuffer.flush();
			return theBuffer.toByteArray();
		}
	}
}
