// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.properties;

/**
 * Provides an accessor for a {@link Properties} property.
 */
public interface PropertiesAccessor {

	/**
	 * Retrieves the properties from the {@link Properties} property.
	 * 
	 * @return The properties stored by the {@link Properties} property.
	 */
	Properties getProperties();

	/**
	 * Provides a builder method for a {@link Properties} property returning the
	 * builder for applying multiple build operations.
	 * 
	 * @param <B> The builder to return in order to be able to apply multiple
	 *        build operations.
	 */
	public interface PropertiesBuilder<B extends PropertiesBuilder<B>> {

		/**
		 * Sets the properties for the {@link Properties} property.
		 * 
		 * @param aProperties The properties to be stored by the
		 *        {@link Properties} property.
		 * 
		 * @return The builder for applying multiple build operations.
		 */
		B withProperties( Properties aProperties );
	}

	/**
	 * Provides a mutator for a {@link Properties} property.
	 */
	public interface PropertiesMutator {

		/**
		 * Sets the properties for the {@link Properties} property.
		 * 
		 * @param aProperties The properties to be stored by the
		 *        {@link Properties} property.
		 */
		void setProperties( Properties aProperties );
	}

	/**
	 * Provides a {@link Properties} property.
	 */
	public interface PropertiesProperty extends PropertiesAccessor, PropertiesMutator {

		/**
		 * This method stores and passes through the given argument, which is
		 * very useful for builder APIs: Sets the given {@link Properties}
		 * (setter) as of {@link #setProperties(Properties)} and returns the
		 * very same value (getter).
		 * 
		 * @param aProperties The {@link Properties} to set (via
		 *        {@link #setProperties(Properties)}).
		 * 
		 * @return Returns the value passed for it to be used in conclusive
		 *         processing steps.
		 */
		default Properties letProperties( Properties aProperties ) {
			setProperties( aProperties );
			return aProperties;
		}
	}
}
