// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.remoting;

import java.util.Iterator;

import org.refcodes.component.OpenException;
import org.refcodes.exception.VetoException;

/**
 * Remote control providing subjects to be operated on by {@link RemoteClient}
 * instances.
 */
public interface RemoteServer extends Remote {

	/**
	 * Returns true if the provided subject is contained inside this
	 * {@link RemoteServer}.
	 * 
	 * @param aSubject The subject to be tested if it is contained inside the
	 *        {@link RemoteServer}.
	 * 
	 * @return True if the given subject is contained inside the
	 *         {@link RemoteServer}.
	 */
	boolean hasSubject( Object aSubject );

	/**
	 * Returns an (immutable) iterator containing all the proxy objects
	 * previously being published. Use the {@link #signOffSubject(Object)}
	 * method in order to remove a published subject.
	 * 
	 * @return An iterator containing the published proxy objects.
	 */
	Iterator<Object> subjects();

	/**
	 * Publishes an object to any {@link RemoteClient} connected to the
	 * {@link RemoteServer}.
	 * 
	 * @param aSubject A subject being published for inter-process communication
	 *        such as remote procedure calls or remote method invocations.
	 * 
	 * @return True is returned if the subject could be published, else false is
	 *         returned
	 * 
	 * @throws OpenException Thrown in case opening or accessing an open line
	 *         (connection, junction, link) caused problems.
	 */
	boolean publishSubject( Object aSubject ) throws OpenException;

	/**
	 * Tries to sign off the (previously published) subject, this can be vetoed
	 * in case the subject is still in use by a {@link RemoteClient}.
	 *
	 * @param aSubject Description is currently not available!
	 * @return True if the removal of the subject has been successful. If the
	 *         subject has not been found then false is returned. If a
	 *         {@link RemoteClient} threw a {@link VetoException} then the
	 *         sign-off is aborted.
	 * @throws VetoException the veto exception
	 * @throws OpenException Thrown in case opening or accessing an open line
	 *         (connection, junction, link) caused problems.
	 */
	boolean signOffSubject( Object aSubject ) throws VetoException, OpenException;

	/**
	 * Signs off the (previously published) subject, this be vetoed even in case
	 * the subject is still in use by a {@link RemoteClient}, but the veto will
	 * only delay the sign off by the given timeout.
	 * 
	 * @param aSubject The subject to be signed off.
	 * 
	 * @param aTimeoutInMs The timeout to be granted in case the sign-off has
	 *        been vetoed, nevertheless the subject will be signed off after the
	 *        timeout elapsed.
	 * 
	 * @return True if the removal of the subject has been successful. If the
	 *         subject has not been found then false is returned. If a
	 *         {@link RemoteClient} threw a {@link VetoException} then the
	 *         sign-off is aborted.
	 * 
	 * @throws OpenException Thrown in case opening or accessing an open line
	 *         (connection, junction, link) caused problems.
	 */
	boolean signOffSubject( Object aSubject, int aTimeoutInMs ) throws OpenException;
}