// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.remoting;

import java.lang.reflect.InvocationHandler;
import java.util.Iterator;

import org.refcodes.component.OpenException;
import org.refcodes.mixin.BusyAccessor;
import org.refcodes.mixin.Disposable.Disposedable;
import org.refcodes.mixin.Lockable;

/**
 * Remote control providing subjects to be operated on by a
 * {@link RemoteServer}.
 */
public interface RemoteClient extends Remote {

	/**
	 * Returns true if the provided proxy is contained inside the
	 * {@link RemoteClient}.
	 * 
	 * @param aProxy The proxy to be tested if it is contained inside the
	 *        {@link RemoteClient}.
	 * 
	 * @return True if the given proxy is contained inside the
	 *         {@link RemoteClient}.
	 */
	boolean hasProxy( Object aProxy );

	/**
	 * Returns a read-only iterator containing all the proxy objects previously
	 * being published. Use the sign-off methods in order to remove a published
	 * proxy object.
	 * 
	 * @return An iterator containing the published proxy objects.
	 */
	Iterator<Object> proxies();

	/**
	 * Returns true if there is at least one proxy of the given type.
	 *
	 * @param aType the type
	 * @return True in case there is at least one proxy of the given type.
	 */
	boolean hasProxy( Class<?> aType );

	/**
	 * Returns the proxy which can be cast to the given type. Note that this
	 * method may throw an {@link AmbiguousProxyException} even if
	 * {@link #hasProxy(Class)} returns true! The method
	 * {@link #hasProxy(Class)} returns true in case at least one proxy of the
	 * given type was found whereas this method only returns a proxy in case
	 * there is exactly one proxy of the required type is present.
	 *
	 * @param <T> the generic type
	 * @param aType the type
	 * @return The proxy for an instance of the given type.
	 * @throws AmbiguousProxyException Thrown in case a proxy for a given type
	 *         was requested but more than one proxies matched the requested
	 *         type.
	 * @throws NoSuchProxyException Thrown in case a proxy for a given type was
	 *         requested but not any proxy matched the requested type.
	 */
	<T> T getProxy( Class<T> aType ) throws AmbiguousProxyException, NoSuchProxyException;

	/**
	 * Signs off an instance previously published using the
	 * publishClassDescriptor() method.
	 * 
	 * @param proxy An object of type GenericInstanceDescriptor containing the
	 *        information needed to sign-off an instance.
	 * @return True is returned if the instance described by the proxy object
	 *         could be signed off, else false is returned
	 * 
	 * @throws OpenException Thrown in case opening or accessing an open line
	 *         (connection, junction, link) caused problems.
	 */
	boolean signOffProxy( Object proxy ) throws OpenException;

	// /////////////////////////////////////////////////////////////////////////
	// INNER INTERFACES:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * 
	 * The {@link ProxyControl} is used to manage a proxy being provided from a
	 * subject be the {@link RemoteServer}. This is an inner interface as it is
	 * just required by one implementation internally; in comparison other
	 * interface's implementations are required by at leat two different
	 * implementations (e.g. client and server).
	 */
	public interface ProxyControl extends InstanceId, InvocationHandler, Lockable, Disposedable, ProxyAccessor, BusyAccessor {

		/**
		 * Returns the class descriptor which has been used to create the Proxy
		 * object.
		 * 
		 * @return An object of type BlueprintClassDescriptor which has been
		 *         used to generate the Proxy object.
		 */
		InstanceDescriptor getClassDescriptor();

		/**
		 * Used to push a reply to a request.
		 * 
		 * @param aMethodReply An object of type BlueprintMethodReply used to
		 *        encapsulate a reply from a method call.
		 * 
		 */
		void pushMethodReply( Reply aMethodReply );

		/**
		 * Checks if is busy.
		 *
		 * @return true, if is busy
		 */
		@Override
		boolean isBusy();

		/**
		 * Dispose.
		 */
		@Override
		void dispose();
	}
}
