// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.runtime;

import org.refcodes.data.CharSet;
import org.refcodes.data.Delimiter;

/**
 * The Class PropertyBuilderImpl.
 */
public class PropertyBuilderImpl implements PropertyBuilder {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private String _key = null;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getKey() {
		return _key;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setKey( String aKey ) {
		_key = aKey;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public PropertyBuilder withKey( String aKey ) {
		setKey( aKey );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toPath( String aKey ) {
		if ( aKey == null || aKey.length() == 0 ) return aKey;
		String theKey = toNormalized( aKey, Delimiter.PATH.getChar() );
		return theKey != null ? Delimiter.PATH.getChar() + theKey.toLowerCase() : theKey;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toSystemProperty( String aKey ) {
		if ( aKey == null || aKey.length() == 0 ) return aKey;
		String theKey = toNormalized( aKey, Delimiter.NAMESPACE.getChar() );
		return theKey != null ? theKey.toLowerCase() : theKey;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toEnvironmentVariable( String aKey ) {
		if ( aKey == null || aKey.length() == 0 ) return aKey;
		String theKey = toNormalized( aKey, Delimiter.SNAKE_CASE.getChar() );
		return theKey != null ? theKey.toUpperCase() : theKey;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getSystemProperty( String aKey ) {
		String theKey = toSystemProperty( aKey );
		return System.getProperty( theKey != null ? theKey : aKey );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getEnvironmentVariable( String aKey ) {
		String theKey = toEnvironmentVariable( aKey );
		return System.getenv( theKey != null ? theKey : aKey );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	protected static String toNormalized( String aText, char aSeparator ) {

		if ( aText == null || aText.length() == 0 ) {
			return null;
		}

		String theRegexSeparator = CharSet.REGEX_SPECIAL_CHARS.hasChar( aSeparator ) ? "\\" + aSeparator : "" + aSeparator;
		String theRegexDoubleSeparator = theRegexSeparator + theRegexSeparator;
		String theDoubleSeparator = aSeparator + "" + aSeparator;
		String theSeparator = aSeparator + "";
		if ( aText == null || aText.length() == 0 ) return aText;
		StringBuffer theBuffer = new StringBuffer();
		while ( aText.length() > 0 && !Character.isAlphabetic( aText.charAt( 0 ) ) && !Character.isDigit( aText.charAt( 0 ) ) ) {
			aText = aText.substring( 1 );
		}
		while ( aText.length() > 0 && !Character.isAlphabetic( aText.charAt( aText.length() - 1 ) ) && !Character.isDigit( aText.charAt( aText.length() - 1 ) ) ) {
			aText = aText.substring( 0, aText.length() - 1 );
		}

		if ( aText == null || aText.length() == 0 ) {
			return null;
		}

		theBuffer.append( aText.charAt( 0 ) );
		for ( int i = 1; i < aText.length(); i++ ) {
			if ( Character.isLowerCase( aText.charAt( i - 1 ) ) && Character.isUpperCase( aText.charAt( i ) ) ) {
				theBuffer.append( aSeparator );
			}
			if ( !Character.isAlphabetic( aText.charAt( i ) ) ) {
				theBuffer.append( aSeparator );
			}
			else {
				theBuffer.append( aText.charAt( i ) );
			}
		}
		String theNormalited = theBuffer.toString();
		while ( theNormalited.indexOf( theDoubleSeparator ) != -1 ) {
			theNormalited = theNormalited.replaceAll( theRegexDoubleSeparator, theSeparator );
		}
		return theNormalited.toLowerCase();
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
