// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.runtime;

import org.refcodes.data.Delimiter;
import org.refcodes.mixin.KeyAccessor.KeyBuilder;
import org.refcodes.mixin.KeyAccessor.KeyProperty;
import org.refcodes.structure.PathMap;

/**
 * Converts a key to the format of a system property (camel-case) or an
 * environment variable (snake-case in upper-case).
 */
public interface PropertyBuilder extends KeyProperty<String>, KeyBuilder<String, PropertyBuilder> {

	/**
	 * Converts the key as of {@link #getKey()} to a path (as of
	 * {@link PathMap}). A path begins with a path delimiter "/" (as of
	 * {@link Delimiter#PATH} and with all non alphanumeric digits being
	 * replaced by the path delimiter. Any non alphanumeric sequence is
	 * truncated to a single path delimiter.
	 * 
	 * @return The path representation of the according key.
	 */
	default String toPath() {
		return toPath( getKey() );
	}

	/**
	 * Converts the provided key as of {@link #getKey()} to a path (as of
	 * {@link PathMap}). A path begins with a path delimiter "/" (as of
	 * {@link Delimiter#PATH} and with all non alphanumeric digits being
	 * replaced by the path delimiter. Any non alphanumeric sequence is
	 * truncated to a single path delimiter.
	 * 
	 * @param aKey The key to be converted.
	 * 
	 * @return The path representation of the according key.
	 */
	String toPath( String aKey );

	/**
	 * Converts the key as of {@link #getKey()} to a system property. A system
	 * property is the camel-case version of the according key.
	 * 
	 * @return The camel-case representation of the according key.
	 */
	default String toSystemProperty() {
		return toSystemProperty( getKey() );
	}

	/**
	 * Converts the key as of {@link #getKey()} to an environment variable. An
	 * environment variable is the upper-case snake-case version of the
	 * according key.
	 * 
	 * @return The upper-case snake-case representation of the according key.
	 */
	default String toEnvironmentVariable() {
		return toEnvironmentVariable( getKey() );
	}

	/**
	 * Converts the provided key to a system property. A system property is the
	 * camel-case version of the according key.
	 * 
	 * @param aKey The key to be converted.
	 * 
	 * @return The camel-case representation of the according key.
	 */
	String toSystemProperty( String aKey );

	/**
	 * Converts the provided key to an environment variable. An environment
	 * variable is the upper-case snake-case version of the according key.
	 * 
	 * @param aKey The key to be converted.
	 * 
	 * @return The upper-case snake-case representation of the according key.
	 */
	String toEnvironmentVariable( String aKey );

	/**
	 * Returns the according system's property after having converted the key as
	 * of {@link #getKey()} to a system property. A system property is the
	 * camel-case version of the according key.
	 * 
	 * @return The system's property of the camel-case representation of the
	 *         according key.
	 */
	default String getSystemProperty() {
		return getSystemProperty( getKey() );
	}

	/**
	 * Returns the according environment variable after having converted the key
	 * as of {@link #getKey()} to an environment variable. An environment
	 * variable is the upper-case snake-case version of the according key.
	 * 
	 * @return The environment variable of the upper-case snake-case
	 *         representation of the according key.
	 */
	default String getEnvironmentVariable() {
		return getEnvironmentVariable( getKey() );
	}

	/**
	 * Returns the according system's property after having converted the
	 * provided key to a system property. A system property is the camel-case
	 * version of the according key.
	 * 
	 * @param aKey The value of the key to be converted.
	 * 
	 * @return The system's property of the camel-case representation of the
	 *         according key.
	 */
	String getSystemProperty( String aKey );

	/**
	 * Returns the according environment variable after having converted the
	 * provided key to an environment variable. An environment variable is the
	 * upper-case snake-case version of the according key.
	 * 
	 * @param aKey The value of the key to be converted.
	 * 
	 * @return The environment variable of the upper-case snake-case
	 *         representation of the according key.
	 */
	String getEnvironmentVariable( String aKey );
}
