// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.runtime;

/**
 * Provides methods to set Request-Correlation behavior: When enabled via
 * {@link #setRequestCorrelation(boolean)} (or the like), then an existing
 * Request-Correlation-ID might be forwarded by setting it into the
 * {@link Correlation} singleton via {@link Correlation#REQUEST}'s
 * {@link Correlation#setId(String)}. When being disabled, then no
 * Request-Correlation handling will be done, any existing
 * {@link Correlation#REQUEST}'s {@link Correlation#getId()} may stay untouched!
 * 
 * @param <B> The builder to return in order to be able to apply multiple build
 *        operations.
 */
public interface RequestCorrelation<B extends RequestCorrelation<B>> {

	/**
	 * Enables or disables the automatic Request-Correlation mechanism.
	 * 
	 * @param hasRequestCorrelation When true then Request-Correlation is
	 *        enabled, else disabled.
	 */
	void setRequestCorrelation( boolean hasRequestCorrelation );

	/**
	 * Builder method for the {@link #setRequestCorrelation(boolean)} operation.
	 * 
	 * @param hasRequestCorrelation When true then Request-Correlation is
	 *        enabled, else disabled.
	 * 
	 * @return The builder for applying multiple build operations.
	 */
	B withRequestCorrelation( boolean hasRequestCorrelation );

	/**
	 * Enables the automatic Request-Correlation mechanism.
	 */
	default void enableRequestCorrelation() {
		setRequestCorrelation( true );
	}

	/**
	 * Builder method for the {@link #enableRequestCorrelation} operation.
	 * 
	 * @return The builder for applying multiple build operations.
	 */
	B withEnableRequestCorrelation();

	/**
	 * Disables the automatic Request-Correlation mechanism.
	 */
	default void disableRequestCorrelation() {
		setRequestCorrelation( false );
	}

	/**
	 * Builder method for the {@link #disableRequestCorrelation} operation.
	 * 
	 * @return The builder for applying multiple build operations.
	 */
	B withDisableRequestCorrelation();

	/**
	 * Determines whether Request-Correlation is being enabled or disabled.
	 * 
	 * @return True in case Request-Correlation is being enabled, else false.
	 */
	boolean hasRequestCorrelation();

}
