// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.runtime;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.Map;

import org.refcodes.data.ConsoleDimension;
import org.refcodes.data.EnvironmentProperty;
import org.refcodes.data.Literal;
import org.refcodes.data.SystemProperty;
import org.refcodes.numerical.NumericalUtility;

import jline.TerminalFactory;

/**
 * Utility for acquiring system information on the machine this process is
 * running in.
 */
public final class SystemUtility {

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Private constructor as of being utility class.
	 */
	private SystemUtility() {}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Determines the operating system as of
	 * {@link OperatingSystem#toOperatingSystem()} and in case a
	 * {@link OperatingSystem#WINDOWS} is being detected, then \r\n" (CRLF) is
	 * returned, else "\n" (LF) is returned.
	 * 
	 * Can be overridden with the {@link SystemProperty#CONSOLE_LINE_BREAK}
	 * (<code>java -Dconsole.lineBreak=...</code>) and the
	 * {@link EnvironmentProperty#CONSOLE_LINE_BREAK}
	 * ("<code>export CONSOLE_LINE_BREAK=...</code>).
	 * 
	 * @return The operating system's specific line break; on Windows it is
	 *         "\r\n" (CRLF) and on all other operating systems it is "\n" (LF).
	 */
	public static String getLineBreak() {
		String theLineBreak = SystemProperty.CONSOLE_LINE_BREAK.getValue();
		if ( theLineBreak == null || theLineBreak.length() == 0 ) {
			theLineBreak = EnvironmentProperty.CONSOLE_LINE_BREAK.getValue();
			if ( theLineBreak == null || theLineBreak.length() == 0 ) {
				theLineBreak = System.lineSeparator();
			}
		}
		if ( theLineBreak == null || theLineBreak.length() == 0 ) {
			theLineBreak = OperatingSystem.toOperatingSystem() == OperatingSystem.WINDOWS ? "\r\n" : "\n";
		}
		return theLineBreak;
	}

	/**
	 * Determines whether wee need an explicit line-break for the given width on
	 * the current operating system and used terminal. E.g. on "win.cmd" we must
	 * not use a line-break in case our line is a s long as the console's width.
	 * There are some more such cases which this method tries to take into
	 * consideration.
	 * 
	 * @param aRowWidth The row width you want to use when printing out to a
	 *        console.
	 * 
	 * @return True in case you should use a line-break.
	 */
	public static boolean isUseLineBreak( int aRowWidth ) {
		boolean isLineBreak = true;
		if ( (Shell.toShell() == Shell.WIN_CMD || (OperatingSystem.toOperatingSystem() == OperatingSystem.WINDOWS && Terminal.toTerminal() == Terminal.CYGWIN && getUname() != null)) && aRowWidth == SystemUtility.getTerminalWidth() ) {
			isLineBreak = false;
		}
		return isLineBreak;
	}

	/**
	 * Uses {@link #isUseLineBreak(int)} to retrieve the character sequence
	 * required to suffix to a line in order to get a line break without risking
	 * any empty lines as of automatic line wrapping. Automatic line wrapping
	 * can happen on Windows environment when the console width is reached. A
	 * line break would cause a superfluous empty line (ugly).
	 * 
	 * @param aRowWidth The row width you want to use when printing out to a
	 *        console.
	 * 
	 * @return The system's line break characters when a line break is
	 *         emphasized or an empty string if no line break sequence is
	 *         recommended.
	 */
	public static String toLineBreak( int aRowWidth ) {
		if ( isUseLineBreak( aRowWidth ) ) return getLineBreak();
		return "";
	}

	/**
	 * Determines the computer's name. First it tries to get it from the
	 * {@link InetAddress}, if it fails it tries to get it from the system's
	 * environment using the {@link EnvironmentProperty#COMPUTERNAME} (on
	 * Windows machines only) and if both fails, it returns the default
	 * {@link Literal#LOCALHOST} identifier.
	 * 
	 * @return The computer's name, as fallback, {@link Literal#LOCALHOST}
	 *         ("localhost") is returned.
	 */
	public static String getComputerName() {
		try {
			return InetAddress.getLocalHost().getHostName();
		}
		catch ( UnknownHostException e ) {
			String theName = EnvironmentProperty.COMPUTERNAME.getValue();
			if ( theName != null && theName.length() > 0 ) {
				return theName;
			}
		}
		return Literal.LOCALHOST.getName();
	}

	/**
	 * If on a *nix alike system, this method returns the output of the "uname
	 * -a" command: "uname" prints system information, "-a" instructs it to
	 * print all information.
	 * 
	 * @return The "uname -a" output or null if "uname" is not known.
	 */
	public static String getUname() {
		try {
			return exec( "uname -a" );
		}
		catch ( IOException | InterruptedException ignore ) {
			return null;
		}
	}

	/**
	 * Determines the width in characters of the system's terminal in use.
	 * 
	 * @return The width of the terminal in characters or -1 if the width cannot
	 *         be determined.
	 */
	public static int getTerminalWidth() {
		int theTerminalWidth = -1;
		try {
			// -----------------------------------------------------------------
			// Force an exception in case of being inside shutdown process to
			// prevent the TerminalFactory being initialized, which, when
			// shutdown is in progress, will show an ugly message including an
			// uninteresting stack trace.
			// -----------------------------------------------------------------
			Thread theDummyThread = new Thread();
			Runtime.getRuntime().addShutdownHook( theDummyThread );
			Runtime.getRuntime().removeShutdownHook( theDummyThread );
			// -----------------------------------------------------------------
			theTerminalWidth = TerminalFactory.get().getWidth();
		}
		catch ( Exception ignore ) {}

		// Try the common *ix like environment variable -->
		if ( theTerminalWidth <= 1 ) {
			String theResult = EnvironmentProperty.TERMINAL_WIDTH.getValue();
			if ( theResult != null ) {
				try {
					theTerminalWidth = Integer.valueOf( theResult.toString() );
				}
				catch ( NumberFormatException e ) {}
			}
		}
		// <-- Try the common *ix like environment variable

		if ( theTerminalWidth <= 1 ) {
			theTerminalWidth = -1;
		}

		return theTerminalWidth;
	}

	/**
	 * Determines the height in characters of the system's terminal in use.
	 * 
	 * @return The height of the terminal in characters or -1 if the height
	 *         cannot be determined.
	 */
	public static int getTerminalHeight() {
		int theTerminalHeight = -1;
		try {
			// Fails in case of being inside shutdown hook.
			theTerminalHeight = TerminalFactory.get().getHeight();
		}
		catch ( Exception ignore ) {}

		// Try the common *ix like environment variable -->
		if ( theTerminalHeight <= 1 ) {
			String theResult = EnvironmentProperty.TERMINAL_HEIGHT.getValue();
			if ( theResult != null ) {
				try {
					theTerminalHeight = Integer.valueOf( theResult.toString() );
				}
				catch ( NumberFormatException e ) {}
			}
		}
		// <-- Try the common *ix like environment variable

		if ( theTerminalHeight <= 1 ) {
			theTerminalHeight = -1;
		}

		return theTerminalHeight;
	}

	/**
	 * Does some calculation to always return a sound console width (never
	 * returns -1). Anything in the REFCODES.ORG artifacts which is intended to
	 * print to the console uses this method to determine some heuristic best
	 * console width.
	 * 
	 * @return A sound heuristic console width.
	 */
	public static int toConsoleWidth() {
		int theConsoleWidth = getTerminalWidth();
		int theWidth = -1;
		String theResult = toPropertyValue( SystemProperty.CONSOLE_WIDTH, EnvironmentProperty.CONSOLE_WIDTH );
		if ( theResult != null ) {
			try {
				theWidth = Integer.valueOf( theResult.toString() );
			}
			catch ( NumberFormatException e ) {}
		}

		if ( theWidth <= 1 ) {
			theWidth = theConsoleWidth;
			if ( theWidth < ConsoleDimension.MIN_WIDTH.getValue() || theWidth <= 1 ) {
				theWidth = ConsoleDimension.MIN_WIDTH.getValue();
			}
		}
		// <-- Fall back in case of crude determined width

		return theWidth;
	}

	/**
	 * Does some calculation to always return a sound console height (never
	 * returns -1). Anything in the REFCODES.ORG artifacts which is intended to
	 * print to the console uses this method to determine some heuristic best
	 * console height.
	 * 
	 * @return A sound heuristic console height.
	 */
	public static int toConsoleHeight() {
		int theConsoleHeight = getTerminalHeight();
		int theHeight = -1;
		String theResult = toPropertyValue( SystemProperty.CONSOLE_HEIGHT, EnvironmentProperty.CONSOLE_HEIGHT );
		if ( theResult != null ) {
			try {
				theHeight = Integer.valueOf( theResult.toString() );
			}
			catch ( NumberFormatException e ) {}
		}

		if ( theHeight <= 1 ) {
			theHeight = theConsoleHeight;
			if ( theHeight < ConsoleDimension.MIN_HEIGHT.getValue() || theHeight <= 1 ) {
				theHeight = ConsoleDimension.MIN_HEIGHT.getValue();
			}
		}
		// <-- Fall back in case of crude determined height

		return theHeight;
	}

	/**
	 * Gets the value for the provided properties, if non was found then the
	 * default value is taken. A {@link SystemProperty} elements wins over the
	 * {@link EnvironmentProperty} elements. The preceding
	 * {@link EnvironmentProperty} element wins over the succeeding
	 * {@link EnvironmentProperty} element. The default value is taken if non
	 * property had a value (a String with length &gt; 0).
	 * 
	 * @param aDefaultValue The default value to take when none other value was
	 *        set.
	 * @param aSystemProperty The system-property passed via <code>java
	 *        -D&lt;name&gt;=&lt;value&gt;</code>
	 * @param aEnvironmentProperties The properties looked for in the system's
	 *        environment variables.
	 * @return The best fitting value.
	 */
	public static String toPropertyValue( String aDefaultValue, SystemProperty aSystemProperty, EnvironmentProperty... aEnvironmentProperties ) {
		String theValue = aSystemProperty.getValue();
		if ( theValue != null && theValue.length() > 0 ) return theValue;
		if ( aEnvironmentProperties != null ) {
			for ( EnvironmentProperty eProperty : aEnvironmentProperties ) {
				theValue = eProperty.getValue();
				if ( theValue != null && theValue.length() > 0 ) return theValue;
			}
		}
		if ( theValue != null && theValue.length() > 0 ) return theValue;
		return aDefaultValue;
	}

	/**
	 * Gets the value for the provided properties, if non was found then null is
	 * returned. A {@link SystemProperty} elements wins over the
	 * {@link EnvironmentProperty} elements. The preceding
	 * {@link EnvironmentProperty} element wins over the succeeding
	 * {@link EnvironmentProperty} element. A null is taken if non property had
	 * a value (a String with length &gt; 0).
	 * 
	 * @param aSystemProperty The system-property passed via <code>java
	 *        -D&lt;name&gt;=&lt;value&gt;</code>
	 * @param aEnvironmentProperties The properties looked for in the system's
	 *        environment variables.
	 * @return The best fitting value or null if none was detected.
	 */
	public static String toPropertyValue( SystemProperty aSystemProperty, EnvironmentProperty... aEnvironmentProperties ) {
		String theValue = aSystemProperty.getValue();
		if ( theValue != null && theValue.length() > 0 ) return theValue;
		if ( aEnvironmentProperties != null ) {
			for ( EnvironmentProperty eProperty : aEnvironmentProperties ) {
				theValue = eProperty.getValue();
				if ( theValue != null && theValue.length() > 0 ) return theValue;
			}
		}
		if ( theValue != null && theValue.length() > 0 ) return theValue;
		return null;
	}

	/**
	 * Executes a command and returns the output.
	 *
	 * @param aCommand the command
	 * @return Null if execution failed, else the according output. An empty
	 *         {@link String} stands fur successful execution.
	 * @throws IOException in case there were problems executing the command.
	 * @throws InterruptedException thrown in case execution as been
	 *         interrupted.
	 */
	public static String exec( String aCommand ) throws IOException, InterruptedException {
		StringBuilder theBuilder = new StringBuilder();
		Runtime theRuntime = Runtime.getRuntime();
		Process theProcess = theRuntime.exec( aCommand );
		theProcess.waitFor();
		BufferedReader theReader = new BufferedReader( new InputStreamReader( theProcess.getInputStream() ) );
		String line;
		while ( (line = theReader.readLine()) != null ) {
			theBuilder.append( line );
		}
		theReader.close();
		return theBuilder.toString();
	}

	/**
	 * Determines whether ANSI escape sequences are supported by the terminal.
	 * 
	 * @return True in case ANSI escape sequences are supported, else false.
	 */
	public static boolean isAnsiTerminal() {
		if ( isCygwin() ) return true;
		jline.Terminal theTerminal = TerminalFactory.get();
		return (theTerminal.isAnsiSupported());
	}

	/**
	 * Determines whether ANSI escape sequences are forced to be supported or
	 * not by REFCODES.ORG artifacts. This overrules {@link #isAnsiTerminal()}
	 * for the REFCODES.ORG artifacts. Various settings such as
	 * {@link SystemProperty#CONSOLE_ANSI} and (in this order)
	 * {@link EnvironmentProperty#CONSOLE_ANSI} overrule
	 * {@link #isAnsiTerminal()}.
	 * 
	 * @return True in case ANSI escape sequences are forced to be used by
	 *         REFCODES.ORG artifacts or not.
	 */
	public static boolean toAnsiConsole() {
		String theResult = toPropertyValue( SystemProperty.CONSOLE_ANSI, EnvironmentProperty.CONSOLE_ANSI );
		if ( theResult != null ) {
			try {
				return NumericalUtility.toBoolean( theResult.toString() );
			}
			catch ( IllegalArgumentException e ) {}
		}
		return isAnsiTerminal();
	}

	/**
	 * Returns the operating saystem's temp folder.
	 * 
	 * @return The folder used by the OS for storing temporary files.
	 */
	public static File getTempDir() {
		String tempDir = SystemProperty.TEMP_DIR.getValue();
		File theTempDir = new File( tempDir );
		return theTempDir;
	}

	/**
	 * Determines the encoding of the system (system's console) in use as of
	 * {@link System#out}.
	 * 
	 * @return The encoding (of the console).
	 */
	public static String getSystemEncoding() {
		OutputStreamWriter theOutWriter = new OutputStreamWriter( System.out );
		return theOutWriter.getEncoding();
	}

	/**
	 * Gathers all available system information from this Artifac's point of
	 * view.
	 * 
	 * @return A {@link Map} containing the available information being
	 *         gathered.
	 */
	public static Map<String, String> toSystemInfo() {
		Map<String, String> theInfo = new HashMap<>();
		theInfo.put( "computerName", getComputerName() + getLineBreak() );
		theInfo.put( "operatingSystem", OperatingSystem.toOperatingSystem() + getLineBreak() );
		theInfo.put( "shell", Shell.toShell() + getLineBreak() );
		theInfo.put( "terminal", Terminal.toTerminal() + getLineBreak() );
		theInfo.put( "terminalWidth", getTerminalWidth() + getLineBreak() );
		theInfo.put( "terminalHeight", getTerminalHeight() + getLineBreak() );
		theInfo.put( "enforceConsoleWidth", toConsoleWidth() + getLineBreak() );
		theInfo.put( "enforceConsleHeight", toConsoleHeight() + getLineBreak() );
		theInfo.put( "consoleWidthLineBreak", isUseLineBreak( getTerminalWidth() ) + getLineBreak() );
		theInfo.put( "ansiSupport", isAnsiTerminal() + getLineBreak() );
		theInfo.put( "enforceAnsiUsage", toAnsiConsole() + getLineBreak() );
		theInfo.put( "systemEncoding", getSystemEncoding() + getLineBreak() );
		theInfo.put( "systemLineBreak", getLineBreak().replaceAll( "\\n", "\\\\n" ).replaceAll( "\\r", "\\\\r" ) + getLineBreak() );
		theInfo.put( "tempDir", getTempDir() + getLineBreak() );
		theInfo.put( "uname", getUname() + getLineBreak() );
		return theInfo;
	}

	/**
	 * Gathers all available system information from this Artifac's point of
	 * view. This method may rely on the output of {@link #toSystemInfo()}.
	 * 
	 * @return A {@link String} containing the available information being
	 *         gathered.
	 */
	public static String toPrettySystemInfo() {
		StringBuffer theBuffer = new StringBuffer();
		Map<String, String> theInfo = toSystemInfo();
		int maxLength = -1;
		for ( String eKey : theInfo.keySet() ) {
			if ( eKey.length() > maxLength ) maxLength = eKey.length();
		}
		String tmpKey;
		for ( String eKey : theInfo.keySet() ) {
			tmpKey = eKey;
			while ( tmpKey.length() < maxLength ) {
				tmpKey = tmpKey + " ";
			}
			theBuffer.append( tmpKey + " := " + theInfo.get( eKey ) );
		}
		return theBuffer.toString();
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Tries to determine whether the command line interpreter (CLI) is a Cygwin
	 * one.
	 * 
	 * @return True in case we think we are running in Cygwin. Use
	 *         {@link #getCommandLineInterpreter()} to test for the type of CLI,
	 *         in case you got to distinguish the {@link Shell#SHELL} type, then
	 *         use this {@link #isCygwin()} method.
	 */
	static boolean isCygwin() {
		if ( OperatingSystem.toOperatingSystem() == OperatingSystem.WINDOWS ) {
			String theUname = getUname();
			if ( theUname != null && theUname.toLowerCase().indexOf( "cygwin" ) != -1 ) {
				// "PWD" is only set by cygwin, not in CMD.EXE:
				String thePwd = System.getenv( "PWD" );
				return (thePwd != null && thePwd.length() != 0);
			}
		}
		return false;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
