// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.runtime;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.lang.management.ManagementFactory;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.refcodes.data.CommandArgPrefix;
import org.refcodes.data.Delimiter;
import org.refcodes.data.Folder;
import org.refcodes.data.Folders;
import org.refcodes.data.Literal;
import org.refcodes.data.Scheme;
import org.refcodes.data.SystemProperty;
import org.refcodes.exception.ExceptionUtility;
import org.refcodes.exception.HiddenException;

/**
 * Utility for acquiring runtime information on software systems, classes or
 * objects.
 */
public final class RuntimeUtility {

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static String[] MAIN_METHODS = {
			"main", "start", "<clinit>", "<init>"
	};

	private static String[] MAIN_METHOD_PREFIXES = {
			"test"
	};

	private static String[] FRAMEWORK_LAUNCHERS = {
			"JarLauncher", "TestRunner", "ForkedBooter"
	};

	private static Scheme[] JAR_PROTOCOLS = new Scheme[] {
			Scheme.JAR, Scheme.ZIP, Scheme.SH
	};

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Private empty constructor to prevent instantiation as of being a utility
	 * with just static public methods.
	 */
	private RuntimeUtility() {}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Gets the stack trace for the current thread.
	 * 
	 * @return The stack trace from the current thread.
	 */
	public static String toStackTrace() {
		StringWriter theStringWriter = new StringWriter();
		new Throwable( "" ).printStackTrace( new PrintWriter( theStringWriter ) );
		return theStringWriter.toString();
	}

	/**
	 * Returns the stack trace element belonging to the direct caller of this
	 * method. When you use this method in your code, you get the stack trace
	 * element of your method (invoking this method).
	 * 
	 * @return The stack element of the direct caller of this method.
	 */
	public static StackTraceElement getCallerStackTraceElement() {
		StackTraceElement[] theStackTraceElements = Thread.currentThread().getStackTrace();
		for ( StackTraceElement eStackTraceElement : theStackTraceElements ) {
			if ( !isSkipStackTraceElement( eStackTraceElement ) ) { return eStackTraceElement; }
		}
		return null;
	}

	/**
	 * Returns the stack trace element belonging to the caller of the callee.
	 * Best you do not use the {@link Object#getClass()} method, instead use
	 * YourClass.class as as of inheritance, {@link Object#getClass()} returns
	 * the wrong type not being the actual callee!
	 * 
	 * @param aCallee The callee class which wants to find out who called it.
	 * 
	 * @return The stack element of the caller of the callee or null if the
	 *         callee is not present or if there is no caller of the given
	 *         callee in the current stack trace.
	 */
	public static StackTraceElement getCallerStackTraceElement( Class<?> aCallee ) {
		StackTraceElement[] theStackTraceElements = Thread.currentThread().getStackTrace();
		String theCalleeClassName = aCallee.getName();
		boolean hasCalleeElement = false;
		for ( StackTraceElement eStackTraceElement : theStackTraceElements ) {
			if ( hasCalleeElement && !eStackTraceElement.getClassName().equals( theCalleeClassName ) ) { return eStackTraceElement; }
			if ( eStackTraceElement.getClassName().equals( theCalleeClassName ) ) {
				hasCalleeElement = true;
			}
		}
		return null;
	}

	/**
	 * Same as {@link #getCallerStackTraceElement(Class)} with the difference
	 * that the passed callees are tried out one after the other until the first
	 * caller determined for a callee is returned.
	 *
	 * @param aCallees the callees
	 * @return the caller stack trace element
	 */
	public static StackTraceElement getCallerStackTraceElement( Class<?>... aCallees ) {
		StackTraceElement eCaller = null;
		for ( Class<?> eClallee : aCallees ) {
			eCaller = getCallerStackTraceElement( eClallee );
			if ( eCaller != null ) { return eCaller; }
		}
		return null;
	}

	/**
	 * Returns the stack trace element belonging to the caller of the callee.
	 * The callee can also be a package namespace where the matchee's must begin
	 * with the given package namespace.
	 * 
	 * @param aCalleeClassName The callee class name or package namespace which
	 *        wants to find out who called it.
	 * 
	 * @return The stack element of the caller of the callee or null if the
	 *         callee is not present or if there is no caller of the given
	 *         callee in the current stack trace.
	 */
	public static StackTraceElement getCallerStackTraceElement( String aCalleeClassName ) {
		StackTraceElement[] theStackTraceElements = Thread.currentThread().getStackTrace();
		boolean hasCalleeElement = false;
		for ( StackTraceElement eStackTraceElement : theStackTraceElements ) {
			if ( hasCalleeElement && !eStackTraceElement.getClassName().startsWith( aCalleeClassName ) ) { return eStackTraceElement; }
			if ( eStackTraceElement.getClassName().startsWith( aCalleeClassName ) ) {
				hasCalleeElement = true;
			}
		}
		return null;
	}

	/**
	 * Same as {@link #getCallerStackTraceElement(String)} with the difference
	 * that the passed callees are tried out one after the other until the first
	 * caller determined for a callee is returned.
	 *
	 * @param aCalleeClassNames the callee class names
	 * @return the caller stack trace element
	 */
	public static StackTraceElement getCallerStackTraceElement( String... aCalleeClassNames ) {
		StackTraceElement eCaller = null;
		for ( String eClallee : aCalleeClassNames ) {
			eCaller = getCallerStackTraceElement( eClallee );
			if ( eCaller != null ) { return eCaller; }
		}
		return null;
	}

	/**
	 * Returns the type of the ({@link Class}) belonging to the direct caller of
	 * this method. When you use this method in your code, you get the
	 * {@link Class} of your method (invoking this method).
	 * 
	 * @return The type ({@link Class}) of the direct caller of this method.
	 */
	public static Class<?> getCallerType() {
		StackTraceElement theStackTraceElement = getCallerStackTraceElement();
		return toClass( theStackTraceElement );
	}

	/**
	 * Returns the type of the ({@link Class}) belonging to the caller of the
	 * callee.
	 * 
	 * @param aCallee The callee class which wants to find out who called it.
	 * 
	 * @return The type ({@link Class}) of the caller of the caller of this
	 *         method.
	 */
	public static Class<?> getCallerType( Class<?> aCallee ) {
		StackTraceElement theStackTraceElement = getCallerStackTraceElement( aCallee );
		return toClass( theStackTraceElement );
	}

	/**
	 * Same as {@link #getCallerType(Class)} with the difference that the passed
	 * callees are tried out one after the other until the first caller
	 * determined for a callee is returned.
	 *
	 * @param aCallees the callees
	 * @return the caller type
	 */
	public static Class<?> getCallerType( Class<?>... aCallees ) {
		StackTraceElement theStackTraceElement = getCallerStackTraceElement( aCallees );
		return toClass( theStackTraceElement );
	}

	/**
	 * Retrieves the {@link Class} type to which the {@link StackTraceElement}
	 * belongs.
	 * 
	 * @param aStackTraceElement The {@link StackTraceElement} for which to get
	 *        the according {@link Class}.
	 * 
	 * @return The type ({@link Class}) of the according
	 *         {@link StackTraceElement}.
	 */
	public static Class<?> toClass( StackTraceElement aStackTraceElement ) {
		try {
			return Class.forName( aStackTraceElement.getClassName() );
		}
		catch ( ClassNotFoundException e ) {
			throw new HiddenException( e );
		}
	}

	/**
	 * Retrieves the method name from a stack trace element.
	 * 
	 * @param aStackTraceElement The stack trace element from which to retrieve
	 *        the method name.
	 * 
	 * @return The method name or null in case the stack trace element was null.
	 */
	public static String toMethodName( StackTraceElement aStackTraceElement ) {
		if ( aStackTraceElement == null ) return null;
		return aStackTraceElement.getMethodName();
	}

	/**
	 * Returns the class name part from a stack trace element.
	 * 
	 * Retrieves the fully qualified class name from a stack trace element.
	 * 
	 * @param aStackTraceElement The stack trace element from which to retrieve
	 *        the class name.
	 * 
	 * @return The class name without the package declaration or null in case
	 *         the stack trace element was null.
	 */
	public static String toClassName( StackTraceElement aStackTraceElement ) {
		if ( aStackTraceElement == null ) return null;
		return toClassName( aStackTraceElement.getClassName() );
	}

	/**
	 * Retrieves the fully qualified class name from a stack trace element.
	 * 
	 * @param aStackTraceElement The stack trace element from which to retrieve
	 *        the fully qualified class name.
	 * 
	 * @return The fully qualified class name or null in case the stack trace
	 *         element was null.
	 */
	public static String toFullyQualifiedClassName( StackTraceElement aStackTraceElement ) {
		if ( aStackTraceElement == null ) return null;
		return aStackTraceElement.getClassName();
	}

	/**
	 * Retrieves the fully qualified method name from a stack trace element.
	 * This adds the method name to the fully qualified path name separated by a
	 * hash "#".
	 * 
	 * @param aStackTraceElement The stack trace element from which to retrieve
	 *        the fully qualified method name.
	 * 
	 * @return The fully qualified method name or null in case the stack trace
	 *         element was null.
	 */
	public static String toFullyQualifiedMethodName( StackTraceElement aStackTraceElement ) {
		if ( aStackTraceElement == null ) return null;
		return aStackTraceElement.getClassName() + Delimiter.METHOD_NAME.getChar() + aStackTraceElement.getMethodName();
	}

	/**
	 * Retrieves the fully qualified method name of the caller of this method.
	 * This adds the method name to the caller's fully qualified path name
	 * separated by a hash "#".
	 * 
	 * @return The fully qualified method name.
	 */
	public static String toFullyQualifiedClassName() {
		StackTraceElement theStackTraceElement = getCallerStackTraceElement();
		return theStackTraceElement.getClassName();
	}

	/**
	 * Retrieves the fully qualified method name of the caller of this method.
	 * This adds the method name to the caller's fully qualified path name
	 * separated by a hash "#".
	 * 
	 * @return The fully qualified method name.
	 */
	public static String toFullyQualifiedMethodName() {
		StackTraceElement theStackTraceElement = getCallerStackTraceElement();
		return theStackTraceElement.getClassName() + Delimiter.METHOD_NAME.getChar() + theStackTraceElement.getMethodName();
	}

	/**
	 * Retrieves the fully qualified method name of the caller of this method.
	 * This adds the method name to the caller's fully qualified path name
	 * separated by a hash "#".
	 * 
	 * @return The fully qualified method name.
	 */
	public static String toMethodName() {
		StackTraceElement theStackTraceElement = getCallerStackTraceElement();
		return theStackTraceElement.getMethodName();
	}

	/**
	 * Retrieves the class name of the caller of this method without the fully
	 * qualified package name part.
	 * 
	 * @return The class name.
	 */
	public static String toClassName() {
		return toClassName( toFullyQualifiedClassName() );
	}

	/**
	 * Retrieves the fully qualified package name of the caller of this method
	 * without the class name part.
	 * 
	 * @return The fully qualified package name.
	 */
	public static String toFullyQualifiedPackageName() {
		return toFullyQualifiedPackageName( toFullyQualifiedClassName() );
	}

	/**
	 * Retrieves the fully qualified package name from a stack trace element.
	 * 
	 * @param aStackTraceElement The stack trace element from which to retrieve
	 *        the fully qualified package name.
	 * 
	 * @return The fully qualified package name.
	 */
	public static String toFullyQualifiedPackageName( StackTraceElement aStackTraceElement ) {
		return toFullyQualifiedPackageName( toFullyQualifiedClassName( aStackTraceElement ) );
	}

	/**
	 * Returns the class name part from a fully qualified class name (which has
	 * the fully qualified package name as part of its name).
	 * 
	 * @param aFullyQualifiedClassName The fully qualified class name.
	 * 
	 * @return The class name without the package declaration.
	 */
	public static String toClassName( String aFullyQualifiedClassName ) {
		String theClassName;
		int theClassIndex = aFullyQualifiedClassName.lastIndexOf( '.' );
		if ( theClassIndex != -1 ) {
			theClassName = aFullyQualifiedClassName.substring( theClassIndex + 1 );
		}
		else {
			theClassName = aFullyQualifiedClassName;
		}
		int theInnerClassIndex = theClassName.indexOf( '$' );
		if ( theInnerClassIndex != -1 ) {
			theClassName = theClassName.substring( theInnerClassIndex + 1 );
		}
		return theClassName;
	}

	/**
	 * Returns the fully qualified package name part from a fully qualified
	 * class name (which has the fully qualified package name as part of its
	 * name).
	 * 
	 * @param aFullyQualifiedClassName The fully qualified class name.
	 * 
	 * @return The fully qualified package name without the class name.
	 */
	public static String toFullyQualifiedPackageName( String aFullyQualifiedClassName ) {
		int theIndex = aFullyQualifiedClassName.lastIndexOf( '.' );
		if ( theIndex != -1 ) { return aFullyQualifiedClassName.substring( 0, theIndex ); }
		return "";
	}

	/**
	 * A {@link Cloneable} object cannot directly be cloned by casting it to be
	 * {@link Cloneable} :-( Thereforee this method does the job.
	 * 
	 * Citation From Josh Bloch's Effective Java: "The {@link Cloneable}
	 * interface was intended as a mixin interface for objects to advertise that
	 * they permit cloning. Unfortunately it fails to serve this purpose ...
	 * This is a highly atypical use of interfaces and not one to be emulated
	 * ... In order for implementing the interface to have any effect on a
	 * class, it and all of its superclasses must obey a fairly complex,
	 * unenforceable and largely undocumented protocol"
	 *
	 * @param <T> the generic type
	 * @param aObj The object to be cloned.
	 * @return The cloned object.
	 * @throws CloneNotSupportedException in case the object cannot be cloned.
	 * @see "http://stackoverflow.com/questions/1138769/why-is-the-clone-method-protected-in-java-lang-object"
	 */

	@SuppressWarnings("unchecked")
	public static <T> T toClone( T aObj ) throws CloneNotSupportedException {
		if ( aObj instanceof Cloneable ) {
			try {
				return (T) aObj.getClass().getMethod( "clone" ).invoke( aObj );
			}
			catch ( IllegalAccessException | IllegalArgumentException | InvocationTargetException | NoSuchMethodException | SecurityException e ) {
				throw new CloneNotSupportedException( ExceptionUtility.toMessage( e ) );
			}
		}
		else {
			throw new CloneNotSupportedException( "The type \"" + aObj.getClass().getName() + "}\" does not implement the \"" + Cloneable.class.getName() + "\" interface." );
		}
	}

	/**
	 * Creates a string of a super class's {@link Object#toString()} method and
	 * the provided "toString" text.
	 * 
	 * @param aToString The provided "toString" text.
	 * 
	 * @param aSuperToString A super class's {@link Object#toString()} method's
	 *        {@link String}.
	 * 
	 * @return The "concatenated" and formatted new {@link String} to be
	 *         returned by an implementing class's {@link Object#toString()}
	 *         method.
	 */
	public static String toString( String aToString, String aSuperToString ) {
		return aToString + " (" + aSuperToString + ")";
	}

	/**
	 * Bad hack to get the JVM's (process ID) PID of the process running your
	 * JVM instance.
	 * 
	 * @return The PID (process ID) of the JVM running your thread.
	 * 
	 * @see "http://stackoverflow.com/questions/35842/how-can-a-java-program-get-its-own-process-id"
	 */
	public static Long getPid() {
		Long thePid = null;
		try {
			thePid = Long.parseLong( SystemProperty.PROCESS_ID.getValue() );
		}
		catch ( NumberFormatException e1 ) {
			String theJvmName = ManagementFactory.getRuntimeMXBean().getName();
			if ( theJvmName.indexOf( '@' ) != -1 ) {
				try {
					thePid = Long.parseLong( theJvmName.substring( 0, theJvmName.indexOf( '@' ) ) );
				}
				catch ( NumberFormatException e2 ) {
					// Unable to determine PID
				}
			}
		}
		return thePid;
	}

	/**
	 * Bad hack to kill an OS thread by PID. The current threads does not wait
	 * till the operation finished.
	 * 
	 * @param aPid The process ID (PID) of the process to kill.
	 * 
	 * @return The {@link Process} object representing the kill operation. This
	 *         instance will let you wait till the operation finished
	 *         {@link Process#waitFor()} and provides access to the
	 *         {@link Process#exitValue()}
	 * 
	 * @throws IOException Thrown in case of failing to successfully execute the
	 *         kill operation.
	 * 
	 * @see "http://stackoverflow.com/questions/9573696/kill-a-process-based-on-pid-in-java"
	 * @see "http://stackoverflow.com/questions/2950338/how-can-i-kill-a-linux-process-in-java-with-sigkill-process-destroy-does-sigte"
	 */
	public static Process killProcess( Long aPid ) throws IOException {
		Process theProcess = null;
		String theCmd;
		switch ( OperatingSystem.toOperatingSystem() ) {
		case WINDOWS:
			theCmd = "taskkill /F /PID " + aPid;
			theProcess = Runtime.getRuntime().exec( theCmd );
			break;
		case UNIX:
			theCmd = "kill -9 " + aPid;
			theProcess = Runtime.getRuntime().exec( theCmd );
			break;
		default:
		}
		return theProcess;
	}

	/**
	 * Bad hack to kill an OS thread by PID. The current threads does wait till
	 * the operation finished.
	 *
	 * @param aPid The process ID (PID) of the process to kill.
	 * @return True in case killing the process was successful (e.g. the kill
	 *         operation returned an exit code 0), else false.
	 * @throws IOException Thrown in case of failing to successfully execute the
	 *         kill operation.
	 * @throws InterruptedException the interrupted exception
	 * @see "http://stackoverflow.com/questions/9573696/kill-a-process-based-on-pid-in-java"
	 * @see "http://stackoverflow.com/questions/2950338/how-can-i-kill-a-linux-process-in-java-with-sigkill-process-destroy-does-sigte"
	 */
	public static boolean kill( Long aPid ) throws IOException, InterruptedException {
		Process theProcess = killProcess( aPid );
		if ( theProcess != null ) {
			theProcess.waitFor();
			return (theProcess.exitValue() == 0);
		}
		return false;
	}

	/**
	 * Generates the base path relative to the given class location. Depending
	 * on the runtime, the path is truncated till the required path is
	 * determined.
	 *
	 * @return The base path of this application.
	 */
	public static File toLauncherDir() {
		Class<?> baseClass = getMainClass();
		if ( baseClass == null ) baseClass = RuntimeUtility.class;
		String basePath = baseClass.getProtectionDomain().getCodeSource().getLocation().getPath();
		if ( basePath.startsWith( Scheme.FILE.toProtocol() ) ) {
			basePath = basePath.substring( Scheme.FILE.toProtocol().length() );
		}
		String truncate;
		try {
			String theCanonicalName = baseClass.getCanonicalName();
			truncate = theCanonicalName.replace( '.', '/' );
		}
		catch ( Exception e ) {
			return null;
		}
		truncate = truncate.substring( 0, truncate.indexOf( baseClass.getSimpleName() ) );
		int endIndex = basePath.indexOf( truncate );
		if ( endIndex > 0 ) {
			basePath = basePath.substring( 0, endIndex );
		}
		if ( basePath.endsWith( "" + Delimiter.PATH.getChar() ) ) {
			basePath = basePath.substring( 0, basePath.length() - 1 );
		}
		if ( basePath.endsWith( Folder.CLASSES.getName() ) ) {
			basePath = basePath.substring( 0, basePath.indexOf( Folder.CLASSES.getName() ) );
		}
		int index = -1;

		for ( Scheme eProtocol : JAR_PROTOCOLS ) {
			index = eProtocol.firstMarkerIndex( basePath.toLowerCase() );
			if ( index != -1 ) break;
		}
		if ( index != -1 ) {
			basePath = basePath.substring( 0, index );
			index = basePath.lastIndexOf( Delimiter.PATH.getChar() );
			if ( index != -1 ) {
				basePath = basePath.substring( 0, index );
			}
		}
		if ( !basePath.endsWith( "" + Delimiter.PATH.getChar() ) ) {
			basePath = basePath + Delimiter.PATH.getChar();
		}
		File baseFile = new File( basePath );
		if ( basePath.endsWith( Folder.TARGET.getName() + Delimiter.PATH.getChar() ) ) { return baseFile; }
		File parentFile = baseFile.getParentFile();
		return parentFile;
	}

	/**
	 * Returns the main class launching the application.
	 *
	 * @return The main class.
	 */
	@SuppressWarnings("rawtypes")
	public static Class getMainClass() {
		StackTraceElement[] stack = Thread.currentThread().getStackTrace();
		StackTraceElement eElement;
		Class<?> mainClass;

		List<StackTraceElement> theCandidates = new ArrayList<>();

		for ( int i = stack.length - 1; i > 0; i-- ) {
			eElement = stack[i];
			for ( String eMain : MAIN_METHODS ) {
				if ( eElement.getMethodName().equals( eMain ) ) {
					theCandidates.add( eElement );
					break;
				}
			}
		}

		for ( int i = stack.length - 1; i > 0; i-- ) {
			eElement = stack[i];
			for ( String ePrefix : MAIN_METHOD_PREFIXES ) {
				if ( eElement.getMethodName().startsWith( ePrefix ) ) {
					theCandidates.add( eElement );
					break;
				}
			}
		}

		// First try to skip framework launchers |-->
		NEXT: for ( StackTraceElement eCandidate : theCandidates ) {
			for ( String eLauncer : FRAMEWORK_LAUNCHERS ) {
				if ( eCandidate.getClassName().toLowerCase().contains( eLauncer.toLowerCase() ) ) {
					continue NEXT;
				}
			}
			try {
				mainClass = Class.forName( eCandidate.getClassName() );
				if ( mainClass.getProtectionDomain().getCodeSource() != null ) { return mainClass; }
			}
			catch ( ClassNotFoundException e ) {}
		}
		// First try to skip framework launchers <--|

		for ( StackTraceElement eCandidate : theCandidates ) {
			try {
				mainClass = Class.forName( eCandidate.getClassName() );
				if ( mainClass.getProtectionDomain().getCodeSource() != null ) { return mainClass; }
			}
			catch ( ClassNotFoundException e ) {}
		}

		return null;
	}

	/**
	 * Returns a list of (existing) folders which are candidates for external
	 * resources. The provided base directories are taken and a list of
	 * directories relative to these base directories are generated as defined
	 * in the {@link Folders#CONFIG_DIRS}: The actual directories being returned
	 * (in case them exist) are as follows, relative to each of your
	 * application's base directories:
	 * 
	 * <ul>
	 * <li>../config"</li>
	 * <li>../etc"</li>
	 * <li>../settings"</li>
	 * <li>../.config"</li>
	 * <li>../settings"</li>
	 * <li>."</li>
	 * <li>./config"</li>
	 * <li>./etc"</li>
	 * <li>./settings"</li>
	 * <li>./.config"</li>
	 * <li>./settings"</li>
	 * </ul>
	 * 
	 * In case you pass a JVM argument via
	 * "-Dconfig.dir=path_to_your_config_dir" (where path_to_your_config_dir
	 * stands for the path to the directory where you placed configuration files
	 * such as the "runtimelogger-config.xml" file), then your
	 * path_to_your_config_dir is placed first in the list (in case the
	 * directory exists).See {@link SystemProperty#CONFIG_DIR}
	 * 
	 * @param aBaseDirectories The application's base directories, in case a
	 *        {@link File} instance represent a file (not a directory), then the
	 *        according file's parent directory is used.
	 * 
	 * @return A list containing potential application configuration folders.
	 * 
	 * @throws IOException Thrown in case there were {@link File} related
	 *         problems determining the folders.
	 */
	public static List<File> toConfigDirs( File... aBaseDirectories ) throws IOException {
		List<File> theDirs = new ArrayList<File>();
		List<File> theBaseDirs = new ArrayList<File>();
		if ( aBaseDirectories == null ) {
			aBaseDirectories = new File[] {
					new File( "." )
			};
		}

		// First is the directory specified by system property |-->
		File theSystemDir = toDirectory( SystemProperty.CONFIG_DIR.getValue() );
		if ( theSystemDir != null ) {
			theDirs.add( theSystemDir );
		}
		// First is the directory specified by system property <--|

		for ( File eBaseDir : aBaseDirectories ) {
			if ( eBaseDir != null ) {
				// Try to get the parent if pointing to a non-existing file |-->
				if ( !eBaseDir.exists() || !eBaseDir.isDirectory() ) {
					File theParentFile = eBaseDir.getAbsoluteFile().getParentFile();
					if ( theParentFile != null ) {
						eBaseDir = theParentFile;
					}
				}
				// Try to get the parent if pointing to a non-existing file <--|

				File eConfigDir;
				if ( eBaseDir != null && eBaseDir.exists() && eBaseDir.isDirectory() && eBaseDir.canRead() ) {
					// Determine according child folders:
					for ( String eDirName : Folders.CONFIG_DIRS.getNames() ) {
						eConfigDir = new File( eBaseDir, eDirName );
						if ( eConfigDir != null && eConfigDir.exists() && eConfigDir.isDirectory() && eConfigDir.canRead() ) {
							if ( !containsDir( eConfigDir, theDirs, theBaseDirs ) ) {
								theDirs.add( eConfigDir );
							}
						}
					}
					// Add this folder:
					if ( !containsDir( eBaseDir, theDirs, theBaseDirs ) ) {
						if ( eBaseDir.getAbsolutePath().endsWith( Folder.TARGET.getName() ) ) {
							theBaseDirs.add( 0, eBaseDir );
						}
						else {
							theBaseDirs.add( eBaseDir );
						}
					}

					// Determine according parent folders:
					for ( String eDirName : Folders.CONFIG_DIRS.getNames() ) {
						eConfigDir = new File( eBaseDir.getParentFile(), eDirName );
						if ( eConfigDir != null && eConfigDir.exists() && eConfigDir.isDirectory() && eConfigDir.canRead() ) {
							if ( !containsDir( eConfigDir, theDirs, theBaseDirs ) ) {
								theDirs.add( eConfigDir );
							}
						}
					}
				}

				// |--> For Maven: Folder ends with "target"? Try "classes"
				if ( eBaseDir.getAbsolutePath().endsWith( Folder.TARGET.getName() ) ) {
					eConfigDir = new File( eBaseDir, Folder.CLASSES.getName() );
					if ( eConfigDir != null && eConfigDir.exists() && eConfigDir.isDirectory() && eConfigDir.canRead() ) {
						if ( !containsDir( eConfigDir, theDirs, theBaseDirs ) ) {
							theDirs.add( eConfigDir );
						}
					}
					eConfigDir = new File( eBaseDir, Folder.TEST_CLASSES.getName() );
					if ( eConfigDir != null && eConfigDir.exists() && eConfigDir.isDirectory() && eConfigDir.canRead() ) {
						if ( !containsDir( eConfigDir, theDirs, theBaseDirs ) ) {
							theDirs.add( eConfigDir );
						}
					}
				}
				else {
					eConfigDir = new File( new File( eBaseDir, Folder.TARGET.getName() ), Folder.CLASSES.getName() );
					if ( eConfigDir != null && eConfigDir.exists() && eConfigDir.isDirectory() && eConfigDir.canRead() ) {
						if ( !containsDir( eConfigDir, theDirs, theBaseDirs ) ) {
							theDirs.add( eConfigDir );
						}
					}
					eConfigDir = new File( new File( eBaseDir, Folder.TARGET.getName() ), Folder.TEST_CLASSES.getName() );
					if ( eConfigDir != null && eConfigDir.exists() && eConfigDir.isDirectory() && eConfigDir.canRead() ) {
						if ( !containsDir( eConfigDir, theDirs, theBaseDirs ) ) {
							theDirs.add( eConfigDir );
						}

					}
				}
				// <--| For Maven: Folder ends with "target"? Try "classes"
			}
		}
		// Prefer dedicated config folders over base folders |-->
		theDirs.addAll( theBaseDirs );
		// Prefer dedicated config folders over base folders <--|
		return theDirs;
	}

	/**
	 * The application's base directory (where your JAR or your classes reside)
	 * is taken using the folder as retrieved by {@link #toLauncherDir()} and
	 * passed to {@link #toConfigDirs(File...)}.
	 * 
	 * @return A list containing potential application configuration folders.
	 * 
	 * @throws IOException Thrown in case there were {@link File} related
	 *         problems determining the folders.
	 */
	public static List<File> toConfigDirs() throws IOException {
		return toConfigDirs( new File[] {
				toLauncherDir()
		} );
	}

	/**
	 * The {@link File} representing the potential base directory together with
	 * the application's base directory where your JAR or your classes reside)
	 * is taken (using the folder as retrieved by {@link #toLauncherDir()}) and
	 * passed to {@link #toConfigDirs(File...)} and tested for application
	 * configuration folders.
	 * 
	 * @param aBaseDirectory The {@link File} representing the potential base
	 *        directory.
	 * 
	 * @return A {@link File} representing a potential application configuration
	 *         folder.
	 * 
	 * @throws IOException Thrown in case there were {@link File} related
	 *         problems determining the folders.
	 */
	public static List<File> toConfigDirs( File aBaseDirectory ) throws IOException {
		return toConfigDirs( aBaseDirectory, toLauncherDir() );
	}

	/**
	 * Creates a {@link Map} from the provided command line arguments using the
	 * {@link CommandArgPrefix#toPrefixes()} prefixes to identify the keys (and
	 * the values) from the provided arguments.
	 * 
	 * @param aArgs The command line arguments to convert to properties.
	 * 
	 * @return The {@link Map} containing the determined properties.
	 */
	public static Map<String, String> toProperties( String[] aArgs ) {
		return toProperties( aArgs, CommandArgPrefix.toPrefixes(), Delimiter.INDEX.getChar() );

	}

	/**
	 * Creates a {@link Map} from the provided command line arguments using the
	 * provided prefixes to identify the keys (and the values) from the provided
	 * arguments.
	 * 
	 * @param aArgs The command line arguments to convert to properties.
	 * @param aPrefixes The prefixes to be used to identify options.
	 * @param aDelimiter The delimiter to use when generating non colliding
	 *        keys.
	 * 
	 * @return The {@link Map} containing the determined properties.
	 */
	public static Map<String, String> toProperties( String[] aArgs, String[] aPrefixes, char aDelimiter ) {
		Map<String, String> theBuilder = new HashMap<>();
		String eOption = null;
		String eNextOption, eArg;
		for ( int i = 0; i < aArgs.length; i++ ) {
			eArg = aArgs[i];
			eNextOption = toOption( eArg, aPrefixes );
			if ( eOption != null ) {
				if ( eNextOption == null ) {
					putProperty( theBuilder, eOption, eArg, aDelimiter );
					eOption = null;
				}
				else {
					putProperty( theBuilder, eOption, Literal.TRUE.getName(), aDelimiter );
					eOption = eNextOption;
					if ( i == aArgs.length - 1 ) {
						putProperty( theBuilder, eOption, Literal.TRUE.getName(), aDelimiter );
					}
				}
			}
			else {
				if ( eNextOption != null ) {
					eOption = eNextOption;
					if ( i == aArgs.length - 1 ) {
						putProperty( theBuilder, eOption, Literal.TRUE.getName(), aDelimiter );
					}
				}
				else {
					putProperty( theBuilder, null, eArg, aDelimiter );
				}
			}
		}
		return theBuilder;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Puts a property and resolves the key in case of name collisions.
	 * 
	 * @param aProperties The properties which to modify.
	 * @param aKey The key which to put.
	 * @param aValue The value to put.
	 * @param aDelimiter The delimiter to use when generating non colliding
	 *        keys.
	 */
	protected static void putProperty( Map<String, String> aProperties, String aKey, String aValue, char aDelimiter ) {
		if ( aProperties.containsKey( aKey ) ) {
			String eNextKey = toNextKey( aProperties, aKey, aDelimiter );
			aProperties.put( eNextKey, aProperties.remove( aKey ) );
			eNextKey = toNextKey( aProperties, aKey, aDelimiter );
			aProperties.put( eNextKey, aValue );
		}
		else if ( aProperties.containsKey( aKey + aDelimiter + "0" ) ) {
			String eNextKey = toNextKey( aProperties, aKey, aDelimiter );
			aProperties.put( eNextKey, aValue );
		}
		else {
			aProperties.put( aKey, aValue );
		}
	}

	/**
	 * Returns the next free key in the properties in case of key name
	 * collisions..
	 * 
	 * @param aProperties The properties for which to determine the next key.
	 * @param aKey The key for which to test for name collisions.
	 * @param aDelimiter The delimiter to use when creating non colliding keys.
	 * 
	 * @return The next non colliding key or null if there are no collisions.
	 */
	protected static String toNextKey( Map<String, String> aProperties, String aKey, char aDelimiter ) {
		if ( aProperties.containsKey( aKey ) || aProperties.containsKey( aKey + aDelimiter + "0" ) ) {
			String eNextKey;
			int index = 0;
			eNextKey = aKey + aDelimiter + index;
			while ( aProperties.containsKey( eNextKey ) ) {
				index++;
				eNextKey = aKey + aDelimiter + index;
			}
			return eNextKey;
		}
		return null;
	}

	/**
	 * When the provided argument starts with one of the provided prefixes, then
	 * the portion of the argument without the prefix is returned, which
	 * represents the option in question.
	 * 
	 * @param aArg The argument for which to retrieve the option's name.
	 * @param aPrefixes The prefixes to use when determining options.
	 * 
	 * @return Either the option's name or null if the argument does not
	 *         represent an option.
	 */
	protected static String toOption( String aArg, String... aPrefixes ) {
		// Make sure we have an order with the "longest" prefix first |-->
		Arrays.sort( aPrefixes, Collections.reverseOrder() );
		// Make sure we have an order with the "longest" prefix first <--|

		for ( String ePrefix : aPrefixes ) {
			if ( aArg.startsWith( ePrefix ) ) { return aArg.substring( ePrefix.length() ); }
		}
		return null;
	}

	@SafeVarargs
	protected static boolean containsDir( File aDir, List<File>... aDirs ) {
		for ( List<File> eDirs : aDirs ) {
			if ( eDirs.contains( aDir ) ) { return true; }
		}
		return false;
	}

	private static File toDirectory( String aDirPath ) {
		if ( aDirPath != null ) {
			File theDir = new File( aDirPath );
			if ( theDir.exists() && theDir.isDirectory() ) { return theDir; }
		}
		return null;
	}

	/**
	 * Determines whether the given {@link StackTraceElement} is not relevant
	 * determining you as the caller.
	 *
	 * @param aStackTraceElement The {@link StackTraceElement} to analyze
	 *        whether it is to be skipped or not.
	 * @return True in case you better skip this one as it does not seem to be
	 *         you.
	 */
	private static boolean isSkipStackTraceElement( StackTraceElement aStackTraceElement ) {
		return aStackTraceElement.getClassName().equals( Thread.class.getName() ) || aStackTraceElement.getClassName().equals( RuntimeUtility.class.getName() ) || aStackTraceElement.getLineNumber() <= 1;
	}
}