// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.runtime;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.concurrent.TimeUnit;

import org.refcodes.mixin.ResultAccessor;

/**
 * Retrieves the Result of the given process as {@link String}.
 */
public class ProcessResult implements ResultAccessor<String> {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private String _result;

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs a {@link ProcessResult} instance which will synchronously
	 * return the process' result via {@link #getResult()} (and
	 * {@link #toString()} accordingly).
	 * 
	 * @param aProcess The {@link Process} from which to retrieve the result.
	 * 
	 * @throws IOException Thrown in case there were problems retrieving the
	 *         result. If possible, the error stream of the process is captured
	 *         in the exception's message.
	 */
	public ProcessResult( Process aProcess ) throws IOException {
		try {
			aProcess.waitFor();
		}
		catch ( InterruptedException ignoree ) {}
		// while ( aProcess.isAlive() ) {
		// 	try {
		// 		Thread.sleep( LoopSleepTime.MIN.getMilliseconds() );
		// 	}
		// 	catch ( InterruptedException ignore ) {}
		// }
		_result = toResult( aProcess );
	}

	/**
	 * Constructs a {@link ProcessResult} instance which will synchronously
	 * return the process' result via {@link #getResult()} (and
	 * {@link #toString()} accordingly).
	 * 
	 * @param aProcess The {@link Process} from which to retrieve the result.
	 * @param aTimeOutMillis The time to wait till to terminate with an
	 *        {@link InterruptedException}.
	 * 
	 * @throws IOException Thrown in case there were problems retrieving the
	 *         result. If possible, the error stream of the process is captured
	 * 
	 * @throws InterruptedException Thrown in case the timeout has exceeded
	 *         before process' termination.
	 */
	public ProcessResult( Process aProcess, int aTimeOutMillis ) throws InterruptedException, IOException {
		if ( aTimeOutMillis != -1 ) {
			aProcess.waitFor( aTimeOutMillis, TimeUnit.MILLISECONDS );
			// double now = System.currentTimeMillis();
			// while ( aProcess.isAlive() && System.currentTimeMillis() < now + aTimeOutMillis ) {
			// 	Thread.sleep( LoopSleepTime.MIN.getMilliseconds() );
			// }
			if ( aProcess.isAlive() ) {
				// long thePid = aProcess.pid();
				try {
					aProcess.destroy();
				}
				catch ( Exception ignore ) {}
				throw new InterruptedException( "Killed process <" + aProcess.toString() + ">!" );
			}
		}
		else {
			aProcess.waitFor();
			// while ( aProcess.isAlive() ) {
			// 	Thread.sleep( LoopSleepTime.MIN.getMilliseconds() );
			// }
		}
		_result = toResult( aProcess );
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getResult() {
		return _result;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return _result;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	private String toResult( Process aProcess ) throws IOException {
		try {
			StringBuilder theOutBuilder = new StringBuilder();
			try (BufferedReader theOutReader = new BufferedReader( new InputStreamReader( aProcess.getInputStream() ) )) {
				String eLine;
				while ( (eLine = theOutReader.readLine()) != null ) {
					theOutBuilder.append( eLine );
				}
			}
			return theOutBuilder.toString();
		}
		catch ( IOException e ) {
			StringBuilder theErrBuilder = new StringBuilder();
			try (BufferedReader theErrReader = new BufferedReader( new InputStreamReader( aProcess.getInputStream() ) )) {
				if ( theErrReader != null ) {
					String eErr;
					while ( (eErr = theErrReader.readLine()) != null ) {
						theErrBuilder.append( eErr );
					}
					theErrReader.close();
				}
			}
			catch ( Exception ignore ) {}
			if ( theErrBuilder.length() != 0 ) {
				throw new IOException( theErrBuilder.toString(), e );
			}
			throw e;
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
