// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.runtime;

/**
 * Provides methods to set Session-Correlation behavior: When enabled via
 * {@link #setSessionCorrelation(boolean)} (or the like), then an existing
 * Session-Correlation-ID might be forwarded by setting it into the
 * {@link Correlation} singleton via {@link Correlation#SESSION}'s
 * {@link Correlation#setId(String)}. When being disabled, then no
 * Session-Correlation handling will be done, any existing
 * {@link Correlation#SESSION}'s {@link Correlation#getId()} may stay untouched!
 * 
 * @param <B> The builder to return in order to be able to apply multiple build
 *        operations.
 */
public interface SessionCorrelation<B extends SessionCorrelation<B>> {

	/**
	 * Enables or disables the automatic Session-Correlation mechanism.
	 * 
	 * @param hasSessionCorrelation When true then Session-Correlation is
	 *        enabled, else disabled.
	 */
	void setSessionCorrelation( boolean hasSessionCorrelation );

	/**
	 * Builder method for the {@link #setSessionCorrelation(boolean)} operation.
	 * 
	 * @param hasSessionCorrelation When true then Session-Correlation is
	 *        enabled, else disabled.
	 * 
	 * @return The builder for applying multiple build operations.
	 */
	B withSessionCorrelation( boolean hasSessionCorrelation );

	/**
	 * Enables the automatic Session-Correlation mechanism.
	 */
	default void enableSessionCorrelation() {
		setSessionCorrelation( true );
	}

	/**
	 * Builder method for the {@link #enableSessionCorrelation} operation.
	 * 
	 * @return The builder for applying multiple build operations.
	 */
	B withEnableSessionCorrelation();

	/**
	 * Disables the automatic Session-Correlation mechanism.
	 */
	default void disableSessionCorrelation() {
		setSessionCorrelation( false );
	}

	/**
	 * Builder method for the {@link #disableSessionCorrelation} operation.
	 * 
	 * @return The builder for applying multiple build operations.
	 */
	B withDisableSessionCorrelation();

	/**
	 * Determines whether Session-Correlation is being enabled or disabled.
	 * 
	 * @return True in case Session-Correlation is being enabled, else false.
	 */
	boolean hasSessionCorrelation();

}
