// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.runtime;

import java.io.IOException;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.refcodes.data.DaemonLoopSleepTime;
import org.refcodes.data.EnvironmentVariable;
import org.refcodes.data.Literal;
import org.refcodes.data.SystemProperty;

/**
 * Utility for acquiring system information on the machine this process is
 * running in.
 */
public final class SystemUtility {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	private static String _uname = null;
	private static boolean _hasUname = false;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Private constructor as of being utility class.
	 */
	private SystemUtility() {}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Determines the computer's name. First it tries to get it from the
	 * {@link InetAddress}, if it fails it tries to get it from the system's
	 * environment using the {@link EnvironmentVariable#COMPUTERNAME} (on
	 * Windows machines only) and if both fails, it returns the default
	 * {@link Literal#LOCALHOST} identifier.
	 * 
	 * @return The computer's name, as fallback, {@link Literal#LOCALHOST}
	 *         ("localhost") is returned.
	 */
	public static String getComputerName() {
		try {
			return InetAddress.getLocalHost().getHostName();
		}
		catch ( UnknownHostException e ) {
			String theName = EnvironmentVariable.COMPUTERNAME.getValue();
			if ( theName == null || theName.length() == 0 ) {
				theName = EnvironmentVariable.HOSTNAME.getValue();
			}
			if ( theName != null && theName.length() > 0 ) {
				return theName;
			}
		}
		return Literal.LOCALHOST.getName();
	}

	/**
	 * If on a *nix alike system, this method returns the output of the "uname
	 * -a" command: "uname" prints system information, "-a" instructs it to
	 * print all information.
	 * 
	 * @return The "uname -a" output or null if "uname" is not known.
	 */
	public static String getUname() {
		if ( _hasUname ) return _uname;
		try {
			_uname = exec( DaemonLoopSleepTime.NORM.getMillis(), "uname -a" );
		}
		catch ( IOException | InterruptedException ignore ) {
			//	try {
			//		_uname = exec( DaemonLoopSleepTime.NORM.getMilliseconds(), "bash", "-c", "uname -a" );
			//	}
			//	catch ( IOException | InterruptedException ignore2 ) {}
		}
		_hasUname = true;
		return _uname;
	}

	/**
	 * Gets the value for the provided properties, if non was found then the
	 * default value is taken. A {@link SystemProperty} elements wins over the
	 * {@link EnvironmentVariable} elements. The preceding
	 * {@link EnvironmentVariable} element wins over the succeeding
	 * {@link EnvironmentVariable} element. The default value is taken if non
	 * property had a value (a String with length &gt; 0).
	 * 
	 * @param aDefaultValue The default value to take when none other value was
	 *        set.
	 * @param aSystemProperty The system-property passed via <code>java
	 *        -D&lt;name&gt;=&lt;value&gt;</code>
	 * @param aEnvironmentProperties The properties looked for in the system's
	 *        environment variables.
	 * @return The best fitting value.
	 */
	public static String toPropertyValue( String aDefaultValue, SystemProperty aSystemProperty, EnvironmentVariable... aEnvironmentProperties ) {
		String theValue = aSystemProperty.getValue();
		if ( theValue != null && theValue.length() > 0 ) return theValue;
		if ( aEnvironmentProperties != null ) {
			for ( EnvironmentVariable eProperty : aEnvironmentProperties ) {
				theValue = eProperty.getValue();
				if ( theValue != null && theValue.length() > 0 ) return theValue;
			}
		}
		if ( theValue != null && theValue.length() > 0 ) return theValue;
		return aDefaultValue;
	}

	/**
	 * Gets the value for the provided properties, if non was found then null is
	 * returned. A {@link SystemProperty} elements wins over the
	 * {@link EnvironmentVariable} elements. The preceding
	 * {@link EnvironmentVariable} element wins over the succeeding
	 * {@link EnvironmentVariable} element. A null is taken if non property had
	 * a value (a String with length &gt; 0).
	 * 
	 * @param aSystemProperty The system-property passed via <code>java
	 *        -D&lt;name&gt;=&lt;value&gt;</code>
	 * @param aEnvironmentProperties The properties looked for in the system's
	 *        environment variables.
	 * @return The best fitting value or null if none was detected.
	 */
	public static String toPropertyValue( SystemProperty aSystemProperty, EnvironmentVariable... aEnvironmentProperties ) {
		String theValue = null;
		if ( aSystemProperty != null ) {
			theValue = aSystemProperty.getValue();
			if ( theValue != null && theValue.length() != 0 ) return theValue;
		}
		if ( aEnvironmentProperties != null ) {
			for ( EnvironmentVariable eProperty : aEnvironmentProperties ) {
				theValue = eProperty.getValue();
				if ( theValue != null && theValue.length() != 0 ) return theValue;
			}
		}
		return null;
	}

	/**
	 * Executes a command and returns the output.
	 *
	 * @param aTimeOutMillis The time in milliseconds to wait till the process
	 *        is killed when not terminated yet.
	 * @param aCommandLine the command
	 * 
	 * @return Null if execution failed, else the according output. An empty
	 *         {@link String} stands fur successful execution.
	 * @throws IOException in case there were problems executing the command.
	 * @throws InterruptedException thrown in case execution as been
	 *         interrupted.
	 */
	public static String exec( int aTimeOutMillis, String aCommandLine ) throws IOException, InterruptedException {
		// System.out.println( "Executing " + aCommandLine );
		ProcessBuilder theProcessBuilder = new ProcessBuilder( aCommandLine );
		/*
		 * theProcessBuilder.redirectError( ProcessBuilder.Redirect.INHERIT );
		 * theProcessBuilder.redirectOutput( ProcessBuilder.Redirect.INHERIT );
		 * theProcessBuilder.redirectInput( ProcessBuilder.Redirect.INHERIT );
		 * theProcessBuilder.inheritIO();
		 */
		Process theProcess = theProcessBuilder.start();
		// Process theProcess = Runtime.getRuntime().exec( aCommandLine );
		String theOut = new ProcessResult( theProcess, aTimeOutMillis ).toString();
		return theOut;
	}

	/**
	 * Executes a command and returns the output.
	 *
	 * @param aTimeOutMillis The time in milliseconds to wait till the process
	 *        is killed when not terminated yet.
	 * @param aCommandLine the command with the arguments to be passed to the
	 *        command.
	 * 
	 * @return Null if execution failed, else the according output. An empty
	 *         {@link String} stands fur successful execution.
	 * @throws IOException in case there were problems executing the command.
	 * @throws InterruptedException thrown in case execution as been
	 *         interrupted.
	 */
	public static String exec( int aTimeOutMillis, String... aCommandLine ) throws IOException, InterruptedException {
		// System.out.println( "Executing " + Arrays.toString( aCommandLine ) );
		ProcessBuilder theProcessBuilder = new ProcessBuilder( aCommandLine );
		/*
		 * theProcessBuilder.redirectError( ProcessBuilder.Redirect.INHERIT );
		 * theProcessBuilder.redirectOutput( ProcessBuilder.Redirect.INHERIT );
		 * theProcessBuilder.redirectInput( ProcessBuilder.Redirect.INHERIT );
		 * theProcessBuilder.inheritIO();
		 */
		Process theProcess = theProcessBuilder.start();
		// Process theProcess = Runtime.getRuntime().exec( aCommandLine );
		String theOut = new ProcessResult( theProcess, aTimeOutMillis ).toString();
		return theOut;
	}

	/**
	 * Executes a command and returns the output.
	 *
	 * @param aCommandLine the command
	 * @return Null if execution failed, else the according output. An empty
	 *         {@link String} stands fur successful execution.
	 * @throws IOException in case there were problems executing the command.
	 */
	public static String exec( String aCommandLine ) throws IOException {
		// System.out.println( "Executing " + aCommandLine );
		// Process theProcess = Runtime.getRuntime().exec( aCommandLine );
		ProcessBuilder theProcessBuilder = new ProcessBuilder( aCommandLine );
		// theProcessBuilder.redirectError( ProcessBuilder.Redirect.INHERIT );
		// theProcessBuilder.redirectOutput(ProcessBuilder.Redirect.INHERIT);
		// theProcessBuilder.inheritIO();
		Process theProcess = theProcessBuilder.start();
		String theOut = new ProcessResult( theProcess ).toString();
		return theOut;
	}

	/**
	 * Executes a command and returns the output.
	 *
	 * @param aCommandLine the command with the arguments to be passed to the
	 *        command.
	 * 
	 * @return Null if execution failed, else the according output. An empty
	 *         {@link String} stands fur successful execution.
	 * @throws IOException in case there were problems executing the command.
	 */
	public static String exec( String... aCommandLine ) throws IOException {
		// System.out.println( "Executing " + Arrays.toString( aCommandLine ) );
		// Process theProcess = Runtime.getRuntime().exec( aCommandLine );
		ProcessBuilder theProcessBuilder = new ProcessBuilder( aCommandLine );
		// theProcessBuilder.redirectError( ProcessBuilder.Redirect.INHERIT );
		// theProcessBuilder.redirectOutput(ProcessBuilder.Redirect.INHERIT);
		// theProcessBuilder.inheritIO();
		Process theProcess = theProcessBuilder.start();
		String theOut = new ProcessResult( theProcess ).toString();
		return theOut;
	}

	/**
	 * Gathers all available system information from this Artifac's point of
	 * view.
	 * 
	 * @return A {@link Map} containing the available information being
	 *         gathered.
	 */
	public static Map<String, String> toSystemInfo() {
		Map<String, String> theInfo = new HashMap<>();
		theInfo.put( "applicationId", SystemContext.APPLICATION.toContextId() + "" );
		theInfo.put( "applicationIds", Arrays.toString( SystemContext.APPLICATION.toContextIds( 3 ) ) );
		theInfo.put( "applicationString", Arrays.toString( SystemContext.APPLICATION.toContextSequence().getBytes() ) );
		theInfo.put( "computerName", getComputerName() );
		theInfo.put( "hostId", SystemContext.HOST.toContextId() + "" );
		theInfo.put( "hostIds", Arrays.toString( SystemContext.HOST.toContextIds( 3 ) ) );
		theInfo.put( "hostString", Arrays.toString( SystemContext.HOST.toContextSequence().getBytes() ) );

		try {
			theInfo.put( "ipAddress", Arrays.toString( toHostIpAddress() ) );
		}
		catch ( SocketException | UnknownHostException e ) {
			theInfo.put( "ipAddress", e.getMessage() );
		}
		theInfo.put( "isAnsiTerminalPreferred", Terminal.isAnsiTerminalPreferred() + "" );
		theInfo.put( "isAnsiTerminal", Terminal.isAnsiTerminal() + "" );
		theInfo.put( "isCygwinTerminal", Terminal.isCygwinTerminal() + "" );
		theInfo.put( "isLineBreakRequired", Terminal.isLineBreakRequired( Terminal.getTerminalWidth() ) + "" );
		theInfo.put( "launcherDir", RuntimeUtility.toLauncherDir().getAbsolutePath() );
		theInfo.put( "lineBreak", Terminal.getLineBreak().replaceAll( "\\n", "\\\\n" ).replaceAll( "\\r", "\\\\r" ) );
		try {
			theInfo.put( "macAddress", Arrays.toString( toHostMacAddress() ) );
		}
		catch ( SocketException | UnknownHostException e ) {
			theInfo.put( "macAddress", e.getMessage() );
		}
		theInfo.put( "mainClass", RuntimeUtility.getMainClass().getName() );
		theInfo.put( "operatingSystem", OperatingSystem.toOperatingSystem() + "" );
		theInfo.put( "preferredTerminalHeight", Terminal.toPreferredTerminalHeight() + "" );
		theInfo.put( "preferredTerminalWidth", Terminal.toPreferredTerminalWidth() + "" );
		theInfo.put( "shell", Shell.toShell() + "" );
		theInfo.put( "systemConsole", (System.console() != null ? System.console().toString() : null) );
		theInfo.put( "terminalEncoding", Terminal.getTerminalEncoding() );
		theInfo.put( "tempDir", SystemProperty.TEMP_DIR.getValue() );
		theInfo.put( "terminal", Terminal.toTerminal() + "" );
		theInfo.put( "terminalHeight", Terminal.getTerminalHeight() + "" );
		theInfo.put( "terminalWidth", Terminal.getTerminalWidth() + "" );
		theInfo.put( "uname", getUname() );
		theInfo.put( "userId", SystemContext.USER.toContextId() + "" );
		theInfo.put( "userIds", Arrays.toString( SystemContext.USER.toContextIds( 3 ) ) );
		theInfo.put( "userString", Arrays.toString( SystemContext.USER.toContextSequence().getBytes() ) );
		return theInfo;
	}

	/**
	 * Gathers all available system information from this Artifac's point of
	 * view. This method may rely on the output of {@link #toSystemInfo()}.
	 * 
	 * @return A {@link String} containing the available information being
	 *         gathered.
	 */
	public static String toPrettySystemInfo() {
		StringBuffer theBuffer = new StringBuffer();
		Map<String, String> theInfo = toSystemInfo();
		int maxLength = -1;
		for ( String eKey : theInfo.keySet() ) {
			if ( eKey.length() > maxLength ) maxLength = eKey.length();
		}
		List<String> theKeys = new ArrayList<>( theInfo.keySet() );
		Collections.sort( theKeys );
		String tmpKey;
		for ( String eKey : theKeys ) {
			tmpKey = eKey;
			while ( tmpKey.length() < maxLength ) {
				tmpKey = tmpKey + " ";
			}
			theBuffer.append( tmpKey + " = " + theInfo.get( eKey ) + Terminal.getLineBreak() );
		}
		return theBuffer.toString();
	}

	/**
	 * Tries to determine a no-localhost IP-Address for this machine. The best
	 * guess is returned. If none no-localhost address is found, then the
	 * localhost's IP-Address may be returned (as of
	 * {@link InetAddress#getLocalHost()}).
	 * 
	 * @return The best guest for a no-localhost IP-Address or as a fall back
	 *         the localhost's IP-Address (as of
	 *         {@link InetAddress#getLocalHost()} may be returned.
	 * @throws SocketException Thrown to indicate that accessing the network
	 *         interfaces caused a problem.
	 * @throws UnknownHostException Thrown to indicate that the IP address of
	 *         the local host could not be determined.
	 */
	public static byte[] toHostIpAddress() throws SocketException, UnknownHostException {
		Enumeration<NetworkInterface> eNetworks;
		eNetworks = NetworkInterface.getNetworkInterfaces();
		NetworkInterface eNetwork;
		Enumeration<InetAddress> eAddresses;
		InetAddress eAddress;
		while ( eNetworks.hasMoreElements() ) {
			eNetwork = eNetworks.nextElement();
			eAddresses = eNetwork.getInetAddresses();
			while ( eAddresses.hasMoreElements() ) {
				try {
					eAddress = eAddresses.nextElement();
					if ( !eAddress.isAnyLocalAddress() && !eAddress.isLoopbackAddress() ) {
						return eAddress.getAddress();
					}
				}
				catch ( IllegalArgumentException ignore ) {
					/* ignore */
				}
			}
		}
		return InetAddress.getLocalHost().getAddress();
	}

	/**
	 * Tries to determine the host Mac-Address for this machine. The best guess
	 * is returned.
	 * 
	 * @return The best guest for a Mac-Address is returned.
	 * @throws SocketException Thrown to indicate that accessing the network
	 *         interfaces caused a problem.
	 * @throws UnknownHostException Thrown to indicate that the IP address of
	 *         the local host could not be determined.
	 */
	public static byte[] toHostMacAddress() throws SocketException, UnknownHostException {
		Enumeration<NetworkInterface> eNetworks;
		eNetworks = NetworkInterface.getNetworkInterfaces();
		NetworkInterface eNetwork;
		byte[] eAddress;
		while ( eNetworks.hasMoreElements() ) {
			eNetwork = eNetworks.nextElement();
			eAddress = eNetwork.getHardwareAddress();
			if ( eAddress != null && eAddress.length != 0 ) {
				return eAddress;
			}
		}
		return NetworkInterface.getByInetAddress( InetAddress.getLocalHost() ).getHardwareAddress();
	}

	// /////////////////////////////////////////////////////////////////////////
	// IDENTIFIERS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////
}
