// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.runtime;

import java.io.OutputStreamWriter;
import java.util.function.IntSupplier;

import org.fusesource.jansi.internal.WindowsSupport;
import org.refcodes.data.ConsoleDimension;
import org.refcodes.data.DaemonLoopSleepTime;
import org.refcodes.data.EnvironmentVariable;
import org.refcodes.data.SystemProperty;
import org.refcodes.exception.BugException;
import org.refcodes.numerical.NumericalUtility;

/**
 * Enumeration with the (relevant) terminals as well as interpolated terminal
 * metrics.
 */
public enum Terminal {

	// TODO: For ANSI-Terminals this could do the trick. Found at;
	//
	// "https://stackoverflow.com/questions/1286461/can-i-find-the-console-width-with-java"
	//
	// There's a trick that you can use based on ANSI Escape Codes. They don't
	// provide a direct way to query the console size, but they do have a
	// command for requesting the current position size. By moving the cursor to
	// a really high row and column and then requesting the console size you can
	// get an accurate measurement.
	//
	// Send the following sequences to the terminal (stdout)
	//
	// "\u001b[s" // save cursor position
	// "\u001b[5000;5000H" // move to col 5000 row 5000
	// "\u001b[6n" // request cursor position
	// "\u001b[u" // restore cursor position
	//
	// Now watch stdin, you should receive a sequence that looks like
	// "\u001b[25;80R", where 25 is the row count, and 80 the columns.

	// /////////////////////////////////////////////////////////////////////////
	// IDENTIFIERS:
	// /////////////////////////////////////////////////////////////////////////

	XTERM, CYGWIN, UNKNOWN;

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	private static Boolean _isAnsiTerminal = null;
	private static Boolean _isCygwinTerminal = null;
	private static Integer _terminalHeight = null;
	private static Integer _terminalWidth = null;
	private static long _lastTerminalHeightUpdate = -1;
	private static long _lastTerminalWidthUpdate = -1;
	private static boolean _canTput = true;
	private static boolean _canBashTput = true;
	private static boolean _canModeCon = true;
	private static boolean _canCmdModeCon = true;
	private static boolean _canPowershell = true;
	private static boolean _canStty = true;
	private static boolean _canBashStty = true;
	private static boolean _canNativeWindows = true;
	private static boolean _canBashTputExe = true;
	private static boolean _canTputExe = true;
	private static boolean _canBashSttyExe = true;
	private static boolean _canSttyExe = true;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final int WIDTH_ABOVE_UNREALSITIC = 1024;
	private static final int HEIGHT_ABOVE_UNREALSITIC = 256;
	private static final int PRESET_TERMINAL_HEIGHT = 25;
	private static final int PRESET_TERMINAL_WIDTH = 80;
	private static final int PRESET_WINCMD_WIDTH = 120;

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Extracts the height from an "ANSICON" environment variable's value. The
	 * "ANSICON" formatted variable looks something like this: "91x19999
	 * (91x51)").
	 * 
	 * @return The determined height or -1 if we have no such variable.
	 */
	private static int getAnsiconHeight() {
		try {
			return toAnsiconHeight( EnvironmentVariable.ANSICON.getValue() );
		}
		catch ( NumberFormatException ignore ) {}
		return -1;
	}

	/**
	 * Extracts the width from an "ANSICON" environment variable's value. The
	 * "ANSICON" formatted variable looks something like this: "91x19999
	 * (91x51)").
	 * 
	 * @return The determined width or -1 if we have no such variable.
	 */
	private static int getAnsiconWidth() {
		try {
			return toAnsiconWidth( EnvironmentVariable.ANSICON.getValue() );
		}
		catch ( NumberFormatException ignore ) {}
		return -1;
	}

	/**
	 * Determines the current height in characters of the system's terminal in
	 * use. This operation is expensive as it always does a full blown height
	 * calculation. To be less expensive, either use
	 * {@link #getTerminalHeight()} or {@link #getTerminalHeight(int)}.
	 * 
	 * @return The height of the terminal in characters or -1 if the height
	 *         cannot be determined.
	 */
	public static int getCurrentTerminalHeight() {
		return getTerminalHeight( 0 );
	}

	/**
	 * Determines the current width in characters of the system's terminal in
	 * use. This operation is expensive as it always does a full blown width
	 * calculation. To be less expensive, either use {@link #getTerminalWidth()}
	 * or {@link #getTerminalWidth(int)}.
	 * 
	 * @return The height of the terminal in characters or -1 if the height
	 *         cannot be determined.
	 */
	public static int getCurrentTerminalWidth() {
		return getTerminalWidth( 0 );
	}

	/**
	 * Determines the operating system as of
	 * {@link OperatingSystem#toOperatingSystem()} and in case a
	 * {@link OperatingSystem#WINDOWS} is being detected, then \r\n" (CRLF) is
	 * returned, else "\n" (LF) is returned.
	 * 
	 * Can be overridden with the {@link SystemProperty#CONSOLE_LINE_BREAK}
	 * (<code>java -Dconsole.lineBreak=...</code>) and the
	 * {@link EnvironmentVariable#CONSOLE_LINE_BREAK}
	 * ("<code>export CONSOLE_LINE_BREAK=...</code>).
	 * 
	 * @return The operating system's specific line break; on Windows it is
	 *         "\r\n" (CRLF) and on all other operating systems it is "\n" (LF).
	 */
	public static String getLineBreak() {
		String theLineBreak = SystemProperty.CONSOLE_LINE_BREAK.getValue();
		if ( theLineBreak == null || theLineBreak.length() == 0 ) {
			theLineBreak = EnvironmentVariable.CONSOLE_LINE_BREAK.getValue();
			if ( theLineBreak == null || theLineBreak.length() == 0 ) {
				theLineBreak = System.lineSeparator();
			}
		}
		if ( theLineBreak == null || theLineBreak.length() == 0 ) {
			theLineBreak = OperatingSystem.toOperatingSystem() == OperatingSystem.WINDOWS ? "\r\n" : "\n";
		}
		return theLineBreak;
	}

	private static int getModeConHeight() {
		if ( _canModeCon ) {
			try {
				String theRersult = SystemUtility.exec( DaemonLoopSleepTime.MIN.getMillis(), "mode.com con" );
				return toModeConHeight( theRersult );
			}
			catch ( Exception ignore ) {
				_canModeCon = false;
			}
		}

		if ( _canCmdModeCon ) {
			try {

				String theRersult = SystemUtility.exec( DaemonLoopSleepTime.MIN.getMillis(), "cmd.exe", "/c", "mode con" );
				return toModeConHeight( theRersult );
			}
			catch ( Exception ignore ) {
				_canCmdModeCon = false;
			}
		}

		return -1;
	}

	private static int getModeConWidth() {
		if ( _canModeCon ) {
			try {
				String theRersult = SystemUtility.exec( DaemonLoopSleepTime.MIN.getMillis(), "mode.com con" );
				return toModeConWidth( theRersult );
			}
			catch ( Exception ignore ) {
				_canModeCon = false;
			}
		}

		if ( _canCmdModeCon ) {
			try {
				String theRersult = SystemUtility.exec( DaemonLoopSleepTime.MIN.getMillis(), "cmd.exe", "/c", "mode con" );
				return toModeConWidth( theRersult );
			}
			catch ( Exception ignore ) {
				_canCmdModeCon = false;
			}
		}
		return -1;
	}

	private static int getNativeWindowsHeight() {
		if ( _canNativeWindows ) {
			try {
				return WindowsSupport.getWindowsTerminalHeight();
				// return TerminalBuilder.terminal().getHeight();
			}
			catch ( Error e ) {
				_canNativeWindows = false;
			}
		}
		return -1;
	}

	private static int getNativeWindowsWidth() {
		if ( _canNativeWindows ) {
			try {
				return WindowsSupport.getWindowsTerminalWidth();
				// return TerminalBuilder.terminal().getWidth();
			}
			catch ( Error e ) {
				_canNativeWindows = false;
			}
		}
		return -1;
	}

	private static int getPowershellHeight() {
		if ( _canPowershell ) {
			try {
				String theRersult = SystemUtility.exec( DaemonLoopSleepTime.MIN.getMillis(), "powershell.exe", "/c", "$host.UI.RawUI.WindowSize.Height" );
				if ( theRersult != null ) {
					return Integer.parseInt( theRersult.trim() );
				}
			}
			catch ( Exception ignore ) {
				_canPowershell = false;
			}
		}
		return -1;
	}

	private static int getPowershellWidth() {
		if ( _canPowershell ) {
			try {
				String theRersult = SystemUtility.exec( DaemonLoopSleepTime.MIN.getMillis(), "powershell.exe", "/c", "$host.UI.RawUI.WindowSize.Width" );
				if ( theRersult != null ) {
					return Integer.parseInt( theRersult.trim() );
				}
			}
			catch ( Exception ignore ) {
				_canPowershell = false;
			}
		}
		return -1;
	}

	private static int getSttyExeHeight() {
		if ( _canSttyExe ) {
			try {
				String theRersult = SystemUtility.exec( DaemonLoopSleepTime.MIN.getMillis(), "stty.exe size" );
				if ( theRersult != null ) {
					String[] theSize = theRersult.trim().split( " " );
					if ( theSize.length == 2 ) {
						return Integer.parseInt( theSize[0] );
					}
				}

			}
			catch ( Exception ignore ) {
				_canSttyExe = false;
			}
		}

		if ( _canBashSttyExe ) {
			try {
				String theRersult = SystemUtility.exec( DaemonLoopSleepTime.MIN.getMillis(), "bash.exe", "-c", "stty size" );
				if ( theRersult != null ) {
					String[] theSize = theRersult.trim().split( " " );
					if ( theSize.length == 2 ) {
						return Integer.parseInt( theSize[0] );
					}
				}
			}
			catch ( Exception ignore ) {
				_canBashSttyExe = false;
			}
		}

		return -1;
	}

	private static int getSttyExeWidth() {
		if ( _canSttyExe ) {
			try {
				String theRersult = SystemUtility.exec( DaemonLoopSleepTime.MIN.getMillis(), "stty.exe size" );
				if ( theRersult != null ) {
					String[] theSize = theRersult.trim().split( " " );
					if ( theSize.length == 2 ) {
						return Integer.parseInt( theSize[1] );
					}
				}

			}
			catch ( Exception ignore ) {
				_canSttyExe = false;
			}
		}

		if ( _canBashSttyExe ) {
			try {
				String theRersult = SystemUtility.exec( DaemonLoopSleepTime.MIN.getMillis(), "bash.exe", "-c", "stty size" );
				if ( theRersult != null ) {
					String[] theSize = theRersult.trim().split( " " );
					if ( theSize.length == 2 ) {
						return Integer.parseInt( theSize[1] );
					}
				}
			}
			catch ( Exception ignore ) {
				_canBashSttyExe = false;
			}
		}

		return -1;
	}

	private static int getSttyHeight() {
		if ( _canStty ) {
			try {
				String theRersult = SystemUtility.exec( DaemonLoopSleepTime.MIN.getMillis(), "stty size" );
				if ( theRersult != null ) {
					String[] theSize = theRersult.trim().split( " " );
					if ( theSize.length == 2 ) {
						return Integer.parseInt( theSize[0] );
					}
				}

			}
			catch ( Exception ignore ) {
				_canStty = false;
			}
		}

		if ( _canBashStty ) {
			try {
				String theRersult = SystemUtility.exec( DaemonLoopSleepTime.MIN.getMillis(), "bash", "-c", "stty size" );
				if ( theRersult != null ) {
					String[] theSize = theRersult.trim().split( " " );
					if ( theSize.length == 2 ) {
						return Integer.parseInt( theSize[0] );
					}
				}
			}
			catch ( Exception ignore ) {
				_canBashStty = false;
			}
		}

		return -1;
	}

	private static int getSttyWidth() {
		if ( _canStty ) {
			try {
				String theRersult = SystemUtility.exec( DaemonLoopSleepTime.MIN.getMillis(), "stty size" );
				if ( theRersult != null ) {
					String[] theSize = theRersult.trim().split( " " );
					if ( theSize.length == 2 ) {
						return Integer.parseInt( theSize[1] );
					}
				}

			}
			catch ( Exception ignore ) {
				_canStty = false;
			}
		}

		if ( _canBashStty ) {
			try {
				String theRersult = SystemUtility.exec( DaemonLoopSleepTime.MIN.getMillis(), "bash", "-c", "stty size" );
				if ( theRersult != null ) {
					String[] theSize = theRersult.trim().split( " " );
					if ( theSize.length == 2 ) {
						return Integer.parseInt( theSize[1] );
					}
				}
			}
			catch ( Exception ignore ) {
				_canBashStty = false;
			}
		}

		return -1;
	}

	private static int getTerminalColumns() {
		String theResult = EnvironmentVariable.TERMINAL_COLUMNS.getValue();
		if ( theResult != null ) {
			try {
				return Integer.valueOf( theResult.toString() );
			}
			catch ( NumberFormatException ignore ) {}
		}
		return -1;
	}

	/**
	 * Determines the encoding of the system (system's console) in use as of
	 * {@link System#out}.
	 * 
	 * @return The encoding (of the console).
	 */
	public static String getTerminalEncoding() {
		OutputStreamWriter theOutWriter = new OutputStreamWriter( System.out );
		return theOutWriter.getEncoding();
	}

	/**
	 * Determines the height in characters of the system's terminal in use.
	 * 
	 * @return The height of the terminal in characters or -1 if the height
	 *         cannot be determined.
	 */
	public static int getTerminalHeight() {
		return getTerminalHeight( -1 );
	}

	/**
	 * Determines the height in characters of the system's terminal in use.
	 * 
	 * @param aValidityTimeMillis The time period till last determination of
	 *        this value is valid, as determining this value is expensive.
	 * 
	 * @return The height of the terminal in characters or -1 if the height
	 *         cannot be determined.
	 */
	public static int getTerminalHeight( int aValidityTimeMillis ) {

		// Use last update? |-->
		if ( _terminalHeight != null && aValidityTimeMillis != 0 ) {
			if ( aValidityTimeMillis == -1 || _lastTerminalHeightUpdate + aValidityTimeMillis > System.currentTimeMillis() ) return _terminalHeight;
		}
		// Use last update? <--|

		int theTerminalHeight = -1;

		// ANSICON |-->
		theTerminalHeight = toTerminalHeight( theTerminalHeight, Terminal::getAnsiconHeight );
		// ANSICON <--|

		// LINES |-->
		theTerminalHeight = toTerminalHeight( theTerminalHeight, Terminal::getTerminalLines );
		// LINES <--|

		// Native |-->
		theTerminalHeight = toTerminalHeight( theTerminalHeight, Terminal::getNativeWindowsHeight );
		// Native <--|

		if ( Terminal.isCygwinTerminal() ) {
			theTerminalHeight = toTerminalHeight( theTerminalHeight, Terminal::getNativeWindowsHeight );
			theTerminalHeight = toTerminalHeight( theTerminalHeight, Terminal::getTputExeHeight );
			theTerminalHeight = toTerminalHeight( theTerminalHeight, Terminal::getSttyExeHeight );
			theTerminalHeight = toTerminalHeight( theTerminalHeight, Terminal::getPowershellHeight );
			theTerminalHeight = toTerminalHeight( theTerminalHeight, Terminal::getModeConHeight );
		}

		OperatingSystem theOs = OperatingSystem.toOperatingSystem();
		switch ( theOs ) {
		case MAC:
			theTerminalHeight = toTerminalHeight( theTerminalHeight, Terminal::getTputHeight );
			theTerminalHeight = toTerminalHeight( theTerminalHeight, Terminal::getSttyHeight );
			break;
		case UNIX:
			theTerminalHeight = toTerminalHeight( theTerminalHeight, Terminal::getTputHeight );
			theTerminalHeight = toTerminalHeight( theTerminalHeight, Terminal::getSttyHeight );
			break;
		case UNKNOWN:
			theTerminalHeight = toTerminalHeight( theTerminalHeight, Terminal::getTputHeight );
			theTerminalHeight = toTerminalHeight( theTerminalHeight, Terminal::getTputExeHeight );
			theTerminalHeight = toTerminalHeight( theTerminalHeight, Terminal::getSttyHeight );
			theTerminalHeight = toTerminalHeight( theTerminalHeight, Terminal::getSttyExeHeight );
			theTerminalHeight = toTerminalHeight( theTerminalHeight, Terminal::getPowershellHeight );
			theTerminalHeight = toTerminalHeight( theTerminalHeight, Terminal::getModeConHeight );
			break;
		case WINDOWS:
			theTerminalHeight = toTerminalHeight( theTerminalHeight, Terminal::getPowershellHeight );
			theTerminalHeight = toTerminalHeight( theTerminalHeight, Terminal::getModeConHeight );
			break;
		default:
			throw new BugException( "Missing case statement for <" + theOs + "> in implementation!" );
		}

		if ( isUnrealsiticHeight( theTerminalHeight ) ) {
			theTerminalHeight = -1;
		}

		_terminalHeight = theTerminalHeight;
		_lastTerminalHeightUpdate = System.currentTimeMillis();
		return theTerminalHeight;
	}

	private static int getTerminalLines() {
		String theResult = EnvironmentVariable.TERMINAL_LINES.getValue();
		if ( theResult != null ) {
			try {
				return Integer.valueOf( theResult.toString() );
			}
			catch ( NumberFormatException ignore ) {}
		}
		return -1;
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Determines the width in characters of the system's terminal in use.
	 * 
	 * @return The width of the terminal in characters or -1 if the width cannot
	 *         be determined.
	 */
	public static int getTerminalWidth() {
		return getTerminalWidth( -1 );
	}

	/**
	 * Determines the width in characters of the system's terminal in use.
	 * 
	 * @param aValidityTimeMillis The time period till last determination of
	 *        this value is valid, as determining this value is expensive.
	 * 
	 * @return The width of the terminal in characters or -1 if the width cannot
	 *         be determined.
	 */
	public static int getTerminalWidth( int aValidityTimeMillis ) {

		// Use last update? |-->
		if ( _terminalWidth != null && aValidityTimeMillis != 0 ) {
			if ( aValidityTimeMillis == -1 || _lastTerminalWidthUpdate + aValidityTimeMillis > System.currentTimeMillis() ) return _terminalWidth;
		}
		// Use last update? <--|

		int theTerminalWidth = -1;

		// ANSICON |-->
		theTerminalWidth = toTerminalWidth( theTerminalWidth, Terminal::getAnsiconWidth );
		// ANSICON <--|
		// COLUMNS |-->
		theTerminalWidth = toTerminalWidth( theTerminalWidth, Terminal::getTerminalColumns );
		// COLUMNS <--|

		// Native |-->
		theTerminalWidth = toTerminalWidth( theTerminalWidth, Terminal::getNativeWindowsWidth );
		// Native <--|

		if ( Terminal.isCygwinTerminal() ) {
			theTerminalWidth = toTerminalWidth( theTerminalWidth, Terminal::getTputExeWidth );
			theTerminalWidth = toTerminalWidth( theTerminalWidth, Terminal::getSttyExeWidth );
			theTerminalWidth = toTerminalWidth( theTerminalWidth, Terminal::getPowershellWidth );
			theTerminalWidth = toTerminalWidth( theTerminalWidth, Terminal::getModeConWidth );
		}

		OperatingSystem theOs = OperatingSystem.toOperatingSystem();
		switch ( theOs ) {
		case MAC:
			theTerminalWidth = toTerminalWidth( theTerminalWidth, Terminal::getTputWidth );
			theTerminalWidth = toTerminalWidth( theTerminalWidth, Terminal::getSttyWidth );
			break;
		case UNIX:
			theTerminalWidth = toTerminalWidth( theTerminalWidth, Terminal::getTputWidth );
			theTerminalWidth = toTerminalWidth( theTerminalWidth, Terminal::getSttyWidth );
			break;
		case UNKNOWN:
			theTerminalWidth = toTerminalWidth( theTerminalWidth, Terminal::getTputWidth );
			theTerminalWidth = toTerminalWidth( theTerminalWidth, Terminal::getTputExeWidth );
			theTerminalWidth = toTerminalWidth( theTerminalWidth, Terminal::getSttyWidth );
			theTerminalWidth = toTerminalWidth( theTerminalWidth, Terminal::getSttyExeWidth );
			theTerminalWidth = toTerminalWidth( theTerminalWidth, Terminal::getPowershellWidth );
			theTerminalWidth = toTerminalWidth( theTerminalWidth, Terminal::getModeConWidth );
			break;
		case WINDOWS:
			theTerminalWidth = toTerminalWidth( theTerminalWidth, Terminal::getPowershellWidth );
			theTerminalWidth = toTerminalWidth( theTerminalWidth, Terminal::getModeConWidth );
			break;
		default:
			throw new BugException( "Missing case statement for <" + theOs + "> in implementation!" );
		}

		if ( isUnrealsiticWidth( theTerminalWidth ) ) {
			theTerminalWidth = -1;
		}

		_terminalWidth = theTerminalWidth;
		_lastTerminalWidthUpdate = System.currentTimeMillis();
		return theTerminalWidth;
	}

	private static int getTputExeHeight() {
		if ( _canTputExe ) {
			try {
				String theRersult = SystemUtility.exec( DaemonLoopSleepTime.MIN.getMillis(), "tput.exe lines" );
				if ( theRersult != null ) {
					return Integer.parseInt( theRersult.trim() );
				}
			}
			catch ( Exception ignore ) {
				_canTputExe = false;
			}
		}

		if ( _canBashTputExe ) {
			try {
				String theRersult = SystemUtility.exec( DaemonLoopSleepTime.MIN.getMillis(), "bash.exe", "-c", "tput lines" );
				if ( theRersult != null ) {
					return Integer.parseInt( theRersult.trim() );
				}
			}
			catch ( Exception ignore ) {
				_canBashTputExe = false;
			}
		}

		return -1;
	}

	private static int getTputExeWidth() {
		if ( _canTputExe ) {
			try {
				String theRersult = SystemUtility.exec( DaemonLoopSleepTime.MIN.getMillis(), "tput.exe cols" );
				if ( theRersult != null ) {
					return Integer.parseInt( theRersult.trim() );
				}
			}
			catch ( Exception ignore ) {
				_canTputExe = false;
			}
		}

		if ( _canBashTputExe ) {
			try {
				String theRersult = SystemUtility.exec( DaemonLoopSleepTime.MIN.getMillis(), "bash.exe", "-c", "tput cols" );
				if ( theRersult != null ) {
					return Integer.parseInt( theRersult.trim() );
				}
			}
			catch ( Exception ignore ) {
				_canBashTputExe = false;
			}
		}

		return -1;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	private static int getTputHeight() {
		if ( _canTput ) {
			try {
				String theRersult = SystemUtility.exec( DaemonLoopSleepTime.MIN.getMillis(), "tput lines" );
				if ( theRersult != null ) {
					return Integer.parseInt( theRersult.trim() );
				}
			}
			catch ( Exception ignore ) {
				_canTput = false;
			}
		}

		if ( _canBashTput ) {
			try {
				String theRersult = SystemUtility.exec( DaemonLoopSleepTime.MIN.getMillis(), "bash", "-c", "tput lines" );
				if ( theRersult != null ) {
					return Integer.parseInt( theRersult.trim() );
				}
			}
			catch ( Exception ignore ) {
				_canBashTput = false;
			}
		}

		return -1;
	}

	private static int getTputWidth() {
		if ( _canTput ) {
			try {
				String theRersult = SystemUtility.exec( DaemonLoopSleepTime.MIN.getMillis(), "tput cols" );
				if ( theRersult != null ) {
					return Integer.parseInt( theRersult.trim() );
				}
			}
			catch ( Exception ignore ) {
				_canTput = false;
			}
		}

		if ( _canBashTput ) {
			try {
				String theRersult = SystemUtility.exec( DaemonLoopSleepTime.MIN.getMillis(), "bash", "-c", "tput cols" );
				if ( theRersult != null ) {
					return Integer.parseInt( theRersult.trim() );
				}
			}
			catch ( Exception ignore ) {
				_canBashTput = false;
			}
		}

		return -1;
	}

	/**
	 * Determines whether ANSI escape sequences are supported by the terminal.
	 * 
	 * @return True in case ANSI escape sequences are supported, else false.
	 */
	public static boolean isAnsiTerminal() {

		if ( _isAnsiTerminal != null ) return _isAnsiTerminal;

		if ( EnvironmentVariable.ANSICON.getValue() != null && EnvironmentVariable.ANSICON.getValue().length() != 0 ) {
			_isAnsiTerminal = true;
		}
		else if ( isCygwinTerminal() ) {
			_isAnsiTerminal = true;
		}
		else if ( Shell.toShell() == Shell.POWER_SHELL ) {
			_isAnsiTerminal = true;
		}
		else if ( Shell.toShell() == Shell.WIN_CMD ) {
			_isAnsiTerminal = true;
		}
		else {
			if ( _canTput ) {
				try {
					String theRersult = SystemUtility.exec( DaemonLoopSleepTime.MIN.getMillis(), "tput colors" );
					if ( theRersult != null ) {
						int theColors = Integer.parseInt( theRersult.trim() );
						if ( theColors > 4 ) {
							_isAnsiTerminal = true;
						}
					}
				}
				catch ( Exception ignore ) {
					_canTput = false;
				}
			}

			if ( _isAnsiTerminal == null ) {
				if ( _canBashTput ) {
					try {
						String theRersult = SystemUtility.exec( DaemonLoopSleepTime.MIN.getMillis(), "bash", "-c", "tput colors" );
						if ( theRersult != null ) {
							int theColors = Integer.parseInt( theRersult.trim() );
							if ( theColors > 4 ) {
								_isAnsiTerminal = true;
							}
						}
					}
					catch ( Exception ignore ) {
						_canBashTput = false;
					}
				}
			}

		}
		if ( _isAnsiTerminal == null ) {
			_isAnsiTerminal = false;
		}

		return _isAnsiTerminal;
	}

	/**
	 * Determines whether ANSI escape sequences are forced to be supported or
	 * not by REFCODES.ORG artifacts. This overrules {@link #isAnsiTerminal()}
	 * for the REFCODES.ORG artifacts. Various settings such as
	 * {@link SystemProperty#CONSOLE_ANSI} and (in this order)
	 * {@link EnvironmentVariable#CONSOLE_ANSI} overrule
	 * {@link #isAnsiTerminal()}.
	 * 
	 * @return True in case ANSI escape sequences are forced to be used by
	 *         REFCODES.ORG artifacts or not.
	 */
	public static boolean isAnsiTerminalPreferred() {
		String theResult = SystemUtility.toPropertyValue( SystemProperty.CONSOLE_ANSI, EnvironmentVariable.CONSOLE_ANSI, EnvironmentVariable.CONSOLE_CONEMU_ANSI );
		if ( theResult != null ) {
			try {
				return NumericalUtility.toBoolean( theResult.toString() );
			}
			catch ( IllegalArgumentException e ) {}
		}
		return isAnsiTerminal();
	}

	/**
	 * Determines whether wee need an explicit line-break for the given width on
	 * the current operating system and used terminal. E.g. on "win.cmd" we must
	 * not use a line-break in case our line is a s long as the console's width.
	 * There are some more such cases which this method tries to take into
	 * consideration. This operation is expensive as it always does a full blown
	 * width calculation. To be less expensive, either use
	 * {@link #isLineBreakRequired(int)} or
	 * {@link #isLineBreakRequired(int, int)}.
	 * 
	 * @param aRowWidth The row width you want to use when printing out to a
	 *        console.
	 * 
	 * @return True in case you should use a line-break.
	 */
	public static boolean isCurrentlyLineBreakRequired( int aRowWidth ) {
		return isLineBreakRequired( aRowWidth, 0 );
	}

	/**
	 * Tries to determine whether the command line interpreter (CLI) is a Cygwin
	 * one.
	 * 
	 * @return True in case we think we are running in Cygwin. Use
	 *         {@link #getCommandLineInterpreter()} to test for the type of CLI,
	 *         in case you got to distinguish the {@link Shell#SHELL} type, then
	 *         use this {@link #isCygwinTerminal()} method.
	 */
	static boolean isCygwinTerminal() {

		if ( _isCygwinTerminal != null ) {
			return _isCygwinTerminal;
		}

		_isCygwinTerminal = false;
		// if ( OperatingSystem.toOperatingSystem() == OperatingSystem.WINDOWS ) {
		if ( "cygwin".equalsIgnoreCase( EnvironmentVariable.TERM.getValue() ) ) {
			_isCygwinTerminal = true;
		}
		// String theUname = SystemUtility.getUname();
		// if ( theUname != null && theUname.toLowerCase().indexOf( "cygwin"
		// ) != -1 ) {
		// // "PWD" is only set by cygwin, not in CMD.EXE:
		// String thePwd = System.getenv( "PWD" );
		// _isCygwinTerminal = (thePwd != null && thePwd.length() != 0);
		// }

		// }
		return _isCygwinTerminal;
	}

	/**
	 * Determines whether wee need an explicit line-break for the given width on
	 * the current operating system and used terminal. E.g. on "win.cmd" we must
	 * not use a line-break in case our line is a s long as the console's width.
	 * There are some more such cases which this method tries to take into
	 * consideration.
	 * 
	 * @param aRowWidth The row width you want to use when printing out to a
	 *        console.
	 * 
	 * @return True in case you should use a line-break.
	 */
	public static boolean isLineBreakRequired( int aRowWidth ) {
		return isLineBreakRequired( aRowWidth, -1 );
	}

	/**
	 * Determines whether wee need an explicit line-break for the given width on
	 * the current operating system and used terminal. E.g. on "win.cmd" we must
	 * not use a line-break in case our line is a s long as the console's width.
	 * There are some more such cases which this method tries to take into
	 * consideration.
	 * 
	 * @param aRowWidth The row width you want to use when printing out to a
	 *        console.
	 * 
	 * @param aValidityTimeMillis The time period till last determination of
	 *        this value is valid, as determining this value is expensive.
	 * 
	 * @return True in case you should use a line-break.
	 */
	public static boolean isLineBreakRequired( int aRowWidth, int aValidityTimeMillis ) {

		// Heurisitc |-->
		if ( RuntimeUtility.isUnderTest() && (aRowWidth == PRESET_TERMINAL_WIDTH || aRowWidth == PRESET_WINCMD_WIDTH) ) return true;
		// Heurisitc <--|

		if ( Shell.toShell() == Shell.NONE ) {
			return true;
		}
		int theWidth = getTerminalWidth( aValidityTimeMillis );
		// if ( (Shell.toShell() == Shell.WIN_CMD || Shell.toShell() ==
		// Shell.POWER_SHELL || isCygwinTerminal()) && getTerminalHeight(
		// aValidityTimeMillis ) > 0 && theWidth != PRESET_TERMINAL_WIDTH &&
		// aRowWidth == theWidth ) {
		if ( (OperatingSystem.toOperatingSystem() == OperatingSystem.WINDOWS) && getTerminalHeight( aValidityTimeMillis ) > 0 && aRowWidth == theWidth ) {
			return false;
		}
		return true;
	}

	/**
	 * Determines whether the given terminal height is to be reconsidered.
	 * 
	 * @param eTerminalHeight The calculated height of the terminal.
	 * @return True in case that height is to be reconsidered.
	 */
	private static boolean isReconsiderableHeight( int eTerminalHeight ) {
		return (eTerminalHeight <= 1 || eTerminalHeight > HEIGHT_ABOVE_UNREALSITIC || eTerminalHeight == PRESET_TERMINAL_HEIGHT);
	}

	/**
	 * Determines whether the given terminal width is to be reconsidered.
	 * 
	 * @param eTerminalWidth The calculated width of the terminal.
	 * @return True in case that width is to be reconsidered.
	 */
	private static boolean isReconsiderableWidth( int eTerminalWidth ) {
		return (eTerminalWidth <= 1 || eTerminalWidth > WIDTH_ABOVE_UNREALSITIC || eTerminalWidth == PRESET_TERMINAL_WIDTH);
	}

	// private static org.jline.terminal.Terminal getTerminal() throws
	// IOException {
	// try {
	// if ( _terminal == null ) {
	// synchronized ( SystemUtility.class ) {
	// if ( _terminal == null ) {
	// _terminal = TerminalBuilder.builder().streams( System.in, System.out
	// ).build();
	// }
	// }
	// }
	// return _terminal;
	// }
	// catch ( IOException e ) {
	// throw e;
	// }
	// catch ( Exception e ) {
	// throw new IOException( "Cannot instantiate Terminal:" + e.getMessage(), e
	// );
	// }
	// }

	/**
	 * Determines whether the given terminal height is to be reconsidered.
	 * 
	 * @param eTerminalHeight The calculated height of the terminal.
	 * @return True in case that height is to be reconsidered.
	 */
	private static boolean isUnrealsiticHeight( int eTerminalHeight ) {
		return (eTerminalHeight <= 1 || eTerminalHeight > HEIGHT_ABOVE_UNREALSITIC);
	}

	/**
	 * Determines whether the given terminal width is to be reconsidered.
	 * 
	 * @param eTerminalWidth The calculated width of the terminal.
	 * @return True in case that width is to be reconsidered.
	 */
	private static boolean isUnrealsiticWidth( int eTerminalWidth ) {
		return (eTerminalWidth <= 1 || eTerminalWidth > WIDTH_ABOVE_UNREALSITIC);
	}

	/**
	 * Extracts the height from an "ANSICON" environment variable's value.
	 * 
	 * @param aAnsiconEnvVarValue The "ANSICON" formatted variable (e.g.
	 *        "91x19999 (91x51)").
	 * 
	 * @return The determined height.
	 * 
	 * @throws NumberFormatException Thrown in case no width was determinable
	 *         from the provided value.
	 */
	static int toAnsiconHeight( String aAnsiconEnvVarValue ) throws NumberFormatException {
		if ( aAnsiconEnvVarValue != null ) {
			int index = aAnsiconEnvVarValue.lastIndexOf( "x" );
			if ( index != -1 && aAnsiconEnvVarValue.length() > index + 1 ) {
				aAnsiconEnvVarValue = aAnsiconEnvVarValue.substring( index + 1 );
				if ( aAnsiconEnvVarValue.length() != 0 && aAnsiconEnvVarValue.endsWith( ")" ) ) {
					aAnsiconEnvVarValue = aAnsiconEnvVarValue.substring( 0, aAnsiconEnvVarValue.length() - 1 );
				}
				return Integer.valueOf( aAnsiconEnvVarValue.toString() );
			}
		}
		throw new NumberFormatException( "Cannot extract width from ANSICON formatted value <" + aAnsiconEnvVarValue + ">." );
	}

	/**
	 * Extracts the width from an "ANSICON" environment variable's value.
	 * 
	 * @param aAnsiconEnvVarValue The "ANSICON" formatted variable (e.g.
	 *        "91x19999 (91x51)").
	 * 
	 * @return The determined width.
	 * 
	 * @throws NumberFormatException Thrown in case no width was determinable
	 *         from the provided value.
	 */
	static int toAnsiconWidth( String aAnsiconEnvVarValue ) throws NumberFormatException {
		if ( aAnsiconEnvVarValue != null ) {
			int index = aAnsiconEnvVarValue.indexOf( "x" );
			if ( index != -1 ) {
				aAnsiconEnvVarValue = aAnsiconEnvVarValue.substring( 0, index );
				return Integer.valueOf( aAnsiconEnvVarValue.toString() );
			}
		}
		throw new NumberFormatException( "Cannot extract width from ANSICON formatted value <" + aAnsiconEnvVarValue + ">." );
	}

	/**
	 * Uses {@link #isLineBreakRequired(int)} to retrieve the character sequence
	 * required to suffix to a line in order to get a line break without risking
	 * any empty lines as of automatic line wrapping. Automatic line wrapping
	 * can happen on Windows environment when the console width is reached. A
	 * line break would cause a superfluous empty line (ugly).
	 * 
	 * @param aRowWidth The row width you want to use when printing out to a
	 *        console.
	 * 
	 * @return The system's line break characters when a line break is
	 *         emphasized or an empty string if no line break sequence is
	 *         recommended.
	 */
	public static String toLineBreak( int aRowWidth ) {
		if ( isLineBreakRequired( aRowWidth ) ) return getLineBreak();
		return "\r";
	}

	/**
	 * Extracts the width from a "mode con" call's result on "cmd.exe" to
	 * determine the console's height.
	 * 
	 * @param aModeConOutput "mode con" call's result.
	 * 
	 * @return The determined height.
	 * 
	 * @throws NumberFormatException Thrown in case no width was determinable
	 *         from the provided value.
	 */
	static int toModeConHeight( String aModeConOutput ) throws NumberFormatException {
		return toModeConValue( aModeConOutput, 3 );
	}

	/**
	 * Extracts the value from a "mode con" call result's specific row on
	 * "cmd.exe".
	 * 
	 * @param aModeConOutput "mode con" call's result.
	 * @param aValueIndex The index of the value to retrieve.
	 * @return The determined value for the given row.
	 * 
	 * @throws NumberFormatException Thrown in case no value was determinable
	 *         from the provided value.
	 */
	static int toModeConValue( String aModeConOutput, int aValueIndex ) throws NumberFormatException {
		try {
			int index = 0;
			for ( int i = 0; i < aValueIndex - 1; i++ ) {
				index = aModeConOutput.indexOf( ':' );
				if ( index == -1 || index == aModeConOutput.length() - 1 ) return -1;
				aModeConOutput = aModeConOutput.substring( index + 1 );
			}
			aModeConOutput = aModeConOutput.trim();
			index = -1;
			for ( int i = 0; i < aModeConOutput.length(); i++ ) {
				if ( Character.isDigit( aModeConOutput.charAt( i ) ) ) {
					index = i;
				}
				else break;
			}
			if ( index == -1 ) return -1;
			aModeConOutput = aModeConOutput.substring( 0, index + 1 );
			int theValue = Integer.parseInt( aModeConOutput );
			return theValue;
		}
		catch ( Exception ignore ) {}
		throw new NumberFormatException( "Cannot extract the value from \"mode con\" formatted value <" + aModeConOutput + "> for row <" + aValueIndex + ">." );
	}

	/**
	 * Extracts the width from a "mode con" call's result on "cmd.exe" to
	 * determine the console's width.
	 * 
	 * @param aModeConOutput "mode con" call's result.
	 * 
	 * @return The determined width.
	 * 
	 * @throws NumberFormatException Thrown in case no width was determinable
	 *         from the provided value.
	 */
	static int toModeConWidth( String aModeConOutput ) throws NumberFormatException {
		return toModeConValue( aModeConOutput, 4 );
	}

	/**
	 * Does some calculation to always return a sound console height (never
	 * returns -1). Anything in the REFCODES.ORG artifacts which is intended to
	 * print to the console uses this method to determine some heuristic best
	 * console height.
	 * 
	 * @return A sound heuristic console height.
	 */
	public static int toPreferredTerminalHeight() {
		int theHeight = -1;
		String theResult = SystemUtility.toPropertyValue( SystemProperty.CONSOLE_HEIGHT, EnvironmentVariable.CONSOLE_HEIGHT );
		if ( theResult != null ) {
			try {
				theHeight = Integer.valueOf( theResult.toString() );
			}
			catch ( NumberFormatException e ) {}
		}
		if ( theHeight <= 1 ) theHeight = getTerminalHeight();
		// Above unrealistic? |-->
		if ( theHeight < ConsoleDimension.MIN_HEIGHT.getValue() || theHeight > HEIGHT_ABOVE_UNREALSITIC ) {
			theHeight = ConsoleDimension.MIN_HEIGHT.getValue();
		}
		// Above unrealistic? <--|
		return theHeight;
	}

	/**
	 * Does some calculation to always return a sound console width (never
	 * returns -1). Anything in the REFCODES.ORG artifacts which is intended to
	 * print to the console uses this method to determine some heuristic best
	 * console width.
	 * 
	 * @return A sound heuristic console width.
	 */
	public static int toPreferredTerminalWidth() {
		int theWidth = -1;
		String theResult = SystemUtility.toPropertyValue( SystemProperty.CONSOLE_WIDTH, EnvironmentVariable.CONSOLE_WIDTH );
		if ( theResult != null ) {
			try {
				theWidth = Integer.valueOf( theResult.toString() );
			}
			catch ( NumberFormatException e ) {}
		}
		if ( theWidth <= 1 ) theWidth = getTerminalWidth();
		// Above unrealistic? |-->
		if ( theWidth < ConsoleDimension.MIN_WIDTH.getValue() || theWidth > WIDTH_ABOVE_UNREALSITIC ) {
			theWidth = ConsoleDimension.MIN_WIDTH.getValue();
		}
		// Above unrealistic? <--|
		return theWidth;
	}

	/**
	 * Determines the terminal your application is currently in.
	 * 
	 * @return The {@link Terminal} being detected.
	 */
	public static Terminal toTerminal() {
		String theTerminal = EnvironmentVariable.TERM.getValue();
		if ( theTerminal != null ) {
			theTerminal = theTerminal.toLowerCase();
			if ( theTerminal.indexOf( "cygwin" ) >= 0 ) {
				return Terminal.CYGWIN;
			}
			if ( theTerminal.indexOf( "xterm" ) >= 0 ) {
				return XTERM;
			}
		}
		return Terminal.UNKNOWN;
	}

	/**
	 * Determines the right value to be returned when encountering the given
	 * current evaluated terminal height and the new terminal height candidate.
	 * 
	 * @param aCurrentTerminalHeight The height to evaluate.
	 * 
	 * 
	 * @return The "right" value.
	 */
	private static int toTerminalHeight( int aCurrentTerminalHeight, IntSupplier aNewHeightCandidate ) {
		if ( isReconsiderableHeight( aCurrentTerminalHeight ) ) {
			int theNewHeight = aNewHeightCandidate.getAsInt();
			if ( !isUnrealsiticHeight( theNewHeight ) ) {
				return theNewHeight;
			}
		}
		return aCurrentTerminalHeight;
	}

	/**
	 * Determines the right value to be returned when encountering the given
	 * current evaluated terminal width and the new terminal width candidate.
	 * 
	 * @param aCurrentTerminalWidth The width to evaluate.
	 * 
	 * 
	 * @return The "right" value.
	 */
	private static int toTerminalWidth( int aCurrentTerminalWidth, IntSupplier aNewWidthCandidate ) {
		if ( isReconsiderableWidth( aCurrentTerminalWidth ) ) {
			int theNewWidthCandidate = aNewWidthCandidate.getAsInt();
			if ( !isUnrealsiticWidth( theNewWidthCandidate ) ) {
				return theNewWidthCandidate;
			}
		}
		return aCurrentTerminalWidth;
	}

}