// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.serial;

import org.refcodes.mixin.AbstractSchema;
import org.refcodes.mixin.AliasAccessor;
import org.refcodes.mixin.DescriptionAccessor;
import org.refcodes.mixin.LengthAccessor;
import org.refcodes.mixin.ValueAccessor;

/**
 * The purpose of a {@link SerialSchema} is automatically generate documentation
 * of serial data. It describes a portion of defined data such as a
 * {@link AllocSectionDecoratorSegment} and its components such as
 * {@link Segment} or {@link Section}. A type supporting its schema
 * representation
 */
public class SerialSchema extends AbstractSchema<SerialSchema> implements AliasAccessor, LengthAccessor, DescriptionAccessor, ValueAccessor<Sequence> {

	private static final long serialVersionUID = 1L;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	public static final String VERBOSE = "VERBOSE";
	public static final String LENGTH = "LENGTH";

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private int _length = -1;
	private Sequence _value = null;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs a {@link SerialSchema} with the provided arguments.
	 * 
	 * @param aType The type providing the this {@link SerialSchema} instance.
	 * @param aDescription The description of the schema described by the
	 *        {@link SerialSchema} providing type.
	 * @param aChildren The children's {@link SerialSchema} descriptions
	 *        representing part of the {@link SerialSchema} providing type.
	 */
	public SerialSchema( Class<?> aType, String aDescription, SerialSchema... aChildren ) {
		super( aType, aDescription, aChildren );
	}

	/**
	 * Constructs a {@link SerialSchema} with the provided arguments.
	 * 
	 * @param aType The type providing the this {@link SerialSchema} instance.
	 * @param aDescription The description of the schema described by the
	 *        {@link SerialSchema} providing type.
	 * @param aValue The content of the {@link SerialSchema} providing type.
	 * @param aLength The length of the {@link SerialSchema} providing type.
	 */
	public SerialSchema( Class<?> aType, String aDescription, Sequence aValue, int aLength ) {
		super( aType, aDescription );
		_value = aValue;
		_length = aLength;
		put( VALUE, aValue );
		put( LENGTH, aLength );
	}

	/**
	 * Constructs a {@link SerialSchema} with the provided arguments.
	 * 
	 * @param aType The type providing the this {@link SerialSchema} instance.
	 * @param aDescription The description of the schema described by the
	 *        {@link SerialSchema} providing type.
	 * @param aValue The content of the {@link SerialSchema} providing type.
	 * @param aVerbose The verbose representation of the value.
	 * @param aLength The length of the {@link SerialSchema} providing type.
	 */
	public SerialSchema( Class<?> aType, String aDescription, Sequence aValue, String aVerbose, int aLength ) {
		super( aType, aDescription );
		_value = aValue;
		_length = aLength;
		put( VALUE, aValue );
		put( VERBOSE, aVerbose );
		put( LENGTH, aLength );
	}

	/**
	 * Constructs a {@link SerialSchema} with the provided arguments.
	 * 
	 * @param aType The type providing the this {@link SerialSchema} instance.
	 * @param aDescription The description of the schema described by the
	 *        {@link SerialSchema} providing type.
	 * @param aValue The content of the {@link SerialSchema} providing type.
	 * @param aLength The length of the {@link SerialSchema} providing type.
	 * @param aChildren The children's {@link SerialSchema} descriptions
	 *        representing part of the {@link SerialSchema} providing type.
	 */
	public SerialSchema( Class<?> aType, String aDescription, Sequence aValue, int aLength, SerialSchema... aChildren ) {
		super( aType, aDescription, aChildren );
		_value = aValue;
		_length = aLength;
		put( VALUE, aValue );
		put( LENGTH, aLength );
	}

	/**
	 * Constructs a {@link SerialSchema} with the provided arguments.
	 * 
	 * @param aType The type providing the this {@link SerialSchema} instance.
	 * @param aDescription The description of the schema described by the
	 *        {@link SerialSchema} providing type.
	 * @param aValue The content of the {@link SerialSchema} providing type.
	 * @param aVerbose The verbose representation of the value.
	 * @param aLength The length of the {@link SerialSchema} providing type.
	 * @param aChildren The children's {@link SerialSchema} descriptions
	 *        representing part of the {@link SerialSchema} providing type.
	 */
	public SerialSchema( Class<?> aType, String aDescription, Sequence aValue, String aVerbose, int aLength, SerialSchema... aChildren ) {
		super( aType, aDescription, aChildren );
		_value = aValue;
		_length = aLength;
		put( VALUE, aValue );
		put( VERBOSE, aVerbose );
		put( LENGTH, aLength );

	}

	// -------------------------------------------------------------------------

	/**
	 * Constructs a {@link SerialSchema} with the provided arguments.
	 * 
	 * @param aAlias The alias (name) of the schema described by the
	 *        {@link SerialSchema} providing type.
	 * @param aDescription The description of the schema described by the
	 *        {@link SerialSchema} providing type.
	 * @param aType The type providing the this {@link SerialSchema} instance.
	 * @param aChildren The children's {@link SerialSchema} descriptions
	 *        representing part of the {@link SerialSchema} providing type.
	 */
	public SerialSchema( String aAlias, String aDescription, Class<?> aType, SerialSchema... aChildren ) {
		super( aType, aDescription, aChildren );
		put( ALIAS, aAlias );
	}

	/**
	 * Constructs a {@link SerialSchema} with the provided arguments.
	 * 
	 * @param aType The type providing the this {@link SerialSchema} instance.
	 * @param aDescription The description of the schema described by the
	 *        {@link SerialSchema} providing type.
	 * @param aAlias The alias (name) of the schema described by the
	 *        {@link SerialSchema} providing type.
	 * @param aLength The length of the {@link SerialSchema} providing type.
	 */
	public SerialSchema( Class<?> aType, String aDescription, String aAlias, int aLength ) {
		super( aType, aDescription );
		_length = aLength;
		put( ALIAS, aAlias );
		put( LENGTH, aLength );
	}

	/**
	 * Constructs a {@link SerialSchema} with the provided arguments.
	 * 
	 * @param aType The type providing the this {@link SerialSchema} instance.
	 * @param aDescription The description of the schema described by the
	 *        {@link SerialSchema} providing type.
	 * @param aAlias The alias (name) of the schema described by the
	 *        {@link SerialSchema} providing type.
	 * @param aValue The content of the {@link SerialSchema} providing type.
	 * @param aLength The length of the {@link SerialSchema} providing type.
	 */
	public SerialSchema( Class<?> aType, String aDescription, String aAlias, Sequence aValue, int aLength ) {
		super( aType, aDescription );
		_value = aValue;
		_length = aLength;
		put( ALIAS, aAlias );
		put( VALUE, aValue );
		put( LENGTH, aLength );
	}

	/**
	 * Constructs a {@link SerialSchema} with the provided arguments.
	 * 
	 * @param aType The type providing the this {@link SerialSchema} instance.
	 * @param aDescription The description of the schema described by the
	 *        {@link SerialSchema} providing type.
	 * @param aAlias The alias (name) of the schema described by the
	 *        {@link SerialSchema} providing type.
	 * @param aValue The content of the {@link SerialSchema} providing type.
	 * @param aVerbose The verbose representation of the value.
	 * @param aLength The length of the {@link SerialSchema} providing type.
	 */
	public SerialSchema( Class<?> aType, String aDescription, String aAlias, Sequence aValue, String aVerbose, int aLength ) {
		super( aType, aDescription );
		_value = aValue;
		_length = aLength;
		put( ALIAS, aAlias );
		put( VALUE, aValue );
		put( VERBOSE, aVerbose );
		put( LENGTH, aLength );
	}

	/**
	 * Constructs a {@link SerialSchema} with the provided arguments.
	 * 
	 * @param aType The type providing the this {@link SerialSchema} instance.
	 * @param aDescription The description of the schema described by the
	 *        {@link SerialSchema} providing type.
	 * @param aAlias The alias (name) of the schema described by the
	 *        {@link SerialSchema} providing type.
	 * @param aValue The content of the {@link SerialSchema} providing type.
	 * @param aLength The length of the {@link SerialSchema} providing type.
	 * @param aChildren The children's {@link SerialSchema} descriptions
	 *        representing part of the {@link SerialSchema} providing type.
	 */
	public SerialSchema( Class<?> aType, String aDescription, String aAlias, Sequence aValue, int aLength, SerialSchema... aChildren ) {
		super( aType, aDescription, aChildren );
		_value = aValue;
		_length = aLength;
		put( ALIAS, aAlias );
		put( VALUE, aValue );
		put( LENGTH, aLength );
	}

	/**
	 * Constructs a {@link SerialSchema} with the provided arguments.
	 * 
	 * @param aType The type providing the this {@link SerialSchema} instance.
	 * @param aDescription The description of the schema described by the
	 *        {@link SerialSchema} providing type.
	 * @param aAlias The alias (name) of the schema described by the
	 *        {@link SerialSchema} providing type.
	 * @param aValue The content of the {@link SerialSchema} providing type.
	 * @param aVerbose The verbose representation of the value.
	 * @param aLength The length of the {@link SerialSchema} providing type.
	 * @param aChildren The children's {@link SerialSchema} descriptions
	 *        representing part of the {@link SerialSchema} providing type.
	 */
	public SerialSchema( Class<?> aType, String aDescription, String aAlias, Sequence aValue, String aVerbose, int aLength, SerialSchema... aChildren ) {
		super( aType, aDescription, aChildren );
		_value = aValue;
		_length = aLength;
		put( ALIAS, aAlias );
		put( VALUE, aValue );
		put( VERBOSE, aVerbose );
		put( LENGTH, aLength );
	}

	// -------------------------------------------------------------------------

	/**
	 * Constructs a {@link SerialSchema} with the properties of the provided
	 * {@link SerialSchema} ("copy constructor"),.
	 *
	 * @param aSchema The {@link SerialSchema} from which to copy the
	 *        attributes.
	 */
	public SerialSchema( SerialSchema aSchema ) {
		super( aSchema );
	}

	/**
	 * Constructs a {@link SerialSchema} with the properties of the provided
	 * {@link SerialSchema} ("copy constructor"),.
	 *
	 * @param aSchema The {@link SerialSchema} from which to copy the
	 *        attributes.
	 * @param aChildren The children's {@link SerialSchema} descriptions
	 *        representing part of the {@link SerialSchema} providing type.
	 */
	public SerialSchema( SerialSchema aSchema, SerialSchema... aChildren ) {
		super( aSchema, aChildren );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Sequence getValue() {
		return _value;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getLength() {
		return _length;
	}

	/**
	 * Retrieves the verbose representation of the value.
	 * 
	 * @return The human readable value representation.
	 */
	public String getVerbose() {
		return (String) get( VERBOSE );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDescription() {
		return (String) get( DESCRIPTION );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getAlias() {
		return (String) get( ALIAS );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String toString( Object aValue ) {
		if ( aValue instanceof Sequence ) {
			return toString( ((Sequence) aValue).toBytes() );
		}
		return super.toString( aValue );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

}