// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.serial;

import static org.junit.jupiter.api.Assertions.*;
import static org.refcodes.serial.SerialSugar.*;

import java.io.IOException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.Provider;
import java.security.Security;

import javax.crypto.Cipher;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;

import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.Test;
import org.refcodes.numerical.CrcAlgorithmConfig;
import org.refcodes.security.alt.chaos.ChaosKey;
import org.refcodes.security.ext.chaos.ChaosProvider;
import org.refcodes.serial.TestFixures.WeatherData;

public class CipherTest extends AbstractLoopbackPortTest {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	private static boolean IS_LOG_TEST_ENABLED = Boolean.getBoolean( "log.test");

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final int LOOPS = 15;

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@Disabled
	@Test
	public void testCipherSimple() throws TransmissionException, InvalidKeyException, NoSuchAlgorithmException, NoSuchPaddingException {
		// @formatter:off
		LengthSegmentDecoratorSegment<?> theSegment = lengthSegment( 
			allocSegment( 
				cipherSection( 
					stringSection("Hello World!"),toCipher( "TopSecret123!")
				)
			)
		);
		// @formatter:on

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( theSegment.toSchema() );
		}
		Sequence theSequence = theSegment.toSequence();

		// @formatter:off
		LengthSegmentDecoratorSegment<?> theOtherSegment = lengthSegment( 
				allocSegment( 
					cipherSection( 
						stringSection(),toCipher( "TopSecret123!")
					)
				)
			);
		// @formatter:on

		theOtherSegment.fromTransmission( theSequence );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( theOtherSegment.toSchema() );
		}

		assertEquals( theSegment, theOtherSegment );
	}

	@Disabled
	@Test
	public void testCipherSimpleFail() throws TransmissionException, InvalidKeyException, NoSuchAlgorithmException, NoSuchPaddingException {
		// @formatter:off
		LengthSegmentDecoratorSegment<?> theSegment = lengthSegment( 
			allocSegment( 
				cipherSection( 
					stringSection("Hello World!"),toCipher( "TopSecret123!")
				)
			)
		);
		// @formatter:on

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( theSegment.toSchema() );
		}
		Sequence theSequence = theSegment.toSequence();

		// @formatter:off
		LengthSegmentDecoratorSegment<?> theOtherSegment = lengthSegment( 
				allocSegment( 
					cipherSection( 
						stringSection(), toCipher( "NotSecret123!")
					)
				)
			);
		// @formatter:on

		theOtherSegment.fromTransmission( theSequence );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( theOtherSegment.toSchema() );
		}

		assertNotEquals( theSegment, theOtherSegment );
	}

	@Disabled
	@Test
	public void testCipherComplex() throws IOException, InvalidKeyException, NoSuchAlgorithmException, NoSuchPaddingException {
		if ( hasPorts() ) {
			Port<?> theTransmitPort = getTransmitterPort().withOpen();
			Port<?> theReceiverPort = getReceiverPort().withOpen();
			WeatherData theSenderData = TestFixures.createWeatherData();
			ComplexTypeSegment<WeatherData> theReceiverSegment;

			for ( int i = 0; i < LOOPS; i++ ) {
				String eSecret = "TopSecret" + ('A' + i);
				Cipher eCipher;
				Segment theSenderSeg = readyToSendSegment( cipherSegment( crcPrefixSegment( complexTypeSegment( theSenderData ), CrcAlgorithmConfig.CRC_16_CCITT_FALSE ), eCipher = toCipher( eSecret ) ) );
				Segment theReceiverSeg = readyToSendSegment( cipherSegment( crcPrefixSegment( theReceiverSegment = complexTypeSegment( WeatherData.class ), CrcAlgorithmConfig.CRC_16_CCITT_FALSE ), eCipher ) );

				SegmentResult<Segment> theResult = theReceiverPort.onReceiveSegment( theReceiverSeg );
				theSenderSeg.transmitTo( theTransmitPort );
				theResult.waitForResult();

				if ( IS_LOG_TEST_ENABLED ) {
					try {
						System.out.println( "Expected [" + eCipher + "]= " + theResult.getResult() );
					}
					catch ( BadCrcChecksumException e ) {
						System.out.println( "Unexpected exceptio = " + e.getMessage() );
						fail( "Not Expecting a <" + BadCrcChecksumException.class.getSimpleName() + "> instead of a resul!t" );
					}
				}

				WeatherData theReceiverData = theReceiverSegment.getPayload();
				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( theReceiverData );
				}
				assertEquals( theSenderData, theReceiverData );
			}

			theTransmitPort.close();
			theReceiverPort.close();
		}
		else {
			System.out.println( "Skipping test, please connect your null modem cable to two serial ports on your box, seeking for exactly two FT232 (ftdi_sio) type ports!" );
		}
	}

	@Disabled
	@Test
	public void testCipherComplexFail() throws IOException, InvalidKeyException, NoSuchAlgorithmException, NoSuchPaddingException {
		if ( hasPorts() ) {
			Port<?> theTransmitPort = getTransmitterPort().withOpen();
			Port<?> theReceiverPort = getReceiverPort().withOpen();
			WeatherData theSenderData = TestFixures.createWeatherData();
			ComplexTypeSegment<WeatherData> theReceiverSegment;
			Segment theSenderSeg = cipherSegment( crcPrefixSegment( complexTypeSegment( theSenderData ), CrcAlgorithmConfig.CRC_16_CCITT_FALSE ), toCipher( "TopSecret123!" ) );
			Segment theReceiverSeg = cipherSegment( crcPrefixSegment( theReceiverSegment = complexTypeSegment( WeatherData.class ), CrcAlgorithmConfig.CRC_16_CCITT_FALSE ), toCipher( "NotSecret123!" ) );

			SegmentResult<Segment> theResult = theReceiverPort.onReceiveSegment( theReceiverSeg );
			theSenderSeg.transmitTo( theTransmitPort );
			theResult.waitForResult();

			if ( IS_LOG_TEST_ENABLED ) {
				try {
					System.out.println( "Result = " + theResult.getResult() );
					fail( "Expecting a <" + BadCrcChecksumException.class.getSimpleName() + "> instead of a resul!t" );
				}
				catch ( BadCrcChecksumException expected ) {
					System.out.println( "Expected = " + expected.getMessage() );
				}
			}

			WeatherData theReceiverData = theReceiverSegment.getPayload();
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( theReceiverData );
			}
			assertNotEquals( theSenderData, theReceiverData );

			theTransmitPort.close();
			theReceiverPort.close();
		}
		else {
			System.out.println( "Skipping test, please connect your null modem cable to two serial ports on your box, seeking for exactly two FT232 (ftdi_sio) type ports!" );
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	public static Cipher toCipher( String aSecret ) throws NoSuchAlgorithmException, NoSuchPaddingException, InvalidKeyException {
		int theIndex = Security.addProvider( new ChaosProvider() );
		if ( IS_LOG_TEST_ENABLED ) {
			for ( Provider eProvider : Security.getProviders() ) {
				System.out.println( eProvider.getName() + " - " + eProvider.getInfo() );
			}
		}
		assertTrue( theIndex >= 0 );
		Cipher c = Cipher.getInstance( ChaosKey.PROVIDER_NAME, new ChaosProvider() );
		SecretKey key = new ChaosKey( aSecret );
		c.init( Cipher.ENCRYPT_MODE, key );
		return c;

		//	ChaosCipher theCipher = new ChaosCipher();
		//	SecretKey key = new ChaosKey( aSecret );
		//	theCipher.init( Cipher.ENCRYPT_MODE, key );
		//	return theCipher;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
