// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.serial;

import static org.junit.jupiter.api.Assertions.*;

import java.io.IOException;
import java.util.Arrays;

import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.Test;
import org.refcodes.runtime.RuntimeUtility;
import org.refcodes.struct.SimpleTypeMap;
import org.refcodes.textual.CaseStyleBuilder;

public class ComplexTypeSegmentTest extends TestFixures {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static boolean IS_LOG_TEST_ENABLED = Boolean.getBoolean( "log.test");

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@Test
	public void testEdgeCase1() throws TransmissionException {
		printTestName( RuntimeUtility.toMethodName() );
		String theValue = "Hallo Welt!";
		ComplexTypeSegment<String> theSegment = new ComplexTypeSegment<>( theValue );
		printSchema( theSegment );
		printSimpleTypeMap( theSegment );
		Sequence theSequence = theSegment.toSequence();
		ComplexTypeSegment<String> theOtherSegment = new ComplexTypeSegment<>( String.class );
		theOtherSegment.fromTransmission( theSequence );
		printSchema( theOtherSegment );
		printSimpleTypeMap( theOtherSegment );
		String theOtherValue = theOtherSegment.getPayload();
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( theOtherValue );
		}
		Sequence theOtherSequence = theOtherSegment.toSequence();
		assertEquals( theValue, theOtherValue );
		assertEquals( theSequence, theOtherSequence );
		assertNotSame( theValue, theOtherValue );
	}

	enum ReadySteadyGo {
		READY, STEADY, GO
	}

	@Test
	public void testEdgeCase2() throws TransmissionException {
		printTestName( RuntimeUtility.toMethodName() );
		ReadySteadyGo theValue = ReadySteadyGo.STEADY;
		ComplexTypeSegment<ReadySteadyGo> theSegment = new ComplexTypeSegment<>( theValue );
		printSchema( theSegment );
		printSimpleTypeMap( theSegment );
		Sequence theSequence = theSegment.toSequence();
		ComplexTypeSegment<ReadySteadyGo> theOtherSegment = new ComplexTypeSegment<>( ReadySteadyGo.class );
		theOtherSegment.fromTransmission( theSequence );
		printSchema( theOtherSegment );
		printSimpleTypeMap( theOtherSegment );
		ReadySteadyGo theOtherValue = theOtherSegment.getPayload();
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( theOtherValue );
		}
		Sequence theOtherSequence = theOtherSegment.toSequence();
		assertEquals( theValue, theOtherValue );
		assertEquals( theSequence, theOtherSequence );
		assertSame( theValue, theOtherValue );
	}

	@Test
	public void testComplexType1() throws TransmissionException {
		printTestName( RuntimeUtility.toMethodName() );
		int[] theValues = new int[] { 1, 2, 3, 4, 5, 6, 7, 8, 9, 0 };
		ComplexTypeSegment<int[]> theSegment = new ComplexTypeSegment<>( theValues );
		printSchema( theSegment );
		printSimpleTypeMap( theSegment );
		Sequence theSequence = theSegment.toSequence();
		ComplexTypeSegment<int[]> theOtherSegment = new ComplexTypeSegment<>( int[].class );
		theOtherSegment.fromTransmission( theSequence );
		printSchema( theOtherSegment );
		printSimpleTypeMap( theOtherSegment );
		int[] theOtherValues = theOtherSegment.getPayload();
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( Arrays.toString( theOtherValues ) );
		}
		Sequence theOtherSequence = theOtherSegment.toSequence();
		assertArrayEquals( theValues, theOtherValues );
		assertEquals( theSequence, theOtherSequence );
		assertNotSame( theValues, theOtherValues );
	}

	@Test
	public void testComplexType2() throws TransmissionException {
		printTestName( RuntimeUtility.toMethodName() );
		Values theValues = createValues();
		ComplexTypeSegment<Values> theSegment = new ComplexTypeSegment<>( theValues );
		printSchema( theSegment );
		printSimpleTypeMap( theSegment );
		Sequence theSequence = theSegment.toSequence();
		ComplexTypeSegment<Values> theOtherSegment = new ComplexTypeSegment<>( Values.class );
		theOtherSegment.fromTransmission( theSequence );
		printSchema( theOtherSegment );
		printSimpleTypeMap( theOtherSegment );
		Values theOtherValues = theOtherSegment.getPayload();
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( theOtherValues );
		}
		Sequence theOtherSequence = theOtherSegment.toSequence();
		assertEquals( theSequence, theOtherSequence );
		assertEquals( theValues, theOtherValues );
		assertNotSame( theValues, theOtherValues );
	}

	@Test
	public void testComplexType3() throws TransmissionException {
		printTestName( RuntimeUtility.toMethodName() );
		SensorValues theSensorValues = createSensorValues();
		ComplexTypeSegment<SensorValues> theSegment = new ComplexTypeSegment<>( theSensorValues );
		printSchema( theSegment );
		printSimpleTypeMap( theSegment );
		Sequence theSequence = theSegment.toSequence();
		ComplexTypeSegment<SensorValues> theOtherSegment = new ComplexTypeSegment<>( SensorValues.class );
		theOtherSegment.fromTransmission( theSequence );
		printSchema( theOtherSegment );
		printSimpleTypeMap( theOtherSegment );
		SensorValues theOtherValues = theOtherSegment.getPayload();
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( theOtherValues );
		}
		Sequence theOtherSequence = theOtherSegment.toSequence();
		assertEquals( theSequence, theOtherSequence );
		assertEquals( theSensorValues, theOtherValues );
		assertNotSame( theSensorValues, theOtherSegment );
	}

	@Test
	public void testComplexType4() throws TransmissionException {
		printTestName( RuntimeUtility.toMethodName() );
		Wrapper theWrapper = createWrapper();
		ComplexTypeSegment<Wrapper> theSegment = new ComplexTypeSegment<>( theWrapper );
		printSchema( theSegment );
		printSimpleTypeMap( theSegment );
		Sequence theSequence = theSegment.toSequence();
		ComplexTypeSegment<Wrapper> theOtherSegment = new ComplexTypeSegment<>( Wrapper.class );
		theOtherSegment.fromTransmission( theSequence );
		printSchema( theOtherSegment );
		printSimpleTypeMap( theOtherSegment );
		Wrapper theOtherWrapper = theOtherSegment.getPayload();
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( theOtherWrapper );
		}
		Sequence theOtherSequence = theOtherSegment.toSequence();
		assertEquals( theSequence, theOtherSequence );
		assertEquals( theWrapper, theOtherWrapper );
		assertNotSame( theWrapper, theOtherWrapper );
	}

	@Test
	public void testComplexType5() throws TransmissionException {
		printTestName( RuntimeUtility.toMethodName() );
		Sensors theSensors = createSensors();
		ComplexTypeSegment<Sensors> theSegment = new ComplexTypeSegment<>( theSensors );
		printSchema( theSegment );
		printSimpleTypeMap( theSegment );
		Sequence theSequence = theSegment.toSequence();
		ComplexTypeSegment<Sensors> theOtherSegment = new ComplexTypeSegment<>( Sensors.class );
		theOtherSegment.fromTransmission( theSequence );
		printSchema( theOtherSegment );
		printSimpleTypeMap( theOtherSegment );
		Sensors theOtherSensors = theOtherSegment.getPayload();
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( theOtherSensors );
		}
		Sequence theOtherSequence = theOtherSegment.toSequence();
		assertEquals( theSequence, theOtherSequence );
		assertEquals( theSensors, theOtherSensors );
		assertNotSame( theSensors, theOtherSensors );
	}

	@Disabled("Under construction")
	@Test
	public void testComplexType6() throws TransmissionException {
		printTestName( RuntimeUtility.toMethodName() );
		Sensors[] theArray = new Sensors[] { createSensors(), createSensors(), createSensors(), createSensors(), createSensors() };
		ComplexTypeSegment<Sensors[]> theSegment = new ComplexTypeSegment<>( theArray );
		printSchema( theSegment );
		printSimpleTypeMap( theSegment );
		Sequence theSequence = theSegment.toSequence();
		ComplexTypeSegment<Sensors[]> theOtherSegment = new ComplexTypeSegment<>( Sensors[].class );
		theOtherSegment.fromTransmission( theSequence );
		printSchema( theOtherSegment );
		printSimpleTypeMap( theOtherSegment );
		Sensors[] theOtherArray = theOtherSegment.getPayload();
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( theOtherArray );
		}
		Sequence theOtherSequence = theOtherSegment.toSequence();
		assertEquals( theSequence, theOtherSequence );
		assertArrayEquals( theArray, theOtherArray );
		assertNotSame( theArray, theOtherArray );
	}

	@Test
	public void testCompositeArrayType1() throws TransmissionException {
		printTestName( RuntimeUtility.toMethodName() );
		Sensors[] theValues = new Sensors[] { createSensors(), createSensors(), createSensors() };
		ComplexTypeSegment<Sensors[]> theSegment = new ComplexTypeSegment<Sensors[]>( theValues );
		printSchema( theSegment );
		printSimpleTypeMap( theSegment );
		Sequence theSequence = theSegment.toSequence();
		ComplexTypeSegment<Sensors[]> theOtherSegment = new ComplexTypeSegment<>( Sensors[].class );
		theOtherSegment.fromTransmission( theSequence );
		printSchema( theOtherSegment );
		printSimpleTypeMap( theOtherSegment );
		Sequence theOtherSequence = theOtherSegment.toSequence();
		Sensors[] theOtherValues = theOtherSegment.getPayload();
		if ( IS_LOG_TEST_ENABLED ) {
			for ( Sensors eValue : theOtherValues ) {
				System.out.println( eValue );
			}
		}
		assertEquals( theSequence, theOtherSequence );
		assertArrayEquals( theValues, theOtherValues );
		assertNotSame( theValues, theOtherValues );
	}

	@Test
	public void testCompositeArrayType2() throws TransmissionException {
		printTestName( RuntimeUtility.toMethodName() );
		SensorArray theValues = new SensorArray( new Sensor[] { new Sensor( "Sensor1", 1 ), new Sensor( "Sensor2", 22 ), new Sensor( "Sensor3", 333 ) } );
		ComplexTypeSegment<SensorArray> theSegment = new ComplexTypeSegment<>( theValues );
		printSchema( theSegment );
		printSimpleTypeMap( theSegment );
		Sequence theSequence = theSegment.toSequence();
		ComplexTypeSegment<SensorArray> theOtherSegment = new ComplexTypeSegment<>( SensorArray.class );
		theOtherSegment.fromTransmission( theSequence );
		printSchema( theOtherSegment );
		printSimpleTypeMap( theOtherSegment );
		Sequence theOtherSequence = theOtherSegment.toSequence();
		SensorArray theOtherValues = theOtherSegment.getPayload();
		if ( IS_LOG_TEST_ENABLED ) {
			for ( Sensor eValue : theOtherValues.getSensors() ) {
				System.out.println( eValue );
			}
		}
		assertEquals( theSequence, theOtherSequence );
		assertEquals( theValues, theOtherValues );
		assertNotSame( theValues, theOtherValues );
	}

	@Test
	public void testCompositeArrayType3() throws TransmissionException {
		printTestName( RuntimeUtility.toMethodName() );
		SensorArray theValues[] = new SensorArray[] { new SensorArray( new Sensor[] { new Sensor( "Sensor1", 1 ), new Sensor( "Sensor2", 22 ), new Sensor( "Sensor3", 333 ) } ), new SensorArray( new Sensor[] { new Sensor( "Sensor4", 4 ), new Sensor( "Sensor5", 55 ), new Sensor( "Sensor6", 666 ) } ), new SensorArray( new Sensor[] { new Sensor( "Sensor7", 7 ), new Sensor( "Sensor8", 88 ), new Sensor( "Sensor9", 999 ) } ) };
		ComplexTypeSegment<SensorArray[]> theSegment = new ComplexTypeSegment<>( theValues );
		printSegment( theSegment );
		printSimpleTypeMap( theSegment );
		Sequence theSequence = theSegment.toSequence();
		ComplexTypeSegment<SensorArray[]> theOtherSegment = new ComplexTypeSegment<>( SensorArray[].class );
		theOtherSegment.fromTransmission( theSequence );
		printSegment( theOtherSegment );
		printSimpleTypeMap( theOtherSegment );
		Sequence theOtherSequence = theOtherSegment.toSequence();
		SensorArray[] theOtherValues = theOtherSegment.getPayload();
		if ( IS_LOG_TEST_ENABLED ) {
			for ( SensorArray eArray : theOtherValues ) {
				for ( Sensor eValue : eArray.getSensors() ) {
					System.out.println( eValue );
				}
			}
		}
		assertEquals( theSequence, theOtherSequence );
		assertArrayEquals( theValues, theOtherValues );
		assertNotSame( theValues, theOtherValues );
	}

	@Test
	public void testCompositeArrayType4() throws TransmissionException {
		printTestName( RuntimeUtility.toMethodName() );
		SensorValues theValues = createSensorValues();
		ComplexTypeSegment<SensorValues> theSegment = new ComplexTypeSegment<>( theValues );
		printSchema( theSegment );
		printSimpleTypeMap( theSegment );
		Sequence theSequence = theSegment.toSequence();
		ComplexTypeSegment<SensorValues> theOtherSegment = new ComplexTypeSegment<>( SensorValues.class );
		theOtherSegment.fromTransmission( theSequence );
		printSchema( theOtherSegment );
		printSimpleTypeMap( theOtherSegment );
		Sequence theOtherSequence = theOtherSegment.toSequence();
		SensorValues theOtherValues = theOtherSegment.getPayload();
		assertEquals( theSequence, theOtherSequence );
		assertEquals( theValues, theOtherValues );
		assertNotSame( theValues, theOtherValues );
	}

	@Test
	public void testCompositeArrayType5() throws TransmissionException {
		printTestName( RuntimeUtility.toMethodName() );
		Values theValues = createValues();
		ComplexTypeSegment<Values> theSegment = new ComplexTypeSegment<>( theValues );
		printSchema( theSegment );
		printSimpleTypeMap( theSegment );
		Sequence theSequence = theSegment.toSequence();
		ComplexTypeSegment<Values> theOtherSegment = new ComplexTypeSegment<>( Values.class );
		theOtherSegment.fromTransmission( theSequence );
		printSchema( theOtherSegment );
		printSimpleTypeMap( theOtherSegment );
		Sequence theOtherSequence = theOtherSegment.toSequence();
		Values theOtherValues = theOtherSegment.getPayload();
		assertEquals( theSequence, theOtherSequence );
		assertEquals( theValues, theOtherValues );
		assertNotSame( theValues, theOtherValues );
	}

	@Test
	public void testCompositeArrayType6() throws TransmissionException {
		printTestName( RuntimeUtility.toMethodName() );
		Wrapper theValues = createWrapper();
		ComplexTypeSegment<Wrapper> theSegment = new ComplexTypeSegment<>( theValues );
		printSchema( theSegment );
		printSimpleTypeMap( theSegment );
		Sequence theSequence = theSegment.toSequence();
		ComplexTypeSegment<Wrapper> theOtherSegment = new ComplexTypeSegment<>( Wrapper.class );
		theOtherSegment.fromTransmission( theSequence );
		printSchema( theOtherSegment );
		printSimpleTypeMap( theOtherSegment );
		Sequence theOtherSequence = theOtherSegment.toSequence();
		Wrapper theOtherValues = theOtherSegment.getPayload();
		assertEquals( theSequence, theOtherSequence );
		assertEquals( theValues, theOtherValues );
		assertNotSame( theValues, theOtherValues );
	}

	@Test
	public void testComplexTypeStream() throws TransmissionException, IOException {
		printTestName( RuntimeUtility.toMethodName() );
		SensorArray theValues[] = new SensorArray[] { new SensorArray( new Sensor[] { new Sensor( "Sensor1", 1 ), new Sensor( "Sensor2", 22 ), new Sensor( "Sensor3", 333 ) } ), new SensorArray( new Sensor[] { new Sensor( "Sensor4", 4 ), new Sensor( "Sensor5", 55 ), new Sensor( "Sensor6", 666 ) } ), new SensorArray( new Sensor[] { new Sensor( "Sensor7", 7 ), new Sensor( "Sensor8", 88 ), new Sensor( "Sensor9", 999 ) } ) };
		ComplexTypeSegment<SensorArray[]> theSegment = new ComplexTypeSegment<>( theValues );
		printSchema( theSegment );
		printSimpleTypeMap( theSegment );
		LoopbackPort theTransceiver = new LoopbackPort( "loop0" ).withOpen();
		theSegment.transmitTo( theTransceiver.getOutputStream() );
		ComplexTypeSegment<SensorArray[]> theOtherSegment = new ComplexTypeSegment<>( SensorArray[].class );
		theOtherSegment.receiveFrom( theTransceiver.getInputStream() );
		printSchema( theOtherSegment );
		printSimpleTypeMap( theOtherSegment );
		SensorArray[] theOtherValues = theOtherSegment.getPayload();
		assertArrayEquals( theValues, theOtherValues );
		assertNotSame( theValues, theOtherValues );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	private void printTestName( String aMethodName ) {
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println();
			System.out.println( "[---------- " + CaseStyleBuilder.asSnakeCase( aMethodName ) + " ----------]" );
			System.out.println();
		}
	}

	private void printSimpleTypeMap( Transmission aSegment ) {
		if ( IS_LOG_TEST_ENABLED ) {
			SimpleTypeMap theSimpleTypeMap = aSegment.toSimpleTypeMap();
			for ( String eKey : theSimpleTypeMap.sortedKeys() ) {
				System.out.println( eKey + " = " + theSimpleTypeMap.get( eKey ) );
			}
			System.out.println();
		}
	}

	private void printSchema( Transmission aSegment ) {
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( aSegment.toSchema() );
			System.out.println();
		}
	}

	private void printSegment( Transmission aSegment ) {
		if ( IS_LOG_TEST_ENABLED ) {
			SimpleTypeMap theSimpleTypeMap = aSegment.toSimpleTypeMap();
			for ( String eKey : theSimpleTypeMap.sortedKeys() ) {
				System.out.println( eKey + " = " + theSimpleTypeMap.get( eKey ) );
			}
			System.out.println();
		}
	}
}
