// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.serial;

import static org.junit.jupiter.api.Assertions.*;
import static org.refcodes.serial.SerialSugar.*;

import org.junit.jupiter.api.Test;
import org.refcodes.numerical.CrcAlgorithmConfig;

public class CrcSegmentTest {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static boolean IS_LOG_TEST_ENABLED = Boolean.getBoolean( "log.test");

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Test crc segment facade.
	 *
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testCrcSegmentFacade() throws TransmissionException {
		// @formatter:off
		Segment theSegment = crcPrefixSegment( 
			allocSegment( 
				stringSection("Hello World!")
			), CrcAlgorithmConfig.CRC_16_CCITT_FALSE
		);
		// @formatter:on

		Sequence theSequence = theSegment.toSequence();

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Sequence (hex) = " + theSequence.toHexString() );
			System.out.println( "Transmission = " + theSegment.toString() );
			System.out.println( "AbstractSchema = " + theSegment.toSchema() );
		}

		// @formatter:off
		Segment theOtherSegment = crcPrefixSegment(
				allocSegment( 
				stringSection()
			), CrcAlgorithmConfig.CRC_16_CCITT_FALSE
		);
		// @formatter:on

		theOtherSegment.fromTransmission( theSequence );

		assertEquals( theSegment, theOtherSegment );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Other segment = " + theOtherSegment.toString() );
			System.out.println( "Other schema = " + theOtherSegment.toSchema() );
		}

		// BadCrcChecksumSequenceException:
		try {
			theSequence.setByteAt( 1, (byte) 1 ); // First 4 Bytes = length
			theOtherSegment.fromTransmission( theSequence );
			fail( "Expected a <BadCrcChecksumSequenceException> here!" );
		}
		catch ( BadCrcChecksumSequenceException expected ) { /* expected */ }

	}

	/**
	 * Test crc body.
	 *
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testCrcBody() throws TransmissionException {
		LengthSegmentDecoratorSegment<?> theSegment = new LengthSegmentDecoratorSegment<>( new IntSegment( 5161 ) );
		CrcSegmentDecorator<Segment> theCrcHeader = new CrcSegmentDecorator<Segment>( theSegment, CrcAlgorithmConfig.CRC_16_CCITT_FALSE );
		long theCrcChecksum = theCrcHeader.getCrcChecksum();
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Checksum for payload size <" + theSegment.getAllocLength() + "> = " + theCrcChecksum );
		}
		Sequence theSequence = theCrcHeader.toSequence();
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "To sequence = " + theSequence.toHexString() );
		}
		theCrcHeader.fromTransmission( theSequence );
		theSequence = theCrcHeader.toSequence();
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "From sequence = " + theSequence.toHexString() );
		}
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Other checksum for payload size <" + theSegment.getAllocLength() + "> = " + theCrcChecksum );
		}
	}

	/**
	 * Test crc header 1.
	 *
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testCrcHeader1() throws TransmissionException {
		String thePayload = "Hello World!";
		// @formatter:off
		CrcSegmentDecorator<?> theSegment = crcPrefixSegment(
			segmentComposite(
				magicBytesSegment( (byte) 0xC0, (byte) 0xDE, (byte) 0xFE, (byte) 0xED ),
				allocSegment( 
					stringSection(thePayload)
				)
			), CrcAlgorithmConfig.CRC_16_CCITT_FALSE
		);
		// @formatter:on

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Transmission = " + theSegment.toString() );
		}

		// @formatter:off
		CrcSegmentDecorator<?> theOtherSegment = crcPrefixSegment(
			segmentComposite(
				magicBytesSegment( (byte) 0xC0, (byte) 0xDE, (byte) 0xFE, (byte) 0xED ),
				intSegment(
					thePayload.length() + 1
				)
			), CrcAlgorithmConfig.CRC_16_CCITT_FALSE
		);
		// @formatter:on

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Other segment = " + theOtherSegment.toString() );
		}

		assertNotEquals( theSegment.getCrcChecksum(), theOtherSegment.getCrcChecksum() );
	}

	/**
	 * Test crc header 2.
	 *
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testCrcHeader2() throws TransmissionException {
		String thePayload = "Hello World!";
		// @formatter:off
		CrcSegmentDecorator<?> theSegment = crcPrefixSegment(
			segmentComposite(
				magicBytesSegment( (byte) 0xC0, (byte) 0xDE, (byte) 0xFE, (byte) 0xED ),
				allocSegment( 
					stringSection(thePayload)
				)
			), CrcAlgorithmConfig.CRC_16_CCITT_FALSE
		);
		// @formatter:on

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Transmission = " + theSegment.toString() );
		}

		// @formatter:off
		CrcSegmentDecorator<?> theOtherSegment = crcPrefixSegment(
			segmentComposite(
				magicBytesSegment( (byte) 0xC0, (byte) 0xDE, (byte) 0xFE, (byte) 0xED ),
				intSegment(
					thePayload.length()
				)
			), CrcAlgorithmConfig.CRC_16_CCITT_FALSE
		);
		// @formatter:on

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Other segment = " + theOtherSegment.toString() );
		}

		assertNotEquals( theSegment.getCrcChecksum(), theOtherSegment.getCrcChecksum() );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
