// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.serial;

import static org.junit.jupiter.api.Assertions.*;
import static org.refcodes.serial.SerialSugar.*;

import java.util.Random;

import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.Test;
import org.refcodes.numerical.Endianess;

public class DataTypesTest {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////
	
	private static boolean IS_LOG_TEST_ENABLED = Boolean.getBoolean( "log.test");

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	// -------------------------------------------------------------------------
	// ASCIIZ:
	// -------------------------------------------------------------------------

	@Test
	public void testAsciizSegment1() throws TransmissionException {
		AsciizSegment theSegment = asciizSegment( "Hallo Welt!" );
		Sequence theSequence = theSegment.toSequence();

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "AllocationBodyFacade (hex) = " + theSequence.toHexString() );
			System.out.println( "AllocationBodyFacade = " + theSegment.toString() );
			System.out.println( "AbstractSchema = " + theSegment.toSchema() );
		}

		assertEquals( 12, theSegment.getLength() );

		AsciizSegment theOtherSegment = asciizSegment();
		assertEquals( 1, theOtherSegment.getLength() );
		theOtherSegment.fromTransmission( theSequence );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Other segment = " + theOtherSegment.toString() );
			System.out.println( "Other schema = " + theOtherSegment.toSchema() );
		}
		assertEquals( theSegment.getPayload(), theOtherSegment.getPayload() );
		assertEquals( 12, theOtherSegment.getLength() );
	}

	@Test
	public void testAsciizSegment2() throws TransmissionException {
		AsciizSegment theSegment = asciizSegment();
		Sequence theSequence = theSegment.toSequence();

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "AllocationBodyFacade (hex) = " + theSequence.toHexString() );
			System.out.println( "AllocationBodyFacade = " + theSegment.toString() );
			System.out.println( "AbstractSchema = " + theSegment.toSchema() );
		}
		assertEquals( 1, theSegment.getLength() );

		AsciizSegment theOtherSegment = asciizSegment();

		theOtherSegment.fromTransmission( theSequence );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Other segment = " + theOtherSegment.toString() );
			System.out.println( "Other schema = " + theOtherSegment.toSchema() );
		}
		assertEquals( theSegment.getPayload(), theOtherSegment.getPayload() );
		assertEquals( 1, theOtherSegment.getLength() );
	}

	@Test
	public void testAsciizSegment3() throws TransmissionException {
		AsciizSegment theSegment = asciizSegment();
		Sequence theSequence = theSegment.toSequence();

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "AllocationBodyFacade (hex) = " + theSequence.toHexString() );
			System.out.println( "AllocationBodyFacade = " + theSegment.toString() );
			System.out.println( "AbstractSchema = " + theSegment.toSchema() );
		}

		assertEquals( 1, theSegment.getLength() );

		AsciizSegment theOtherSegment = asciizSegment();
		assertEquals( 1, theOtherSegment.getLength() );
		theOtherSegment.fromTransmission( theSequence );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Other segment = " + theOtherSegment.toString() );
			System.out.println( "Other schema = " + theOtherSegment.toSchema() );
		}
		assertEquals( theSegment.getPayload(), theOtherSegment.getPayload() );
		assertEquals( 1, theOtherSegment.getLength() );
	}

	@Test
	public void testAsciizSegment4() throws TransmissionException {
		IntSegment theIntSegment;
		AsciizSegment theAsciizSegment1;
		AsciizSegment theAsciizSegment2;
		FloatSegment theFloatSegment;
		Segment theSegment = segmentComposite( theIntSegment = intSegment( 2 ), theAsciizSegment1 = asciizSegment( "Hello world!" ), theAsciizSegment2 = asciizSegment( "Hallo Welt!" ), theFloatSegment = floatSegment( 3.14F ) );
		Sequence theSequence = theSegment.toSequence();

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "AllocationBodyFacade (hex) = " + theSequence.toHexString() );
			System.out.println( "AllocationBodyFacade = " + theSegment.toString() );
			System.out.println( "AbstractSchema = " + theSegment.toSchema() );
		}
		assertEquals( 33, theSegment.getLength() );

		IntSegment theOtherIntSegment;
		AsciizSegment theOtherAsciizSegment1;
		AsciizSegment theOtherAsciizSegment2;
		FloatSegment theOtherFloatSegment;
		Segment theOtherSegment = segmentComposite( theOtherIntSegment = intSegment(), theOtherAsciizSegment1 = asciizSegment(), theOtherAsciizSegment2 = asciizSegment(), theOtherFloatSegment = floatSegment() );

		theOtherSegment.fromTransmission( theSequence );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Other segment = " + theOtherSegment.toString() );
			System.out.println( "Other schema = " + theOtherSegment.toSchema() );
		}
		assertEquals( theIntSegment.getPayload(), theOtherIntSegment.getPayload() );
		assertEquals( theAsciizSegment1.getPayload(), theOtherAsciizSegment1.getPayload() );
		assertEquals( theAsciizSegment2.getPayload(), theOtherAsciizSegment2.getPayload() );
		assertEquals( theFloatSegment.getPayload(), theOtherFloatSegment.getPayload() );
		assertEquals( 33, theOtherSegment.getLength() );
	}

	// -------------------------------------------------------------------------
	// ASCIIZ:
	// -------------------------------------------------------------------------

	@Test
	public void testAsciizArraySegment1() throws TransmissionException {
		AsciizArraySegment theSegment = asciizArraySegment( "Hallo Welt!" );
		Sequence theSequence = theSegment.toSequence();

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "AllocationBodyFacade (hex) = " + theSequence.toHexString() );
			System.out.println( "AllocationBodyFacade = " + theSegment.toString() );
			System.out.println( "AbstractSchema = " + theSegment.toSchema() );
		}

		assertEquals( 16, theSegment.getLength() );

		AsciizArraySegment theOtherSegment = asciizArraySegment();
		assertEquals( 4, theOtherSegment.getLength() );
		theOtherSegment.fromTransmission( theSequence );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Other segment = " + theOtherSegment.toString() );
			System.out.println( "Other schema = " + theOtherSegment.toSchema() );
		}
		assertArrayEquals( theSegment.getPayload(), theOtherSegment.getPayload() );
		assertEquals( 16, theOtherSegment.getLength() );
	}

	@Test
	public void testAsciizArraySegment2() throws TransmissionException {
		AsciizArraySegment theSegment = asciizArraySegment( "Hallo Welt!", "Hello world!" );
		Sequence theSequence = theSegment.toSequence();

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "AllocationBodyFacade (hex) = " + theSequence.toHexString() );
			System.out.println( "AllocationBodyFacade = " + theSegment.toString() );
			System.out.println( "AbstractSchema = " + theSegment.toSchema() );
		}

		assertEquals( 29, theSegment.getLength() );

		AsciizArraySegment theOtherSegment = asciizArraySegment();
		assertEquals( 4, theOtherSegment.getLength() );
		theOtherSegment.fromTransmission( theSequence );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Other segment = " + theOtherSegment.toString() );
			System.out.println( "Other schema = " + theOtherSegment.toSchema() );
		}
		assertArrayEquals( theSegment.getPayload(), theOtherSegment.getPayload() );
		assertEquals( 29, theOtherSegment.getLength() );
	}

	@Test
	public void testAsciizArraySegment3() throws TransmissionException {
		AsciizArraySegment theSegment = asciizArraySegment();
		Sequence theSequence = theSegment.toSequence();

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "AllocationBodyFacade (hex) = " + theSequence.toHexString() );
			System.out.println( "AllocationBodyFacade = " + theSegment.toString() );
			System.out.println( "AbstractSchema = " + theSegment.toSchema() );
		}

		assertEquals( 4, theSegment.getLength() );

		AsciizArraySegment theOtherSegment = asciizArraySegment();
		assertEquals( 4, theOtherSegment.getLength() );
		theOtherSegment.fromTransmission( theSequence );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Other segment = " + theOtherSegment.toString() );
			System.out.println( "Other schema = " + theOtherSegment.toSchema() );
		}
		assertArrayEquals( theSegment.getPayload(), theOtherSegment.getPayload() );
		assertEquals( 4, theOtherSegment.getLength() );
	}

	@Test
	public void testAsciizArraySegment4() throws TransmissionException {
		IntSegment theIntSegment;
		AsciizArraySegment theAsciizArraySegment;
		FloatSegment theFloatSegment;
		Segment theSegment = segmentComposite( theIntSegment = intSegment( 2 ), theAsciizArraySegment = asciizArraySegment( "Hello world!" ), theFloatSegment = floatSegment( 3.14F ) );
		Sequence theSequence = theSegment.toSequence();

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "AllocationBodyFacade (hex) = " + theSequence.toHexString() );
			System.out.println( "AllocationBodyFacade = " + theSegment.toString() );
			System.out.println( "AbstractSchema = " + theSegment.toSchema() );
		}
		assertEquals( 25, theSegment.getLength() );

		IntSegment theOtherIntSegment;
		AsciizArraySegment theOtherAsciizArraySegment;
		FloatSegment theOtherFloatSegment;
		Segment theOtherSegment = segmentComposite( theOtherIntSegment = intSegment(), theOtherAsciizArraySegment = asciizArraySegment(), theOtherFloatSegment = floatSegment() );

		theOtherSegment.fromTransmission( theSequence );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Other segment = " + theOtherSegment.toString() );
			System.out.println( "Other schema = " + theOtherSegment.toSchema() );
		}
		assertEquals( theIntSegment.getPayload(), theOtherIntSegment.getPayload() );
		assertArrayEquals( theAsciizArraySegment.getPayload(), theOtherAsciizArraySegment.getPayload() );
		assertEquals( theFloatSegment.getPayload(), theOtherFloatSegment.getPayload() );
		assertEquals( 25, theOtherSegment.getLength() );
	}

	@Test
	public void testAsciizArraySegment5() throws TransmissionException {
		IntSegment theIntSegment;
		AsciizArraySegment theAsciizArraySegment;
		FloatSegment theFloatSegment;
		Segment theSegment = segmentComposite( theIntSegment = intSegment( 2 ), theAsciizArraySegment = asciizArraySegment( "Hello world!", "Hallo Welt!" ), theFloatSegment = floatSegment( 3.14F ) );
		Sequence theSequence = theSegment.toSequence();

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "AllocationBodyFacade (hex) = " + theSequence.toHexString() );
			System.out.println( "AllocationBodyFacade = " + theSegment.toString() );
			System.out.println( "AbstractSchema = " + theSegment.toSchema() );
		}
		assertEquals( 37, theSegment.getLength() );

		IntSegment theOtherIntSegment;
		AsciizArraySegment theOtherAsciizArraySegment;
		FloatSegment theOtherFloatSegment;
		Segment theOtherSegment = segmentComposite( theOtherIntSegment = intSegment(), theOtherAsciizArraySegment = asciizArraySegment(), theOtherFloatSegment = floatSegment() );

		theOtherSegment.fromTransmission( theSequence );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Other segment = " + theOtherSegment.toString() );
			System.out.println( "Other schema = " + theOtherSegment.toSchema() );
		}
		assertEquals( theIntSegment.getPayload(), theOtherIntSegment.getPayload() );
		assertArrayEquals( theAsciizArraySegment.getPayload(), theOtherAsciizArraySegment.getPayload() );
		assertEquals( theFloatSegment.getPayload(), theOtherFloatSegment.getPayload() );
		assertEquals( 37, theOtherSegment.getLength() );
	}

	@Test
	public void testAsciizArraySegment6() throws TransmissionException {
		IntSegment theIntSegment;
		AsciizArraySegment theAsciizArraySegment;
		FloatSegment theFloatSegment;
		Segment theSegment = segmentComposite( theIntSegment = intSegment( 2 ), theAsciizArraySegment = asciizArraySegment(), theFloatSegment = floatSegment( 3.14F ) );
		Sequence theSequence = theSegment.toSequence();

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "AllocationBodyFacade (hex) = " + theSequence.toHexString() );
			System.out.println( "AllocationBodyFacade = " + theSegment.toString() );
			System.out.println( "AbstractSchema = " + theSegment.toSchema() );
		}
		assertEquals( 12, theSegment.getLength() );

		IntSegment theOtherIntSegment;
		AsciizArraySegment theOtherAsciizArraySegment;
		FloatSegment theOtherFloatSegment;
		Segment theOtherSegment = segmentComposite( theOtherIntSegment = intSegment(), theOtherAsciizArraySegment = asciizArraySegment(), theOtherFloatSegment = floatSegment() );

		theOtherSegment.fromTransmission( theSequence );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Other segment = " + theOtherSegment.toString() );
			System.out.println( "Other schema = " + theOtherSegment.toSchema() );
		}
		assertEquals( theIntSegment.getPayload(), theOtherIntSegment.getPayload() );
		assertArrayEquals( theAsciizArraySegment.getPayload(), theOtherAsciizArraySegment.getPayload() );
		assertEquals( theFloatSegment.getPayload(), theOtherFloatSegment.getPayload() );
		assertEquals( 12, theOtherSegment.getLength() );
	}

	@Test
	public void testAsciizArraySegment7() throws TransmissionException {
		IntSegment theIntSegment;
		AsciizArraySegment theAsciizArraySegment;
		FloatSegment theFloatSegment;
		Segment theSegment = segmentComposite( theIntSegment = intSegment( 2 ), theAsciizArraySegment = asciizArraySegment( 1, Endianess.BIG, "Hello world!", "Hallo Welt!" ), theFloatSegment = floatSegment( 3.14F ) );
		Sequence theSequence = theSegment.toSequence();

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "AllocationBodyFacade (hex) = " + theSequence.toHexString() );
			System.out.println( "AllocationBodyFacade = " + theSegment.toString() );
			System.out.println( "AbstractSchema = " + theSegment.toSchema() );
		}
		assertEquals( 34, theSegment.getLength() );

		IntSegment theOtherIntSegment;
		AsciizArraySegment theOtherAsciizArraySegment;
		FloatSegment theOtherFloatSegment;
		Segment theOtherSegment = segmentComposite( theOtherIntSegment = intSegment(), theOtherAsciizArraySegment = asciizArraySegment( 1, Endianess.BIG ), theOtherFloatSegment = floatSegment() );

		theOtherSegment.fromTransmission( theSequence );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Other segment = " + theOtherSegment.toString() );
			System.out.println( "Other schema = " + theOtherSegment.toSchema() );
		}
		assertEquals( theIntSegment.getPayload(), theOtherIntSegment.getPayload() );
		assertArrayEquals( theAsciizArraySegment.getPayload(), theOtherAsciizArraySegment.getPayload() );
		assertEquals( theFloatSegment.getPayload(), theOtherFloatSegment.getPayload() );
		assertEquals( 34, theOtherSegment.getLength() );
	}

	// -------------------------------------------------------------------------
	// Boolean:
	// -------------------------------------------------------------------------

	/**
	 * Test boolean segment.
	 *
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testBooleanSegment() throws TransmissionException {
		for ( boolean eBool : new boolean[] { true, false } ) {
			// @formatter:off
			LengthSegmentDecoratorSegment<BooleanSegment> theSegment = lengthSegment( 
				booleanSegment( eBool )
			);
			// @formatter:on

			Sequence theSequence = theSegment.toSequence();

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
				System.out.println( "Transmission = " + theSegment.toString() );
				System.out.println( "AbstractSchema = " + theSegment.toSchema() );
			}

			// @formatter:off
			LengthSegmentDecoratorSegment<BooleanSegment> theOtherSegment = lengthSegment( 
					booleanSegment()
			);
			// @formatter:on

			theOtherSegment.fromTransmission( theSequence );

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Other segment = " + theOtherSegment.toString() );
				System.out.println( "Other schema = " + theOtherSegment.toSchema() );
			}

			assertEquals( theSegment.getDecoratee().getPayload(), theOtherSegment.getDecoratee().getPayload() );
		}
	}

	// -------------------------------------------------------------------------
	// Boolean array:
	// -------------------------------------------------------------------------

	/**
	 * Test boolean array.
	 *
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testBooleanArray() throws TransmissionException {
		for ( int i = 0; i < 1024; i++ ) {
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println();
				System.out.println( "---------- Testing for <" + i + "> random boolean digits ----------" );
			}

			// @formatter:off
			AllocSectionDecoratorSegment<BooleanArraySection> theSegment = allocSegment(  
				booleanArraySection( toRndBooleanArray(i))
			);
			// @formatter:on

			Sequence theSequence = theSegment.toSequence();

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
				System.out.println( "Transmission = " + theSegment.toString() );
				System.out.println( "AbstractSchema = " + theSegment.toSchema() );
			}

			// @formatter:off
			AllocSectionDecoratorSegment<BooleanArraySection> theOtherSegment = allocSegment(  
				booleanArraySection()
			);
			// @formatter:on

			theOtherSegment.fromTransmission( theSequence );

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Other segment = " + theOtherSegment.toString() );
				System.out.println( "Other schema = " + theOtherSegment.toSchema() );
			}

			assertArrayEquals( theSegment.getDecoratee().getPayload(), theOtherSegment.getDecoratee().getPayload() );
		}
	}

	private boolean[] toRndBooleanArray( int aLength ) {
		boolean[] theResult = new boolean[aLength];
		for ( int i = 0; i < theResult.length; i++ ) {
			theResult[i] = Math.random() < 0.5;
		}
		return theResult;
	}

	// -------------------------------------------------------------------------
	// Byte:
	// -------------------------------------------------------------------------

	/**
	 * Test byte segment.
	 *
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testByteSegment() throws TransmissionException {
		for ( int i = Byte.MIN_VALUE; i <= Byte.MAX_VALUE; i++ ) {
			// @formatter:off
			LengthSegmentDecoratorSegment<ByteSegment> theSegment = lengthSegment( 
				byteSegment( (byte) i )
			);
			// @formatter:on

			Sequence theSequence = theSegment.toSequence();

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
				System.out.println( "Transmission = " + theSegment.toString() );
				System.out.println( "AbstractSchema = " + theSegment.toSchema() );
			}

			// @formatter:off
			LengthSegmentDecoratorSegment<ByteSegment> theOtherSegment = lengthSegment( 
				byteSegment()
			);
			// @formatter:on

			theOtherSegment.fromTransmission( theSequence );

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Other segment = " + theOtherSegment.toString() );
				System.out.println( "Other schema = " + theOtherSegment.toSchema() );
			}

			assertEquals( theSegment.getDecoratee().getPayload(), theOtherSegment.getDecoratee().getPayload() );
		}
	}

	// -------------------------------------------------------------------------
	// Byte array:
	// -------------------------------------------------------------------------

	/**
	 * Test byte array.
	 *
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testByteArray() throws TransmissionException {
		for ( int i = 0; i < 1024; i++ ) {
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println();
				System.out.println( "---------- Testing for <" + i + "> random byte digits ----------" );
			}

			// @formatter:off
			AllocSectionDecoratorSegment<ByteArraySection> theSegment = allocSegment(  
				byteArraySection( toRndByteArray(i))
			);
			// @formatter:on

			Sequence theSequence = theSegment.toSequence();

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
				System.out.println( "Transmission = " + theSegment.toString() );
				System.out.println( "AbstractSchema = " + theSegment.toSchema() );
			}

			// @formatter:off
			AllocSectionDecoratorSegment<ByteArraySection> theOtherSegment = allocSegment(  
				byteArraySection()
			);
			// @formatter:on

			theOtherSegment.fromTransmission( theSequence );

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Other segment = " + theOtherSegment.toString() );
				System.out.println( "Other schema = " + theOtherSegment.toSchema() );
			}

			assertArrayEquals( theSegment.getDecoratee().getPayload(), theOtherSegment.getDecoratee().getPayload() );
		}
	}

	private byte[] toRndByteArray( int aLength ) {
		Random theRnd = new Random();
		byte[] theResult = new byte[aLength];
		theRnd.nextBytes( theResult );
		return theResult;
	}

	/**
	 * Test fixed length byte array.
	 *
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testFixedLengthByteArray() throws TransmissionException {
		for ( int i = 0; i < 1024; i++ ) {
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println();
				System.out.println( "---------- Testing for <" + i + "> random byte digits ----------" );
			}

			// @formatter:off
			AllocSectionDecoratorSegment<FixedSegmentArraySection<?>> theSegment = allocSegment(  
				fixedSegmentArraySection( toRndByteSegmentArray(i))
			);
			// @formatter:on

			Sequence theSequence = theSegment.toSequence();

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
				System.out.println( "Transmission = " + theSegment.toString() );
			}

			// @formatter:off
			AllocSectionDecoratorSegment<FixedSegmentArraySection<?>> theOtherSegment = allocSegment(  
				fixedSegmentArraySection( ByteSegment.class )
			);
			// @formatter:on

			theOtherSegment.fromTransmission( theSequence );

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Other segment = " + theOtherSegment.toString() );
			}

			assertArrayEquals( theSegment.getDecoratee().getArray(), theOtherSegment.getDecoratee().getArray() );

			// -----------------------------------------------------------------

			// @formatter:off
			AllocSectionDecoratorSegment<ByteArraySection> theReferenceSegment = allocSegment(  
				byteArraySection()
			);
			// @formatter:on

			theReferenceSegment.fromTransmission( theOtherSegment.toSequence() );
			Sequence theReferenceSequence = theReferenceSegment.toSequence();

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Reference sequence (hex) = " + theReferenceSequence.toHexString( ", " ) );
				System.out.println( "Reference segment = " + theReferenceSegment.toString() );
			}

			assertArrayEquals( theSegment.toSequence().toBytes(), theReferenceSegment.toSequence().toBytes() );

		}
	}

	private ByteSegment[] toRndByteSegmentArray( int aLength ) {
		ByteSegment[] theByteSegmentArray = new ByteSegment[aLength];
		byte[] theBytes = toRndByteArray( aLength );
		for ( int i = 0; i < theByteSegmentArray.length; i++ ) {
			theByteSegmentArray[i] = new ByteSegment( theBytes[i] );
		}
		return theByteSegmentArray;
	}

	// -------------------------------------------------------------------------
	// Short:
	// -------------------------------------------------------------------------

	/**
	 * Test short segment.
	 *
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testShortSegment() throws TransmissionException {
		for ( Endianess eEndianess : Endianess.values() ) {
			for ( int i = Short.MIN_VALUE; i <= Short.MAX_VALUE; i++ ) {
				// @formatter:off
				LengthSegmentDecoratorSegment<ShortSegment> theSegment = lengthSegment( 
					shortSegment( (short) i, eEndianess )
				);
				// @formatter:on

				Sequence theSequence = theSegment.toSequence();

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
					System.out.println( "Transmission = " + theSegment.toString() );
				}

				// @formatter:off
				LengthSegmentDecoratorSegment<ShortSegment> theOtherSegment = lengthSegment( 
					shortSegment( eEndianess )
				);
				// @formatter:on

				theOtherSegment.fromTransmission( theSequence );

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Other segment = " + theOtherSegment.toString() );
				}

				assertEquals( theSegment.getDecoratee().getPayload(), theOtherSegment.getDecoratee().getPayload() );
			}
		}
	}

	// -------------------------------------------------------------------------
	// Short array:
	// -------------------------------------------------------------------------

	/**
	 * Test short array.
	 *
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testShortArray() throws TransmissionException {
		for ( int i = 0; i < 1024; i++ ) {
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println();
				System.out.println( "---------- Testing for <" + i + "> random short digits ----------" );
			}

			// @formatter:off
			AllocSectionDecoratorSegment<ShortArraySection> theSegment = allocSegment(  
				shortArraySection( toRndShortArray(i))
			);
			// @formatter:on

			Sequence theSequence = theSegment.toSequence();

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
				System.out.println( "Transmission = " + theSegment.toString() );
			}

			// @formatter:off
			AllocSectionDecoratorSegment<ShortArraySection> theOtherSegment = allocSegment(  
				shortArraySection()
			);
			// @formatter:on

			theOtherSegment.fromTransmission( theSequence );

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Other segment = " + theOtherSegment.toString() );
			}

			assertArrayEquals( theSegment.getDecoratee().getPayload(), theOtherSegment.getDecoratee().getPayload() );
		}
	}

	private short[] toRndShortArray( int aLength ) {
		Random theRnd = new Random();
		short[] theResult = new short[aLength];
		for ( int i = 0; i < theResult.length; i++ ) {
			theResult[i] = (short) (theRnd.nextBoolean() ? theRnd.nextFloat() * Short.MAX_VALUE : theRnd.nextFloat() * Short.MIN_VALUE);
		}
		return theResult;
	}

	/**
	 * Test fixed length short array.
	 *
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testFixedLengthShortArray() throws TransmissionException {
		for ( int i = 0; i < 1024; i++ ) {
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println();
				System.out.println( "---------- Testing for <" + i + "> random short digits ----------" );
			}

			// @formatter:off
				AllocSectionDecoratorSegment<FixedSegmentArraySection<?>> theSegment = allocSegment(  
					fixedSegmentArraySection( toRndShortSegmentArray(i))
				);
				// @formatter:on

			Sequence theSequence = theSegment.toSequence();

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
				System.out.println( "Transmission = " + theSegment.toString() );
			}

			// @formatter:off
				AllocSectionDecoratorSegment<FixedSegmentArraySection<?>> theOtherSegment = allocSegment(  
					fixedSegmentArraySection( ShortSegment.class )
				);
				// @formatter:on

			theOtherSegment.fromTransmission( theSequence );

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Other segment = " + theOtherSegment.toString() );
			}

			assertArrayEquals( theSegment.getDecoratee().getArray(), theOtherSegment.getDecoratee().getArray() );

			// -----------------------------------------------------------------

			// @formatter:off
				AllocSectionDecoratorSegment<ShortArraySection> theReferenceSegment = allocSegment(  
					shortArraySection()
				);
				// @formatter:on

			theReferenceSegment.fromTransmission( theOtherSegment.toSequence() );
			Sequence theReferenceSequence = theReferenceSegment.toSequence();

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Reference sequence (hex) = " + theReferenceSequence.toHexString( ", " ) );
				System.out.println( "Reference segment = " + theReferenceSegment.toString() );
			}

			assertArrayEquals( theSegment.toSequence().toBytes(), theReferenceSegment.toSequence().toBytes() );

		}
	}

	private ShortSegment[] toRndShortSegmentArray( int aLength ) {
		ShortSegment[] theShortSegmentArray = new ShortSegment[aLength];
		short[] theShorts = toRndShortArray( aLength );
		for ( int i = 0; i < theShortSegmentArray.length; i++ ) {
			theShortSegmentArray[i] = new ShortSegment( theShorts[i] );
		}
		return theShortSegmentArray;
	}

	// -------------------------------------------------------------------------
	// Integer:
	// -------------------------------------------------------------------------

	/**
	 * Test int segment.
	 *
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testIntSegment() throws TransmissionException {
		int l;
		for ( Endianess eEndianess : Endianess.values() ) {
			for ( int i = Short.MIN_VALUE; i <= Short.MAX_VALUE; i++ ) {
				l = (i == Short.MIN_VALUE ? Integer.MIN_VALUE : (i == Short.MAX_VALUE ? Integer.MAX_VALUE : i));
				// @formatter:off
				LengthSegmentDecoratorSegment<IntSegment> theSegment = lengthSegment( 
					intSegment( l, eEndianess )
				);
				// @formatter:on

				Sequence theSequence = theSegment.toSequence();

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
					System.out.println( "Transmission = " + theSegment.toString() );
				}

				// @formatter:off
				LengthSegmentDecoratorSegment<IntSegment> theOtherSegment = lengthSegment( 
					intSegment( eEndianess )
				);
				// @formatter:on

				theOtherSegment.fromTransmission( theSequence );

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Other segment = " + theOtherSegment.toString() );
				}

				assertEquals( theSegment.getDecoratee().getPayload(), theOtherSegment.getDecoratee().getPayload() );
			}
		}
	}

	// -------------------------------------------------------------------------
	// Integer array:
	// -------------------------------------------------------------------------

	/**
	 * Test int array.
	 *
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testIntArray() throws TransmissionException {
		for ( int i = 0; i < 1024; i++ ) {
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println();
				System.out.println( "---------- Testing for <" + i + "> random int digits ----------" );
			}

			// @formatter:off
			AllocSectionDecoratorSegment<IntArraySection> theSegment = allocSegment(  
				intArraySection( toRndIntArray(i))
			);
			// @formatter:on

			Sequence theSequence = theSegment.toSequence();

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
				System.out.println( "Transmission = " + theSegment.toString() );
			}

			// @formatter:off
			AllocSectionDecoratorSegment<IntArraySection> theOtherSegment = allocSegment(  
				intArraySection()
			);
			// @formatter:on

			theOtherSegment.fromTransmission( theSequence );

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Other segment = " + theOtherSegment.toString() );
			}

			assertArrayEquals( theSegment.getDecoratee().getPayload(), theOtherSegment.getDecoratee().getPayload() );
		}
	}

	private int[] toRndIntArray( int aLength ) {
		Random theRnd = new Random();
		int[] theResult = new int[aLength];
		for ( int i = 0; i < theResult.length; i++ ) {
			theResult[i] = theRnd.nextInt();
		}
		return theResult;
	}

	/**
	 * Test fixed length int array.
	 *
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testFixedLengthIntArray() throws TransmissionException {
		for ( int i = 0; i < 1024; i++ ) {
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println();
				System.out.println( "---------- Testing for <" + i + "> random int digits ----------" );
			}

			// @formatter:off
			AllocSectionDecoratorSegment<FixedSegmentArraySection<?>> theSegment = allocSegment(  
				fixedSegmentArraySection( toRndIntSegmentArray(i))
			);
			// @formatter:on

			Sequence theSequence = theSegment.toSequence();

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
				System.out.println( "Transmission = " + theSegment.toString() );
			}

			// @formatter:off
			AllocSectionDecoratorSegment<FixedSegmentArraySection<?>> theOtherSegment = allocSegment(  
				fixedSegmentArraySection( IntSegment.class )
			);
			// @formatter:on

			theOtherSegment.fromTransmission( theSequence );

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Other segment = " + theOtherSegment.toString() );
			}

			assertArrayEquals( theSegment.getDecoratee().getArray(), theOtherSegment.getDecoratee().getArray() );

			// -----------------------------------------------------------------

			// @formatter:off
			AllocSectionDecoratorSegment<IntArraySection> theReferenceSegment = allocSegment(  
				intArraySection()
			);
			// @formatter:on

			theReferenceSegment.fromTransmission( theOtherSegment.toSequence() );
			Sequence theReferenceSequence = theReferenceSegment.toSequence();

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Reference sequence (hex) = " + theReferenceSequence.toHexString( ", " ) );
				System.out.println( "Reference segment = " + theReferenceSegment.toString() );
			}

			assertArrayEquals( theSegment.toSequence().toBytes(), theReferenceSegment.toSequence().toBytes() );

		}
	}

	private IntSegment[] toRndIntSegmentArray( int aLength ) {
		IntSegment[] theIntSegmentArray = new IntSegment[aLength];
		int[] theIntegers = toRndIntArray( aLength );
		for ( int i = 0; i < theIntSegmentArray.length; i++ ) {
			theIntSegmentArray[i] = new IntSegment( theIntegers[i] );
		}
		return theIntSegmentArray;
	}

	// -------------------------------------------------------------------------
	// Long:
	// -------------------------------------------------------------------------

	/**
	 * Test long segment.
	 *
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testLongSegment() throws TransmissionException {
		long l;
		for ( Endianess eEndianess : Endianess.values() ) {
			for ( int i = Short.MIN_VALUE; i <= Short.MAX_VALUE; i++ ) {
				l = (i == Short.MIN_VALUE ? Long.MIN_VALUE : (i == Short.MAX_VALUE ? Long.MAX_VALUE : i));
				// @formatter:off
				LengthSegmentDecoratorSegment<LongSegment> theSegment = lengthSegment( 
					longSegment( l, eEndianess )
				);
				// @formatter:on

				Sequence theSequence = theSegment.toSequence();

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
					System.out.println( "Transmission = " + theSegment.toString() );
				}

				// @formatter:off
				LengthSegmentDecoratorSegment<LongSegment> theOtherSegment = lengthSegment( 
					longSegment( eEndianess )
				);
				// @formatter:on

				theOtherSegment.fromTransmission( theSequence );

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Other segment = " + theOtherSegment.toString() );
				}

				assertEquals( theSegment.getDecoratee().getPayload(), theOtherSegment.getDecoratee().getPayload() );
			}
		}
	}

	// -------------------------------------------------------------------------
	// Long array:
	// -------------------------------------------------------------------------

	/**
	 * Test long array.
	 *
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testLongArray() throws TransmissionException {
		for ( int i = 0; i < 1024; i++ ) {
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println();
				System.out.println( "---------- Testing for <" + i + "> random long digits ----------" );
			}

			// @formatter:off
			AllocSectionDecoratorSegment<LongArraySection> theSegment = allocSegment(  
				longArraySection( toRndLongArray(i))
			);
			// @formatter:on

			Sequence theSequence = theSegment.toSequence();

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
				System.out.println( "Transmission = " + theSegment.toString() );
			}

			// @formatter:off
			AllocSectionDecoratorSegment<LongArraySection> theOtherSegment = allocSegment(  
				longArraySection()
			);
			// @formatter:on

			theOtherSegment.fromTransmission( theSequence );

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Other segment = " + theOtherSegment.toString() );
			}

			assertArrayEquals( theSegment.getDecoratee().getPayload(), theOtherSegment.getDecoratee().getPayload() );
		}
	}

	private long[] toRndLongArray( int aLength ) {
		Random theRnd = new Random();
		long[] theResult = new long[aLength];
		for ( int i = 0; i < theResult.length; i++ ) {
			theResult[i] = theRnd.nextLong();
		}
		return theResult;
	}

	/**
	 * Test fixed length long array.
	 *
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testFixedLengthLongArray() throws TransmissionException {
		for ( int i = 0; i < 1024; i++ ) {
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println();
				System.out.println( "---------- Testing for <" + i + "> random long digits ----------" );
			}

			// @formatter:off
			AllocSectionDecoratorSegment<FixedSegmentArraySection<?>> theSegment = allocSegment(  
				fixedSegmentArraySection( toRndLongSegmentArray(i))
			);
			// @formatter:on

			Sequence theSequence = theSegment.toSequence();

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
				System.out.println( "Transmission = " + theSegment.toString() );
			}

			// @formatter:off
			AllocSectionDecoratorSegment<FixedSegmentArraySection<?>> theOtherSegment = allocSegment(  
				fixedSegmentArraySection( LongSegment.class )
			);
			// @formatter:on

			theOtherSegment.fromTransmission( theSequence );

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Other segment = " + theOtherSegment.toString() );
			}

			assertArrayEquals( theSegment.getDecoratee().getArray(), theOtherSegment.getDecoratee().getArray() );

			// -----------------------------------------------------------------

			// @formatter:off
			AllocSectionDecoratorSegment<LongArraySection> theReferenceSegment = allocSegment(  
				longArraySection()
			);
			// @formatter:on

			theReferenceSegment.fromTransmission( theOtherSegment.toSequence() );
			Sequence theReferenceSequence = theReferenceSegment.toSequence();

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Reference sequence (hex) = " + theReferenceSequence.toHexString( ", " ) );
				System.out.println( "Reference segment = " + theReferenceSegment.toString() );
			}

			assertArrayEquals( theSegment.toSequence().toBytes(), theReferenceSegment.toSequence().toBytes() );

		}
	}

	private LongSegment[] toRndLongSegmentArray( int aLength ) {
		LongSegment[] theLongSegmentArray = new LongSegment[aLength];
		long[] theLongs = toRndLongArray( aLength );
		for ( int i = 0; i < theLongSegmentArray.length; i++ ) {
			theLongSegmentArray[i] = new LongSegment( theLongs[i] );
		}
		return theLongSegmentArray;
	}

	// -------------------------------------------------------------------------
	// Float:
	// -------------------------------------------------------------------------

	/**
	 * Test float segment.
	 *
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testFloatSegment() throws TransmissionException {
		float l;
		for ( Endianess eEndianess : Endianess.values() ) {
			for ( int i = Short.MIN_VALUE; i <= Short.MAX_VALUE; i++ ) {
				l = (i == Short.MIN_VALUE ? Float.MIN_VALUE : (i == Short.MAX_VALUE ? Float.MAX_VALUE : i * (float) Math.PI));
				if ( IS_LOG_TEST_ENABLED ) {
					System.out.print( "Endianess = " + eEndianess + ", value= " );
					System.out.printf( "%f%n", l );
				}

				// @formatter:off
				LengthSegmentDecoratorSegment<FloatSegment> theSegment = lengthSegment( 
					floatSegment( l, eEndianess )
				);
				// @formatter:on

				Sequence theSequence = theSegment.toSequence();

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
					System.out.println( "Transmission = " + theSegment.toString() );
				}

				// @formatter:off
				LengthSegmentDecoratorSegment<FloatSegment> theOtherSegment = lengthSegment( 
					floatSegment( eEndianess )
				);
				// @formatter:on

				theOtherSegment.fromTransmission( theSequence );

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Other sequence (hex) = " + theOtherSegment.toSequence().toHexString( ", " ) );
					System.out.println( "Other segment = " + theOtherSegment.toString() );
				}

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "theSegment.value = " + theSegment.getDecoratee().getPayload() );
					System.out.println( "theOtherSegment.value = " + theOtherSegment.getDecoratee().getPayload() );
				}

				assertEquals( theSegment.getDecoratee().getPayload(), theOtherSegment.getDecoratee().getPayload() );
			}
		}
	}

	// -------------------------------------------------------------------------
	// Float array:
	// -------------------------------------------------------------------------

	@Test
	public void testFloatArray() throws TransmissionException {
		for ( int i = 0; i < 1024; i++ ) {
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println();
				System.out.println( "---------- Testing for <" + i + "> random float digits ----------" );
			}

			// @formatter:off
			AllocSectionDecoratorSegment<FloatArraySection> theSegment = allocSegment(  
				floatArraySection( toRndFloatArray(i))
			);
			// @formatter:on

			Sequence theSequence = theSegment.toSequence();

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
				System.out.println( "Transmission = " + theSegment.toString() );
			}

			// @formatter:off
			AllocSectionDecoratorSegment<FloatArraySection> theOtherSegment = allocSegment(  
				floatArraySection()
			);
			// @formatter:on

			theOtherSegment.fromTransmission( theSequence );

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Other segment = " + theOtherSegment.toString() );
			}

			assertArrayEquals( theSegment.getDecoratee().getPayload(), theOtherSegment.getDecoratee().getPayload() );
		}
	}

	private float[] toRndFloatArray( int aLength ) {
		Random theRnd = new Random();
		float[] theResult = new float[aLength];
		for ( int i = 0; i < theResult.length; i++ ) {
			theResult[i] = theRnd.nextBoolean() ? theRnd.nextFloat() * Float.MAX_VALUE : theRnd.nextFloat() * Float.MIN_VALUE * -1;
		}
		return theResult;
	}

	@Test
	public void testFixedLengthFloatArray() throws TransmissionException {
		for ( int i = 0; i < 1024; i++ ) {
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println();
				System.out.println( "---------- Testing for <" + i + "> random float digits ----------" );
			}

			// @formatter:off
			AllocSectionDecoratorSegment<FixedSegmentArraySection<?>> theSegment = allocSegment(  
				fixedSegmentArraySection( toRndFloatSegmentArray(i))
			);
			// @formatter:on

			Sequence theSequence = theSegment.toSequence();

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
				System.out.println( "Transmission = " + theSegment.toString() );
			}

			// @formatter:off
			AllocSectionDecoratorSegment<FixedSegmentArraySection<?>> theOtherSegment = allocSegment(  
				fixedSegmentArraySection( FloatSegment.class )
			);
			// @formatter:on

			theOtherSegment.fromTransmission( theSequence );

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Other segment = " + theOtherSegment.toString() );
			}

			assertArrayEquals( theSegment.getDecoratee().getArray(), theOtherSegment.getDecoratee().getArray() );

			// -----------------------------------------------------------------

			// @formatter:off
			AllocSectionDecoratorSegment<FloatArraySection> theReferenceSegment = allocSegment(  
				floatArraySection()
			);
			// @formatter:on

			theReferenceSegment.fromTransmission( theOtherSegment.toSequence() );
			Sequence theReferenceSequence = theReferenceSegment.toSequence();

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Reference sequence (hex) = " + theReferenceSequence.toHexString( ", " ) );
				System.out.println( "Reference segment = " + theReferenceSegment.toString() );
			}

			assertArrayEquals( theSegment.toSequence().toBytes(), theReferenceSegment.toSequence().toBytes() );

		}
	}

	private FloatSegment[] toRndFloatSegmentArray( int aLength ) {
		FloatSegment[] theFloatSegmentArray = new FloatSegment[aLength];
		float[] theFloats = toRndFloatArray( aLength );
		for ( int i = 0; i < theFloatSegmentArray.length; i++ ) {
			theFloatSegmentArray[i] = new FloatSegment( theFloats[i] );
		}
		return theFloatSegmentArray;
	}

	// -------------------------------------------------------------------------
	// Double:
	// -------------------------------------------------------------------------

	@Test
	public void testDoubleSegment() throws TransmissionException {
		double l;
		for ( Endianess eEndianess : Endianess.values() ) {
			for ( int i = Short.MIN_VALUE; i <= Short.MAX_VALUE; i++ ) {
				l = (i == Short.MIN_VALUE ? Double.MIN_VALUE : (i == Short.MAX_VALUE ? Double.MAX_VALUE : i * (double) Math.PI));
				// @formatter:off
				LengthSegmentDecoratorSegment<DoubleSegment> theSegment = lengthSegment( 
					doubleSegment( l, eEndianess )
				);
				// @formatter:on

				Sequence theSequence = theSegment.toSequence();

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
					System.out.println( "Transmission = " + theSegment.toString() );
				}

				// @formatter:off
				LengthSegmentDecoratorSegment<DoubleSegment> theOtherSegment = lengthSegment( 
					doubleSegment( eEndianess )
				);
				// @formatter:on

				theOtherSegment.fromTransmission( theSequence );

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Other segment = " + theOtherSegment.toString() );
				}

				assertEquals( theSegment.getDecoratee().getPayload(), theOtherSegment.getDecoratee().getPayload() );
			}
		}
	}

	// -------------------------------------------------------------------------
	// Double array:
	// -------------------------------------------------------------------------

	@Test
	public void testDoubleArray() throws TransmissionException {
		for ( int i = 0; i < 1024; i++ ) {
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println();
				System.out.println( "---------- Testing for <" + i + "> random double digits ----------" );
			}

			// @formatter:off
			AllocSectionDecoratorSegment<DoubleArraySection> theSegment = allocSegment(  
				doubleArraySection( toRndDoubleArray(i))
			);
			// @formatter:on

			Sequence theSequence = theSegment.toSequence();

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
				System.out.println( "Transmission = " + theSegment.toString() );
			}

			// @formatter:off
			AllocSectionDecoratorSegment<DoubleArraySection> theOtherSegment = allocSegment(  
				doubleArraySection()
			);
			// @formatter:on

			theOtherSegment.fromTransmission( theSequence );

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Other segment = " + theOtherSegment.toString() );
			}

			assertArrayEquals( theSegment.getDecoratee().getPayload(), theOtherSegment.getDecoratee().getPayload() );
		}
	}

	private double[] toRndDoubleArray( int aLength ) {
		Random theRnd = new Random();
		double[] theResult = new double[aLength];
		for ( int i = 0; i < theResult.length; i++ ) {
			theResult[i] = theRnd.nextBoolean() ? theRnd.nextDouble() * Double.MAX_VALUE : theRnd.nextDouble() * Double.MIN_VALUE * -1;
		}
		return theResult;
	}

	@Test
	public void testFixedLengthDoubleArray() throws TransmissionException {
		for ( int i = 0; i < 1024; i++ ) {
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println();
				System.out.println( "---------- Testing for <" + i + "> random double digits ----------" );
			}

			// @formatter:off
			AllocSectionDecoratorSegment<FixedSegmentArraySection<?>> theSegment = allocSegment(  
				fixedSegmentArraySection( toRndDoubleSegmentArray(i))
			);
			// @formatter:on

			Sequence theSequence = theSegment.toSequence();

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
				System.out.println( "Transmission = " + theSegment.toString() );
			}

			// @formatter:off
			AllocSectionDecoratorSegment<FixedSegmentArraySection<?>> theOtherSegment = allocSegment(  
				fixedSegmentArraySection( DoubleSegment.class )
			);
			// @formatter:on

			theOtherSegment.fromTransmission( theSequence );

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Other segment = " + theOtherSegment.toString() );
			}

			assertArrayEquals( theSegment.getDecoratee().getArray(), theOtherSegment.getDecoratee().getArray() );

			// -----------------------------------------------------------------

			// @formatter:off
			AllocSectionDecoratorSegment<DoubleArraySection> theReferenceSegment = allocSegment(  
				doubleArraySection()
			);
			// @formatter:on

			theReferenceSegment.fromTransmission( theOtherSegment.toSequence() );
			Sequence theReferenceSequence = theReferenceSegment.toSequence();

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Reference sequence (hex) = " + theReferenceSequence.toHexString( ", " ) );
				System.out.println( "Reference segment = " + theReferenceSegment.toString() );
			}

			assertArrayEquals( theSegment.toSequence().toBytes(), theReferenceSegment.toSequence().toBytes() );

		}
	}

	private DoubleSegment[] toRndDoubleSegmentArray( int aLength ) {
		DoubleSegment[] theDoubleSegmentArray = new DoubleSegment[aLength];
		double[] theDoubles = toRndDoubleArray( aLength );
		for ( int i = 0; i < theDoubleSegmentArray.length; i++ ) {
			theDoubleSegmentArray[i] = new DoubleSegment( theDoubles[i] );
		}
		return theDoubleSegmentArray;
	}

	//	// -------------------------------------------------------------------------
	//	// String:
	// -------------------------------------------------------------------------

	@Test
	public void testStringSegment() throws TransmissionException {

		// @formatter:off
		AllocSectionDecoratorSegment<StringSection> theSegment = allocSegment( 
			stringSection("Hallo Welt!")
		);
		// @formatter:on

		Sequence theSequence = theSegment.toSequence();

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "AllocationBodyFacade (hex) = " + theSequence.toHexString() );
			System.out.println( "AllocationBodyFacade = " + theSegment.toString() );
			System.out.println( "AbstractSchema = " + theSegment.toSchema() );
		}

		// @formatter:off
		AllocSectionDecoratorSegment<StringSection> theOtherSegment = allocSegment( 
			stringSection()
		);
		// @formatter:on

		theOtherSegment.fromTransmission( theSequence );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Other segment = " + theOtherSegment.toString() );
			System.out.println( "Other schema = " + theOtherSegment.toSchema() );
		}

		assertEquals( theSegment.getDecoratee().getPayload(), theOtherSegment.getDecoratee().getPayload() );
	}

	// -------------------------------------------------------------------------
	// String array:
	// -------------------------------------------------------------------------

	@Test
	public void testStringArrayBody() throws TransmissionException {

		// @formatter:off
		AllocSectionDecoratorSegment<StringArraySection> theSegment = allocSegment( 
			stringArraySection("Hallo Welt!", "Hello World!", "สวัสดีจักรวาล!")
		);
		// @formatter:on

		Sequence theSequence = theSegment.toSequence();

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Sequence (hex) = " + theSequence.toHexString() );
			System.out.println( "Transmission = " + theSegment.toString() );
			System.out.println( "AbstractSchema = " + theSegment.toSchema() );
		}

		// @formatter:off
		AllocSectionDecoratorSegment<StringArraySection> theOtherSegment = allocSegment( 
			stringArraySection()
		);
		// @formatter:on

		theOtherSegment.fromTransmission( theSequence );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Other segment = " + theOtherSegment.toString() );
			System.out.println( "Other schema = " + theOtherSegment.toSchema() );
		}

		assertArrayEquals( theSegment.getDecoratee().getPayload(), theOtherSegment.getDecoratee().getPayload() );

	}

	// -------------------------------------------------------------------------
	// Number:
	// -------------------------------------------------------------------------

	@Test
	public void testNumberSegment1() throws TransmissionException {
		long l;
		for ( Endianess eEndianess : Endianess.values() ) {
			for ( int i = Short.MIN_VALUE; i <= Short.MAX_VALUE; i++ ) {
				l = (i == Short.MIN_VALUE ? Integer.MIN_VALUE : (i == Short.MAX_VALUE ? Integer.MAX_VALUE : i));
				// @formatter:off
				LengthSegmentDecoratorSegment<NumberSegment> theSegment = lengthSegment( 
					numberSegment( Integer.BYTES, l, eEndianess )
				);
				// @formatter:on

				Sequence theSequence = theSegment.toSequence();

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
					System.out.println( "Transmission = " + theSegment.toString() );
				}

				// @formatter:off
				LengthSegmentDecoratorSegment<NumberSegment> theOtherSegment = lengthSegment( 
						numberSegment( Integer.BYTES, eEndianess )
				);
				// @formatter:on

				theOtherSegment.fromTransmission( theSequence );

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Other segment = " + theOtherSegment.toString() );
				}

				assertEquals( theSegment.getDecoratee().getPayload(), theOtherSegment.getDecoratee().getPayload() );
			}
		}
	}

	@Test
	public void testNumberSegment2() throws TransmissionException {
		long l;
		for ( Endianess eEndianess : Endianess.values() ) {
			for ( int i = Short.MIN_VALUE; i <= Short.MAX_VALUE; i++ ) {
				l = (i == Short.MIN_VALUE ? Long.MIN_VALUE : (i == Short.MAX_VALUE ? Long.MAX_VALUE : i));
				// @formatter:off
				LengthSegmentDecoratorSegment<NumberSegment> theSegment = lengthSegment( 
					numberSegment( Long.BYTES, l, eEndianess )
				);
				// @formatter:on

				Sequence theSequence = theSegment.toSequence();

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
					System.out.println( "Transmission = " + theSegment.toString() );
				}

				// @formatter:off
				LengthSegmentDecoratorSegment<NumberSegment> theOtherSegment = lengthSegment( 
						numberSegment( Long.BYTES, eEndianess )
				);
				// @formatter:on

				theOtherSegment.fromTransmission( theSequence );

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Other segment = " + theOtherSegment.toString() );
				}

				assertEquals( theSegment.getDecoratee().getPayload(), theOtherSegment.getDecoratee().getPayload() );
			}
		}
	}

	@Test
	public void testNumberSegment3() throws TransmissionException {
		long l;
		for ( Endianess eEndianess : Endianess.values() ) {
			for ( int i = Short.MIN_VALUE; i <= Short.MAX_VALUE; i++ ) {
				l = (i == Short.MIN_VALUE ? Integer.MIN_VALUE : (i == Short.MAX_VALUE ? Integer.MAX_VALUE : i));
				// @formatter:off
				LengthSegmentDecoratorSegment<NumberSegment> theSegment = lengthSegment( 
					numberSegment( Integer.BYTES + 1, l, eEndianess )
				);
				// @formatter:on

				Sequence theSequence = theSegment.toSequence();

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
					System.out.println( "Transmission = " + theSegment.toString() );
				}

				// @formatter:off
				LengthSegmentDecoratorSegment<NumberSegment> theOtherSegment = lengthSegment( 
						numberSegment( Integer.BYTES + 1, eEndianess )
				);
				// @formatter:on

				theOtherSegment.fromTransmission( theSequence );

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Other segment = " + theOtherSegment.toString() );
				}

				assertEquals( theSegment.getDecoratee().getPayload(), theOtherSegment.getDecoratee().getPayload() );
			}
		}
	}

	// -------------------------------------------------------------------------
	// EdgeCase:
	// -------------------------------------------------------------------------

	@Disabled
	@Test
	public void testEdgeCase1() throws TransmissionException {

		int i = -128;

		// @formatter:off
		LengthSegmentDecoratorSegment<ShortSegment> theSegment = lengthSegment( 
			shortSegment( (short) i )
		);
		// @formatter:on

		Sequence theSequence = theSegment.toSequence();

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
			System.out.println( "Transmission = " + theSegment.toString() );
		}

		// @formatter:off
		LengthSegmentDecoratorSegment<ShortSegment> theOtherSegment = lengthSegment( 
			shortSegment()
		);
		// @formatter:on

		theOtherSegment.fromTransmission( theSequence );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Other sequence (hex) = " + theOtherSegment.toSequence().toHexString( ", " ) );
			System.out.println( "Other segment = " + theOtherSegment.toString() );
		}

		assertEquals( theSegment.getDecoratee().getPayload(), theOtherSegment.getDecoratee().getPayload() );
	}

	@Test
	public void testEdgeCase2() throws TransmissionException {
		float l = -78131.414063F;
		Endianess theEndianess = Endianess.LITTLE;

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.print( "Endianess = " + theEndianess + ", value= " );
			System.out.printf( "%f%n", l );
		}

		// @formatter:off
		LengthSegmentDecoratorSegment<FloatSegment> theSegment = lengthSegment( 
			floatSegment( l, theEndianess )
		);
		// @formatter:on

		Sequence theSequence = theSegment.toSequence();

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
			System.out.println( "Transmission = " + theSegment.toString() );
		}

		// @formatter:off
		LengthSegmentDecoratorSegment<FloatSegment> theOtherSegment = lengthSegment( 
			floatSegment( theEndianess )
		);
		// @formatter:on

		theOtherSegment.fromTransmission( theSequence );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Other sequence (hex) = " + theOtherSegment.toSequence().toHexString( ", " ) );
			System.out.println( "Other segment = " + theOtherSegment.toString() );
		}

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.print( "theSegment.value = " );
			System.out.printf( "%f%n", theSegment.getDecoratee().getPayload() );
			System.out.print( "theOtherSegment.value = " );
			System.out.printf( "%f%n", theOtherSegment.getDecoratee().getPayload() );
		}

		assertEquals( theSegment.getDecoratee().getPayload(), theOtherSegment.getDecoratee().getPayload() );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
