// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.serial;

import static org.junit.jupiter.api.Assertions.*;
import static org.refcodes.serial.SerialSugar.*;

import java.io.File;
import java.io.IOException;
import java.util.Scanner;

import org.junit.jupiter.api.Test;
import org.refcodes.data.SystemProperty;
import org.refcodes.data.Text;

public class FileTest extends AbstractLoopbackPortTest {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final String PASSWORDS_TXT = "passwords.txt";
	private static final String LF = SystemProperty.LINE_SEPARATOR.getValue();
	private static final String EOB = LF + "-----------------------" + LF;
	private static final String SHORT_TEXT = Text.ARECIBO_MESSAGE.getText() + EOB + Text.ARECIBO_MESSAGE.getText() + EOB + Text.ARECIBO_MESSAGE.getText() + EOB + Text.ARECIBO_MESSAGE.getText() + EOB + Text.ARECIBO_MESSAGE.getText() + EOB + Text.ARECIBO_MESSAGE.getText() + EOB + Text.ARECIBO_MESSAGE.getText() + EOB + Text.ARECIBO_MESSAGE.getText() + EOB + Text.ARECIBO_MESSAGE.getText() + EOB + Text.ARECIBO_MESSAGE.getText() + EOB + Text.ARECIBO_MESSAGE.getText() + EOB + Text.ARECIBO_MESSAGE.getText() + EOB + Text.ARECIBO_MESSAGE.getText() + EOB + Text.ARECIBO_MESSAGE.getText() + EOB + Text.ARECIBO_MESSAGE.getText() + EOB;
	private static boolean IS_LOG_TEST_ENABLED = Boolean.getBoolean( "log.test");

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Test file section 1.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws InterruptedException the interrupted exception
	 */
	@Test
	public void testFileSection1() throws IOException, InterruptedException {
		if ( hasPorts() ) {
			Port<?> theTransmitPort = getTransmitterPort().withOpen();
			Port<?> theReceiverPort = getReceiverPort().withOpen();
			TransmissionMetrics theMetrics = new TransmissionMetrics();
			Segment theSendSegment = crcSegment( allocSegment( stringSection( SHORT_TEXT ) ), theMetrics );
			File theTempFile = File.createTempFile( getClass().getName(), ".tmp" );
			theTempFile.deleteOnExit();
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Before: Temp file = " + theTempFile.getAbsolutePath() );
				System.out.println( "Before: Temp file size = " + theTempFile.length() );
			}
			Segment theReceiveSegment = crcSegment( allocSegment( new FileSection( theTempFile ) ), theMetrics );
			SegmentResult<?> theResult = theReceiverPort.onReceiveSegment( theReceiveSegment );
			long start = System.currentTimeMillis();
			theTransmitPort.transmitSegment( theSendSegment );
			long durationInSecs = (System.currentTimeMillis() - start) / 1000;
			theResult.waitForResult();
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Duration in Seconds = " + durationInSecs );
				System.out.println( "Bytes/S = " + (durationInSecs != 0 ? (theReceiveSegment.getLength() / durationInSecs) : "~") );
				System.out.println( "After: Temp file = " + theTempFile.getAbsolutePath() );
				System.out.println( "After: Temp file size = " + theTempFile.length() );
				System.out.println( theReceiveSegment.toSchema() );
			}

			// Assert |-->
			String[] theReferenceText = SHORT_TEXT.split( LF );
			int i = 0;
			try (Scanner theScanner = new Scanner( theTempFile, theMetrics.getEncoding().name() )) {
				theScanner.useDelimiter( LF );
				String eFileLine;
				while ( theScanner.hasNext() ) {
					eFileLine = theScanner.next();
					if ( IS_LOG_TEST_ENABLED ) {
						System.out.println( "\"" + eFileLine + "\" [" + i + "]" );
					}
					assertEquals( theReferenceText[i], eFileLine );
					i++;
				}
			}
			assertEquals( theReferenceText.length, i );
			// Assert <--|

			// -----------------------------------------------------------------

			StringSection theStringSection;
			theSendSegment = crcSegment( allocSegment( new FileSection( theTempFile ) ), theMetrics );
			theReceiveSegment = crcSegment( allocSegment( theStringSection = stringSection() ), theMetrics );
			theResult = theReceiverPort.onReceiveSegment( theReceiveSegment );
			theTransmitPort.transmitSegment( theSendSegment );
			theResult.waitForResult();

			//	if ( IS_LOG_TEST_ENABLED ) {
			//		System.out.println( theStringSection.getPayload() );
			//	}
			assertEquals( SHORT_TEXT, theStringSection.getPayload() );

			theTransmitPort.close();
			theReceiverPort.close();
		}
	}

	/**
	 * Test file section 2.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws InterruptedException the interrupted exception
	 */
	@Test
	public void testFileSection2() throws IOException, InterruptedException {
		if ( hasPorts() ) {
			Port<?> theTransmitPort = getTransmitterPort().withOpen();
			Port<?> theReceiverPort = getReceiverPort().withOpen();

			File theTempFile = File.createTempFile( getClass().getName(), ".tmp" );
			theTempFile.deleteOnExit();
			String theBasePath = theTempFile.getParent();

			TransmissionMetrics theMetrics = new TransmissionMetrics();
			Segment theSendSegment = crcSegment( segmentComposite( allocSegment( stringSection( PASSWORDS_TXT ) ), allocSegment( stringSection( SHORT_TEXT ) ) ), theMetrics );
			final StringSection theFilenameSection;
			FileSection theFileSection;
			Segment theReceiveSegment = crcSegment( segmentComposite( allocSegment( theFilenameSection = stringSection() ), allocSegment( theFileSection = fileSection( theBasePath, () -> theFilenameSection.getPayload() ) ) ), theMetrics );

			SegmentResult<?> theResult = theReceiverPort.onReceiveSegment( theReceiveSegment );
			theTransmitPort.transmitSegment( theSendSegment );
			theResult.waitForResult();

			File thePayloadFile = theFileSection.toPayloadFile();
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( thePayloadFile.getAbsolutePath() );
			}
			thePayloadFile.deleteOnExit();
			assertTrue( thePayloadFile.getAbsolutePath().endsWith( PASSWORDS_TXT ) );

			// -----------------------------------------------------------------

			theSendSegment = crcSegment( segmentComposite( allocSegment( stringSection( PASSWORDS_TXT ) ), allocSegment( fileSection( thePayloadFile ) ) ), theMetrics );
			StringSection theStringSection;
			theReceiveSegment = crcSegment( segmentComposite( allocSegment( stringSection() ), allocSegment( theStringSection = stringSection() ) ), theMetrics );
			theResult = theReceiverPort.onReceiveSegment( theReceiveSegment );
			theTransmitPort.transmitSegment( theSendSegment );
			theResult.waitForResult();

			//	if ( IS_LOG_TEST_ENABLED ) {
			//		System.out.println( theStringSection.getPayload() );
			//	}
			assertEquals( SHORT_TEXT, theStringSection.getPayload() );

			theTransmitPort.close();
			theReceiverPort.close();
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
