// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.serial;

import static org.junit.jupiter.api.Assertions.*;
import static org.refcodes.serial.SerialSugar.*;

import java.io.IOException;

import org.junit.jupiter.api.Test;
import org.refcodes.numerical.CrcAlgorithmConfig;
import org.refcodes.numerical.Invertible;
import org.refcodes.serial.TestFixures.WeatherData;

public class InvertibleTest extends AbstractLoopbackPortTest {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	private static boolean IS_LOG_TEST_ENABLED = Boolean.getBoolean( "log.test");

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final int LOOPS = 15;

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@Test
	public void testInvertibleSimple() throws TransmissionException {
		// @formatter:off
		LengthSegmentDecoratorSegment<?> theSegment = lengthSegment( 
			allocSegment( 
				invertibleSection( 
					stringSection("Hello World!"), new InvertibleXor( (byte) 0xFE )
				)
			)
		);
		// @formatter:on

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( theSegment.toSchema() );
		}
		Sequence theSequence = theSegment.toSequence();

		// @formatter:off
		LengthSegmentDecoratorSegment<?> theOtherSegment = lengthSegment( 
				allocSegment( 
					invertibleSection( 
						stringSection(), new InvertibleXor( (byte) 0xFE )
					)
				)
			);
		// @formatter:on

		theOtherSegment.fromTransmission( theSequence );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( theOtherSegment.toSchema() );
		}

		assertEquals( theSegment, theOtherSegment );
	}

	@Test
	public void testInvertibleSimpleFail() throws TransmissionException {
		// @formatter:off
		LengthSegmentDecoratorSegment<?> theSegment = lengthSegment( 
			allocSegment( 
				invertibleSection( 
					stringSection("Hello World!"), new InvertibleXor( (byte) 0xFE )
				)
			)
		);
		// @formatter:on

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( theSegment.toSchema() );
		}
		Sequence theSequence = theSegment.toSequence();

		// @formatter:off
		LengthSegmentDecoratorSegment<?> theOtherSegment = lengthSegment( 
				allocSegment( 
					invertibleSection( 
						stringSection(), new InvertibleXor( (byte) 0xFF )
					)
				)
			);
		// @formatter:on

		theOtherSegment.fromTransmission( theSequence );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( theOtherSegment.toSchema() );
		}

		assertNotEquals( theSegment, theOtherSegment );
	}

	@Test
	public void testInvertibleComplex() throws IOException {
		if ( hasPorts() ) {
			Port<?> theTransmitPort = getTransmitterPort().withOpen();
			Port<?> theReceiverPort = getReceiverPort().withOpen();
			WeatherData theSenderData = TestFixures.createWeatherData();
			ComplexTypeSegment<WeatherData> theReceiverSegment;
			InvertibleXor theInvertibleXor;
			Segment theSenderSeg = readyToSendSegment( invertibleSegment( crcPrefixSegment( complexTypeSegment( theSenderData ), CrcAlgorithmConfig.CRC_16_CCITT_FALSE ), theInvertibleXor = new InvertibleXor( (byte) 0x10 ) ) );
			Segment theReceiverSeg = readyToSendSegment( invertibleSegment( crcPrefixSegment( theReceiverSegment = complexTypeSegment( WeatherData.class ), CrcAlgorithmConfig.CRC_16_CCITT_FALSE ), theInvertibleXor ) );

			for ( int i = 0; i < LOOPS; i++ ) {
				SegmentResult<Segment> theResult = theReceiverPort.onReceiveSegment( theReceiverSeg );
				theSenderSeg.transmitTo( theTransmitPort );
				theResult.waitForResult();

				if ( IS_LOG_TEST_ENABLED ) {
					try {
						System.out.println( "Expected [" + theInvertibleXor.bijectionXor + "]= " + theResult.getResult() );
					}
					catch ( BadCrcChecksumException e ) {
						System.out.println( "Unexpected exceptio = " + e.getMessage() );
						fail( "Not Expecting a <" + BadCrcChecksumException.class.getSimpleName() + "> instead of a resul!t" );
					}
				}

				WeatherData theReceiverData = theReceiverSegment.getPayload();
				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( theReceiverData );
				}
				assertEquals( theSenderData, theReceiverData );

				theInvertibleXor.bijectionXor++;
				theInvertibleXor.inversionXor = theInvertibleXor.bijectionXor;
			}

			theTransmitPort.close();
			theReceiverPort.close();
		}
		else {
			System.out.println( "Skipping test, please connect your null modem cable to two serial ports on your box, seeking for exactly two FT232 (ftdi_sio) type ports!" );
		}
	}

	@Test
	public void testInvertibleComplexFail() throws IOException {
		if ( hasPorts() ) {
			Port<?> theTransmitPort = getTransmitterPort().withOpen();
			Port<?> theReceiverPort = getReceiverPort().withOpen();
			WeatherData theSenderData = TestFixures.createWeatherData();
			ComplexTypeSegment<WeatherData> theReceiverSegment;
			Segment theSenderSeg = invertibleSegment( crcPrefixSegment( complexTypeSegment( theSenderData ), CrcAlgorithmConfig.CRC_16_CCITT_FALSE ), new InvertibleXor( (byte) 0x10 ) );
			Segment theReceiverSeg = invertibleSegment( crcPrefixSegment( theReceiverSegment = complexTypeSegment( WeatherData.class ), CrcAlgorithmConfig.CRC_16_CCITT_FALSE ), new InvertibleXor( (byte) 0xC3 ) );

			SegmentResult<Segment> theResult = theReceiverPort.onReceiveSegment( theReceiverSeg );
			theSenderSeg.transmitTo( theTransmitPort );
			theResult.waitForResult();

			if ( IS_LOG_TEST_ENABLED ) {
				try {
					System.out.println( "Result = " + theResult.getResult() );
					fail( "Expecting a <" + BadCrcChecksumException.class.getSimpleName() + "> instead of a resul!t" );
				}
				catch ( BadCrcChecksumException expected ) {
					System.out.println( "Expected = " + expected.getMessage() );
				}
			}

			WeatherData theReceiverData = theReceiverSegment.getPayload();
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( theReceiverData );
			}
			assertNotEquals( theSenderData, theReceiverData );

			theTransmitPort.close();
			theReceiverPort.close();
		}
		else {
			System.out.println( "Skipping test, please connect your null modem cable to two serial ports on your box, seeking for exactly two FT232 (ftdi_sio) type ports!" );
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

	class InvertibleXor implements Invertible<Byte, Byte> {

		private byte bijectionXor;
		private byte inversionXor;

		public InvertibleXor( byte aXor ) {
			bijectionXor = aXor;
			inversionXor = aXor;
		}

		public InvertibleXor( byte aBijectionXor, byte aInversionXor ) {
			bijectionXor = aBijectionXor;
			inversionXor = aInversionXor;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public Byte applyBijection( Byte aValue ) {
			return (byte) (aValue ^ bijectionXor);
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public Byte applyInversion( Byte aValue ) {
			return (byte) (aValue ^ inversionXor);
		}
	}
}
