// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.serial;

import static org.junit.jupiter.api.Assertions.*;
import static org.refcodes.serial.SerialSugar.*;

import java.io.IOException;
import java.util.UUID;

import org.junit.jupiter.api.Test;
import org.refcodes.numerical.CrcAlgorithmConfig;
import org.refcodes.numerical.Endianess;
import org.refcodes.properties.Properties.PropertiesBuilder;
import org.refcodes.properties.PropertiesBuilderImpl;

public class LoopbackPortTest {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static boolean IS_LOG_TEST_ENABLED = Boolean.getBoolean( "log.test");

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Test segment composition.
	 *
	 * @throws TransmissionException the transmission exception
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testSegmentComposition() throws TransmissionException, IOException {

		LoopbackPort thePort = new LoopbackPort( "/dev/ttyLoopback" + UUID.randomUUID().toString() ).withOpen();

		PropertiesBuilder theBuilder = new PropertiesBuilderImpl();
		theBuilder.put( "/sensor/0/name", "temp01" );
		theBuilder.putFloat( "/sensor/0/value", 12.37F );
		theBuilder.put( "/sensor/1/name", "temp01" );
		theBuilder.putFloat( "/sensor/1/value", 8.71F );

		for ( int eLengthWidth = 2; eLengthWidth <= 8; eLengthWidth++ ) {
			for ( Endianess eEndianess : Endianess.values() ) {

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "---------- Length width = <" + eLengthWidth + ">, endianess = <" + eEndianess + "> ----------" );
				}

				AllocSegmentHead theLen;

				// @formatter:off
				Segment theSegment = crcPrefixSegment(
					segmentComposite(
						theLen = allocSegmentHead(eLengthWidth, eEndianess),
						booleanSegment( true ),
						intSegment( 5161 ), 
						theLen.letBody(
							allocSegmentBody( 
								new PropertiesSection( theBuilder )
							)
						)
					), CrcAlgorithmConfig.CRC_16_CCITT_FALSE
				);
				// @formatter:on

				thePort.transmitSegment( theSegment );
				Sequence theSequence = theSegment.toSequence();

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Sequence (hex) = " + theSequence.toHexString() );
					System.out.println( "Transmission = " + theSegment.toString() );
					System.out.println( "AbstractSchema = " + theSegment.toSchema() );
				}

				AllocSegmentHead theOtherLen;

				// @formatter:off
				Segment theOtherSegment = crcPrefixSegment(
					segmentComposite(
						theOtherLen = allocSegmentHead(eLengthWidth, eEndianess),
						booleanSegment(),
						intSegment(), 
						theOtherLen.letBody( 
							allocSegmentBody( 
									new PropertiesSection( )
							)
						)
					), CrcAlgorithmConfig.CRC_16_CCITT_FALSE
				);
				// @formatter:on

				thePort.receiveSegment( theOtherSegment );

				assertEquals( theOtherSegment, theOtherSegment );

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Other segment = " + theOtherSegment.toString() );
					System.out.println( "Other schema = " + theOtherSegment.toSchema() );
				}
			}
		}
	}

	/**
	 * Test float segment.
	 *
	 * @throws TransmissionException the transmission exception
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testFloatSegment() throws TransmissionException, IOException {

		LoopbackPort thePort = new LoopbackPort( "/dev/ttyLoopback" + UUID.randomUUID().toString() ).withOpen();

		float l;
		for ( Endianess eEndianess : Endianess.values() ) {
			for ( int i = Short.MIN_VALUE; i <= Short.MAX_VALUE; i++ ) {

				l = (i == Short.MIN_VALUE ? Float.MIN_VALUE : (i == Short.MAX_VALUE ? Float.MAX_VALUE : i * (float) Math.PI));

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "---------- Endianess = <" + eEndianess + ">, l = <" + l + "> ----------" );
				}

				// @formatter:off
				LengthSegmentDecoratorSegment<FloatSegment> theSegment = lengthSegment( 
					floatSegment( l, eEndianess )
				);
				// @formatter:on

				thePort.transmitSegment( theSegment );
				Sequence theSequence = theSegment.toSequence();

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
					System.out.println( "Transmission = " + theSegment.toString() );
				}

				// @formatter:off
				LengthSegmentDecoratorSegment<FloatSegment> theOtherSegment = lengthSegment( 
					floatSegment( eEndianess )
				);
				// @formatter:on

				thePort.receiveSegment( theOtherSegment );

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Other segment = " + theOtherSegment.toString() );
				}

				assertEquals( theSegment.getDecoratee().getPayload(), theOtherSegment.getDecoratee().getPayload() );
			}
		}
	}

	/**
	 * Test crossover loopback port 1.
	 *
	 * @throws TransmissionException the transmission exception
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testCrossoverLoopbackPort1() throws TransmissionException, IOException {

		LoopbackPort theTransmitter = new LoopbackPort( "/dev/ttyLoopback" + UUID.randomUUID().toString() ).withOpen();
		LoopbackPort theReceiver = new CrossoverLoopbackPort( theTransmitter ).withOpen();

		PropertiesBuilder theBuilder = new PropertiesBuilderImpl();
		theBuilder.put( "/sensor/0/name", "temp01" );
		theBuilder.putFloat( "/sensor/0/value", 12.37F );
		theBuilder.put( "/sensor/1/name", "temp01" );
		theBuilder.putFloat( "/sensor/1/value", 8.71F );

		for ( int eLengthWidth = 2; eLengthWidth <= 8; eLengthWidth++ ) {
			for ( Endianess eEndianess : Endianess.values() ) {

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "---------- Length width = <" + eLengthWidth + ">, endianess = <" + eEndianess + "> ----------" );
				}

				AllocSegmentHead theLen;

				// @formatter:off
				Segment theSegment = crcPrefixSegment(
					segmentComposite(
						theLen = allocSegmentHead(eLengthWidth, eEndianess),
						booleanSegment( true ),
						intSegment( 5161 ), 
						theLen.letBody(
							allocSegmentBody( 
								new PropertiesSection( theBuilder )
							)
						)
					), CrcAlgorithmConfig.CRC_16_CCITT_FALSE
				);
				// @formatter:on

				theTransmitter.transmitSegment( theSegment );
				Sequence theSequence = theSegment.toSequence();

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Sequence (hex) = " + theSequence.toHexString() );
					System.out.println( "Transmission = " + theSegment.toString() );
					System.out.println( "AbstractSchema = " + theSegment.toSchema() );
				}

				AllocSegmentHead theOtherLen;

				// @formatter:off
				Segment theOtherSegment = crcPrefixSegment(
					segmentComposite(
						theOtherLen = allocSegmentHead(eLengthWidth, eEndianess),
						booleanSegment(),
						intSegment(), 
						theOtherLen.letBody( 
							allocSegmentBody( 
									new PropertiesSection( )
							)
						)
					), CrcAlgorithmConfig.CRC_16_CCITT_FALSE
				);
				// @formatter:on

				theReceiver.receiveSegment( theOtherSegment );

				assertEquals( theOtherSegment, theOtherSegment );

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Other segment = " + theOtherSegment.toString() );
					System.out.println( "Other schema = " + theOtherSegment.toSchema() );
				}
			}
		}
	}

	/**
	 * Test crossover loopback port 2.
	 *
	 * @throws TransmissionException the transmission exception
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws NoSuchPortExcpetion the no such port excpetion
	 */
	@Test
	public void testCrossoverLoopbackPort2() throws TransmissionException, IOException, NoSuchPortExcpetion {

		LoopbackPortHub theTransmitterHub = new LoopbackPortHub();
		LoopbackPortHub theReceiverHub = new CrossoverLoopbackPortHub( theTransmitterHub );

		String thePortName = "/dev/ttyLoopback" + UUID.randomUUID().toString();
		LoopbackPort theTransmitter = theTransmitterHub.toPort( thePortName ).withOpen();
		LoopbackPort theReceiver = theReceiverHub.toPort( thePortName ).withOpen();

		float l;
		for ( Endianess eEndianess : Endianess.values() ) {
			for ( int i = Short.MIN_VALUE; i <= Short.MAX_VALUE; i++ ) {

				l = (i == Short.MIN_VALUE ? Float.MIN_VALUE : (i == Short.MAX_VALUE ? Float.MAX_VALUE : i * (float) Math.PI));

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "---------- Endianess = <" + eEndianess + ">, l = <" + l + "> ----------" );
				}

				// @formatter:off
				LengthSegmentDecoratorSegment<FloatSegment> theSegment = lengthSegment( 
					floatSegment( l, eEndianess )
				);
				// @formatter:on

				theTransmitter.transmitSegment( theSegment );
				Sequence theSequence = theSegment.toSequence();

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
					System.out.println( "Transmission = " + theSegment.toString() );
				}

				// @formatter:off
				LengthSegmentDecoratorSegment<FloatSegment> theOtherSegment = lengthSegment( 
					floatSegment( eEndianess )
				);
				// @formatter:on

				theReceiver.receiveSegment( theOtherSegment );

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Other segment = " + theOtherSegment.toString() );
				}

				assertEquals( theSegment.getDecoratee().getPayload(), theOtherSegment.getDecoratee().getPayload() );
			}
		}
	}

	/**
	 * Test edge case.
	 *
	 * @throws TransmissionException the transmission exception
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testEdgeCase() throws TransmissionException, IOException {

		LoopbackPort thePort = new LoopbackPort( "/dev/ttyLoopback" + UUID.randomUUID().toString() ).withOpen();

		float l;
		Endianess eEndianess = Endianess.LITTLE;
		l = -102239.99F;

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "---------- Endianess = <" + eEndianess + ">, l = <" + l + "> ----------" );
		}

		// @formatter:off
		LengthSegmentDecoratorSegment<FloatSegment> theSegment = lengthSegment( 
			floatSegment( l, eEndianess )
		);
		// @formatter:on

		thePort.transmitSegment( theSegment );
		Sequence theSequence = theSegment.toSequence();

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Sequence (hex) = " + theSequence.toHexString( ", " ) );
			System.out.println( "Transmission = " + theSegment.toString() );
		}

		// @formatter:off
		LengthSegmentDecoratorSegment<FloatSegment> theOtherSegment = lengthSegment( 
			floatSegment( eEndianess )
		);
		// @formatter:on

		thePort.receiveSegment( theOtherSegment );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Other sequence (hex) = " + theOtherSegment.toSequence().toHexString() );
			System.out.println( "Other segment = " + theOtherSegment.toString() );
		}

		assertEquals( theSegment.getDecoratee().getPayload(), theOtherSegment.getDecoratee().getPayload() );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
