// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.serial;

import static org.junit.jupiter.api.Assertions.*;
import static org.refcodes.serial.SerialSugar.*;

import java.io.IOException;
import java.nio.charset.StandardCharsets;

import org.junit.jupiter.api.Test;
import org.refcodes.exception.TimeoutIOException;
import org.refcodes.numerical.CrcAlgorithmConfig;
import org.refcodes.struct.SimpleTypeMap;
import org.refcodes.textual.VerboseTextBuilder;

public class MagicBytesTest extends AbstractLoopbackPortTest {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	private static boolean IS_LOG_TEST_ENABLED = Boolean.getBoolean( "log.test");

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final int WAIT_TIMEOUT = 5000;
	private static final int LOOPS = 15;

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@Test
	public void testMagicBytesSegmentDispatcher1() throws TransmissionException {

		StringSegment theString1;
		StringSegment theString2;
		StringSegment theString3;

		MagicBytesSegmentDecorator<StringSegment> theMagicBytes1 = assertMagicBytesSegment( theString1 = stringSegment(), "A00".getBytes() );
		MagicBytesSegmentDecorator<StringSegment> theMagicBytes2 = assertMagicBytesSegment( theString2 = stringSegment(), "A01".getBytes() );
		MagicBytesSegmentDecorator<StringSegment> theMagicBytes3 = assertMagicBytesSegment( theString3 = stringSegment(), "B02".getBytes() );

		MagicBytesSegmentMultiplexer theMultiplexer = new MagicBytesSegmentMultiplexer( theMagicBytes1, theMagicBytes2, theMagicBytes3 );

		MagicBytesSegmentDecorator<StringSegment> theTransmission = magicBytesSegment( stringSegment( "A01!" ), "A01".getBytes() );

		theMultiplexer.fromTransmission( theTransmission.toSequence() );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( new String( theMagicBytes1.getPayload() ) + ": " + theString1.getPayload() );
			System.out.println( new String( theMagicBytes2.getPayload() ) + ": " + theString2.getPayload() );
			System.out.println( new String( theMagicBytes3.getPayload() ) + ": " + theString3.getPayload() );
		}

		assertNull( theString1.getPayload() );
		assertEquals( "A01!", theString2.getPayload() );
		assertNull( theString3.getPayload() );
	}

	@Test
	public void testMagicBytesSegmentDispatcher2() throws TransmissionException {

		StringSegment theString1;
		StringSegment theString2;
		StringSegment theString3;

		MagicBytesSegmentDecorator<StringSegment> theMagicBytes1 = assertMagicBytesSegment( theString1 = stringSegment(), "A00".getBytes() );
		MagicBytesSegmentDecorator<StringSegment> theMagicBytes2 = assertMagicBytesSegment( theString2 = stringSegment(), "A01".getBytes() );
		MagicBytesSegmentDecorator<StringSegment> theMagicBytes3 = assertMagicBytesSegment( theString3 = stringSegment(), "B02".getBytes() );

		MagicBytesSegmentMultiplexer theMultiplexer = new MagicBytesSegmentMultiplexer( theMagicBytes1, theMagicBytes2, theMagicBytes3 );

		MagicBytesSegmentDecorator<StringSegment> theTransmission = magicBytesSegment( stringSegment( "A00!" ), "A00".getBytes() );
		theMultiplexer.fromTransmission( theTransmission.toSequence() );
		theTransmission = magicBytesSegment( stringSegment( "A01!" ), "A01".getBytes() );
		theMultiplexer.fromTransmission( theTransmission.toSequence() );
		theTransmission = magicBytesSegment( stringSegment( "B02!" ), "B02".getBytes() );
		theMultiplexer.fromTransmission( theTransmission.toSequence() );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( new String( theMagicBytes1.getPayload() ) + ": " + theString1.getPayload() );
			System.out.println( new String( theMagicBytes2.getPayload() ) + ": " + theString2.getPayload() );
			System.out.println( new String( theMagicBytes3.getPayload() ) + ": " + theString3.getPayload() );
		}

		assertEquals( "A00!", theString1.getPayload() );
		assertEquals( "A01!", theString2.getPayload() );
		assertEquals( "B02!", theString3.getPayload() );
	}

	@Test
	public void testMagicBytesSegmentDispatcher3() throws IOException {

		if ( hasPorts() ) {
			Port<?> theTransmitPort = getTransmitterPort().withOpen();
			Port<?> theReceiverPort = getReceiverPort().withOpen();

			StringSegment theString1;
			StringSegment theString2;
			StringSegment theString3;

			MagicBytesSegmentDecorator<StringSegment> theMagicBytes1 = assertMagicBytesSegment( theString1 = stringSegment(), "A00".getBytes() );
			MagicBytesSegmentDecorator<StringSegment> theMagicBytes2 = assertMagicBytesSegment( theString2 = stringSegment(), "A01".getBytes() );
			MagicBytesSegmentDecorator<StringSegment> theMagicBytes3 = assertMagicBytesSegment( theString3 = stringSegment(), "B02".getBytes() );

			MagicBytesSegmentMultiplexer theMultiplexer = new MagicBytesSegmentMultiplexer( theMagicBytes1, theMagicBytes2, theMagicBytes3 );

			MagicBytesSegmentDecorator<StringSegment> theTransmission = magicBytesSegment( stringSegment( "A01!" ), "A01".getBytes() );

			theTransmitPort.transmitSegment( theTransmission );
			SegmentResult<?> theResult = theReceiverPort.onReceiveSegment( theMultiplexer );
			theResult.waitForResult();

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( new String( theMagicBytes1.getPayload() ) + ": " + theString1.getPayload() );
				System.out.println( new String( theMagicBytes2.getPayload() ) + ": " + theString2.getPayload() );
				System.out.println( new String( theMagicBytes3.getPayload() ) + ": " + theString3.getPayload() );
			}

			assertNull( theString1.getPayload() );
			assertEquals( "A01!", theString2.getPayload() );
			assertNull( theString3.getPayload() );
		}
		else {
			System.out.println( "Skipping test, please connect your null modem cable to two serial ports on your box, seeking for exactly two FT232 (ftdi_sio) type ports!" );
		}
	}

	@Test
	public void testMagicBytesSectionDispatcher1() throws TransmissionException {

		StringSection theString1;
		StringSection theString2;
		StringSection theString3;

		MagicBytesSectionDecorator<StringSection> theMagicBytes1 = assertMagicBytesSection( theString1 = stringSection(), "A00".getBytes() );
		MagicBytesSectionDecorator<StringSection> theMagicBytes2 = assertMagicBytesSection( theString2 = stringSection(), "A01".getBytes() );
		MagicBytesSectionDecorator<StringSection> theMagicBytes3 = assertMagicBytesSection( theString3 = stringSection(), "B02".getBytes() );

		MagicBytesSectionMultiplexer theMultiplexer = new MagicBytesSectionMultiplexer( theMagicBytes1, theMagicBytes2, theMagicBytes3 );

		MagicBytesSectionDecorator<StringSection> theTransmission = magicBytesSection( stringSection( "A01!" ), "A01".getBytes() );

		theMultiplexer.fromTransmission( theTransmission.toSequence(), theTransmission.getLength() );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( new String( theMagicBytes1.getPayload() ) + ": " + theString1.getPayload() );
			System.out.println( new String( theMagicBytes2.getPayload() ) + ": " + theString2.getPayload() );
			System.out.println( new String( theMagicBytes3.getPayload() ) + ": " + theString3.getPayload() );
		}

		assertNull( theString1.getPayload() );
		assertEquals( "A01!", theString2.getPayload() );
		assertNull( theString3.getPayload() );
	}

	@Test
	public void testMagicBytesSectionDispatcher2() throws TransmissionException {

		StringSection theString1;
		StringSection theString2;
		StringSection theString3;

		MagicBytesSectionDecorator<StringSection> theMagicBytes1 = assertMagicBytesSection( theString1 = stringSection(), "A00".getBytes() );
		MagicBytesSectionDecorator<StringSection> theMagicBytes2 = assertMagicBytesSection( theString2 = stringSection(), "A01".getBytes() );
		MagicBytesSectionDecorator<StringSection> theMagicBytes3 = assertMagicBytesSection( theString3 = stringSection(), "B02".getBytes() );

		MagicBytesSectionMultiplexer theMultiplexer = new MagicBytesSectionMultiplexer( theMagicBytes1, theMagicBytes2, theMagicBytes3 );

		MagicBytesSectionDecorator<StringSection> theTransmission = magicBytesSection( stringSection( "A00!" ), "A00".getBytes() );
		theMultiplexer.fromTransmission( theTransmission.toSequence(), theTransmission.getLength() );
		theTransmission = magicBytesSection( stringSection( "A01!" ), "A01".getBytes() );
		theMultiplexer.fromTransmission( theTransmission.toSequence(), theTransmission.getLength() );
		theTransmission = magicBytesSection( stringSection( "B02!" ), "B02".getBytes() );
		theMultiplexer.fromTransmission( theTransmission.toSequence(), theTransmission.getLength() );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( new String( theMagicBytes1.getPayload() ) + ": " + theString1.getPayload() );
			System.out.println( new String( theMagicBytes2.getPayload() ) + ": " + theString2.getPayload() );
			System.out.println( new String( theMagicBytes3.getPayload() ) + ": " + theString3.getPayload() );
		}

		assertEquals( "A00!", theString1.getPayload() );
		assertEquals( "A01!", theString2.getPayload() );
		assertEquals( "B02!", theString3.getPayload() );
	}

	@Test
	public void testMagicBytesPlain() throws TransmissionException {

		MagicBytesSegment theSegment = magicBytesSegment( "FEED C0DE", StandardCharsets.US_ASCII );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( theSegment.toSchema() );
		}
		Sequence theSequence = theSegment.toSequence();

		MagicBytesSegment theOtherSegment = magicBytesSegment( 9 );

		theOtherSegment.fromTransmission( theSequence );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( theOtherSegment.toSchema() );
		}

		assertEquals( theSegment, theOtherSegment );
	}

	// -------------------------------------------------------------------------

	@Test
	public void testAssertMagicBytesPlainPass() throws TransmissionException {

		AssertMagicBytesSegment theSegment = assertMagicBytesSegment( "FEED C0DE", StandardCharsets.US_ASCII );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( theSegment.toSchema() );
		}
		Sequence theSequence = theSegment.toSequence();

		AssertMagicBytesSegment theOtherSegment = assertMagicBytesSegment( "FEED C0DE", StandardCharsets.US_ASCII );

		theOtherSegment.fromTransmission( theSequence );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( theOtherSegment.toSchema() );
		}

		assertEquals( theSegment, theOtherSegment );
	}

	// -------------------------------------------------------------------------

	@Test
	public void testAssertMagicBytesPlainFail() throws TransmissionException {

		AssertMagicBytesSegment theSegment = assertMagicBytesSegment( "FEED C0DE", StandardCharsets.US_ASCII );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( theSegment.toSchema() );
		}
		Sequence theSequence = theSegment.toSequence();

		AssertMagicBytesSegment theOtherSegment = assertMagicBytesSegment( "BAD", StandardCharsets.US_ASCII );

		try {
			theOtherSegment.fromTransmission( theSequence );
			fail( "Expected a <BadMagicBytesException> here!" );
		}
		catch ( BadMagicBytesException e ) {
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "Expected: " + e.getMessage() );
			}
		}
	}

	// -------------------------------------------------------------------------

	@Test
	public void testMagicBytesComposite() throws TransmissionException {

		// @formatter:off
		SegmentComposite<MagicBytesSegment> theSegment = segmentComposite(
			magicBytesSegment( "FEED C0DE", StandardCharsets.US_ASCII ),
			magicBytesSegment(  new byte[] {-64, -34, -2, -19} ) // C0 DE FE ED = CODE FEED
		);
		// @formatter:on

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( theSegment.toSchema() );
		}
		Sequence theSequence = theSegment.toSequence();

		// @formatter:off
		SegmentComposite<MagicBytesSegment> theOtherSegment = segmentComposite(
			magicBytesSegment( 9 ),
			magicBytesSegment( 4 )
		);
		// @formatter:on

		theOtherSegment.fromTransmission( theSequence );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( theOtherSegment.toSchema() );
		}

		assertEquals( theSegment, theOtherSegment );
	}

	@Test
	public void testMagicBytesSimple() throws TransmissionException {
		// @formatter:off
		LengthSegmentDecoratorSegment<MagicBytesSegmentDecorator<?>> theSegment = lengthSegment( 
			magicBytesSegment( 
				allocSegment( 
					stringSection("Hello World!")
				), new byte[] {-64, -34, -2, -19} // C0 DE FE ED = CODE FEED
			)
		);
		// @formatter:on

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( theSegment.toSchema() );
		}
		Sequence theSequence = theSegment.toSequence();

		// @formatter:off
		LengthSegmentDecoratorSegment<MagicBytesSegmentDecorator<?>> theOtherSegment = lengthSegment( 
			magicBytesSegment( 
				allocSegment( 
					stringSection()
				), 4
			)
		);
		// @formatter:on

		theOtherSegment.fromTransmission( theSequence );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( theOtherSegment.toSchema() );
		}

		assertEquals( theSegment, theOtherSegment );
	}

	@Test
	public void testMagicBytesComplex() throws TransmissionException {
		// @formatter:off
		CrcSegmentDecorator<LengthSegmentDecoratorSegment<?>> theSegment = crcPrefixSegment(
			lengthSegment(
				magicBytesSegment( 
					allocSegment( 
						stringSection("Hello World!")
					), new byte[] {-64, -34, -2, -19} // C0 DE FE ED = CODE FEED
				) 
			), CrcAlgorithmConfig.CRC_16_X25
		);
		// @formatter:on

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( theSegment.toSchema() );
		}
		Sequence theSequence = theSegment.toSequence();

		// @formatter:off
		CrcSegmentDecorator<LengthSegmentDecoratorSegment<?>> theOtherSegment = crcPrefixSegment(
			lengthSegment(
				magicBytesSegment( 
					allocSegment( 
						stringSection()
					), 4
				) 
			), CrcAlgorithmConfig.CRC_16_X25
		);
		// @formatter:on

		theOtherSegment.fromTransmission( theSequence );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( theOtherSegment.toSchema() );
		}

		assertEquals( theSegment, theOtherSegment );

		theSequence.setByteAt( 0, (byte) 65 );

		try {
			theOtherSegment.fromTransmission( theSequence );
			fail( "Expected an exception of type <BadCrcChecksumSequenceException>!" );
		}
		catch ( BadCrcChecksumSequenceException expected ) {
			// expected!
		}

		if ( IS_LOG_TEST_ENABLED ) {
			SimpleTypeMap theSimpleTypeMap = theSegment.toSimpleTypeMap();
			for ( String eKey : theSimpleTypeMap.sortedKeys() ) {
				System.out.println( eKey + " = " + theSimpleTypeMap.get( eKey ) );
			}
		}
	}

	// -------------------------------------------------------------------------

	@Test
	public void testAssertMagicBytesPass() throws TransmissionSequenceException, IOException, NoSuchPortExcpetion, InterruptedException {
		if ( hasPorts() ) {
			Port<?> theTransmitPort = getTransmitterPort().withOpen();
			Port<?> theReceiverPort = getReceiverPort().withOpen();

			StringArraySection theReceiverPayloadSec;
			String[] thePayload = { "Hello", "World", "!!!" };
			Segment theSenderSeg = segmentComposite( assertMagicBytesSegment( "MZ" ), allocSegment( stringArraySection( thePayload ) ) );
			Segment theReceiverSeg = segmentComposite( assertMagicBytesSegment( "MZ" ), allocSegment( theReceiverPayloadSec = stringArraySection() ) );

			for ( int i = 0; i < LOOPS; i++ ) {
				theReceiverPayloadSec.setPayload( null );
				SegmentResult<?> theResult = theReceiverPort.onReceiveSegment( theReceiverSeg );
				theSenderSeg.transmitTo( theTransmitPort );
				// DEBUG |-->
				try {
					theResult.waitForResult( WAIT_TIMEOUT );
				}
				catch ( TimeoutIOException e ) {
					System.err.println( e.getMessage() );
					try {
						System.err.println( theResult.getResult() );
						String[] theReceiverData = theReceiverPayloadSec.getPayload();
						System.err.println( VerboseTextBuilder.asString( theReceiverData ) );
					}
					catch ( TimeoutIOException ignore ) {}
					throw e;
				}
				// DEBUG <--|

				String[] theReceiverData = theReceiverPayloadSec.getPayload();
				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( VerboseTextBuilder.asString( theReceiverData ) );
				}
				assertArrayEquals( thePayload, theReceiverData );
			}

			theTransmitPort.close();
			theReceiverPort.close();
		}
		else {
			System.out.println( "Skipping test, please connect your null modem cable to two serial ports on your box, seeking for exactly two FT232 (ftdi_sio) type ports!" );
		}
	}

	@Test
	public void testAssertMagicBytesFail() throws TransmissionSequenceException, IOException, NoSuchPortExcpetion, InterruptedException {
		if ( hasPorts() ) {
			Port<?> theTransmitPort = getTransmitterPort().withOpen();
			Port<?> theReceiverPort = getReceiverPort().withOpen();

			String[] thePayload = { "Hello", "World", "!!!" };
			Segment theSenderSeg = segmentComposite( assertMagicBytesSegment( "MZ" ), allocSegment( stringArraySection( thePayload ) ) );
			StringArraySection theReceiverPayloadSec;
			Segment theReceiverSeg = segmentComposite( assertMagicBytesSegment( "P2P" ), allocSegment( theReceiverPayloadSec = stringArraySection() ) );

			SegmentResult<?> theResult = theReceiverPort.onReceiveSegment( theReceiverSeg );
			theSenderSeg.transmitTo( theTransmitPort );
			// DEBUG |-->
			try {
				theResult.waitForResult( WAIT_TIMEOUT );
			}
			catch ( TimeoutIOException e ) {
				System.err.println( e.getMessage() );
				try {
					System.err.println( theResult.getResult() );
					String[] theReceiverData = theReceiverPayloadSec.getPayload();
					System.err.println( VerboseTextBuilder.asString( theReceiverData ) );
				}
				catch ( TimeoutIOException ignore ) {}
				throw e;
			}
			// DEBUG <--|

			try {
				theResult.getResult();
				fail( "Expected a <BadMagicBytesException> here!" );
			}
			catch ( BadMagicBytesException e ) {
				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Expected: " + e.getMessage() );
				}
			}

			theTransmitPort.close();
			theReceiverPort.close();
		}
		else {
			System.out.println( "Skipping test, please connect your null modem cable to two serial ports on your box, seeking for exactly two FT232 (ftdi_sio) type ports!" );
		}
	}

	// -------------------------------------------------------------------------

	@Test
	public void testAssertMagicBytesSegmentPass() throws TransmissionSequenceException, IOException, NoSuchPortExcpetion, InterruptedException {
		if ( hasPorts() ) {
			Port<?> theTransmitPort = getTransmitterPort().withOpen();
			Port<?> theReceiverPort = getReceiverPort().withOpen();

			StringArraySection theReceiverPayloadSec, theSenderPayloadSec;
			String[] ePayload;
			Segment theSenderSeg = assertMagicBytesSegment( allocSegment( theSenderPayloadSec = stringArraySection() ), new byte[] { -1, -2 } );
			Segment theReceiverSeg = assertMagicBytesSegment( allocSegment( theReceiverPayloadSec = stringArraySection() ), new byte[] { -1, -2 } );

			for ( int i = 0; i < LOOPS; i++ ) {
				ePayload = new String[] { "Hello", "World", "!!!", "" + i };
				theSenderPayloadSec.setPayload( ePayload );
				theReceiverPayloadSec.setPayload( null );
				SegmentResult<?> theResult = theReceiverPort.onReceiveSegment( theReceiverSeg );
				theSenderSeg.transmitTo( theTransmitPort );
				// DEBUG |-->
				try {
					theResult.waitForResult( WAIT_TIMEOUT );
				}
				catch ( TimeoutIOException e ) {
					System.err.println( e.getMessage() );
					try {
						System.err.println( theResult.getResult() );
						String[] theReceiverData = theReceiverPayloadSec.getPayload();
						System.err.println( VerboseTextBuilder.asString( theReceiverData ) );
					}
					catch ( TimeoutIOException ignore ) {}
					throw e;
				}
				// DEBUG <--|
				String[] theReceiverData = theReceiverPayloadSec.getPayload();
				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( VerboseTextBuilder.asString( theReceiverData ) );
				}
				assertArrayEquals( ePayload, theReceiverData );
			}

			theTransmitPort.close();
			theReceiverPort.close();
		}
		else {
			System.out.println( "Skipping test, please connect your null modem cable to two serial ports on your box, seeking for exactly two FT232 (ftdi_sio) type ports!" );
		}
	}

	@Test
	public void testAssertMagicBytesSegmentFail() throws TransmissionSequenceException, IOException, NoSuchPortExcpetion, InterruptedException {
		if ( hasPorts() ) {
			Port<?> theTransmitPort = getTransmitterPort().withOpen();
			Port<?> theReceiverPort = getReceiverPort().withOpen();

			String[] thePayload = { "Hello", "World", "!!!" };
			Segment theSenderSeg = assertMagicBytesSegment( allocSegment( stringArraySection( thePayload ) ), new byte[] { -1, -2 } );
			StringArraySection theReceiverPayloadSec;
			Segment theReceiverSeg = assertMagicBytesSegment( allocSegment( theReceiverPayloadSec = stringArraySection() ), new byte[] { -1, -5, 127 } );

			SegmentResult<?> theResult = theReceiverPort.onReceiveSegment( theReceiverSeg );
			theSenderSeg.transmitTo( theTransmitPort );
			// DEBUG |-->
			try {
				theResult.waitForResult( WAIT_TIMEOUT );
			}
			catch ( TimeoutIOException e ) {
				System.err.println( e.getMessage() );
				try {
					System.err.println( theResult.getResult() );
					String[] theReceiverData = theReceiverPayloadSec.getPayload();
					System.err.println( VerboseTextBuilder.asString( theReceiverData ) );
				}
				catch ( TimeoutIOException ignore ) {}
				throw e;
			}
			// DEBUG <--|
			try {
				theResult.getResult();
				fail( "Expected a <BadMagicBytesException> here!" );
			}
			catch ( BadMagicBytesException e ) {
				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Expected: " + e.getMessage() );
				}
			}

			theTransmitPort.close();
			theReceiverPort.close();
		}
		else {
			System.out.println( "Skipping test, please connect your null modem cable to two serial ports on your box, seeking for exactly two FT232 (ftdi_sio) type ports!" );
		}
	}

	// -------------------------------------------------------------------------

	@Test
	public void testAssertMagicBytesSectionPass() throws TransmissionSequenceException, IOException, NoSuchPortExcpetion, InterruptedException {
		if ( hasPorts() ) {
			Port<?> theTransmitPort = getTransmitterPort().withOpen();
			Port<?> theReceiverPort = getReceiverPort().withOpen();

			StringArraySection theReceiverPayloadSec;
			String[] thePayload = { "Hello", "World", "!!!" };
			Segment theSenderSeg = allocSegment( assertMagicBytesSection( stringArraySection( thePayload ), "MZ" ) );
			Segment theReceiverSeg = allocSegment( assertMagicBytesSection( theReceiverPayloadSec = stringArraySection(), "MZ" ) );

			for ( int i = 0; i < LOOPS; i++ ) {
				theReceiverPayloadSec.setPayload( null );
				SegmentResult<?> theResult = theReceiverPort.onReceiveSegment( theReceiverSeg );
				theSenderSeg.transmitTo( theTransmitPort );
				// DEBUG |-->
				try {
					theResult.waitForResult( WAIT_TIMEOUT );
				}
				catch ( TimeoutIOException e ) {
					System.err.println( e.getMessage() );
					try {
						System.err.println( theResult.getResult() );
						String[] theReceiverData = theReceiverPayloadSec.getPayload();
						System.err.println( VerboseTextBuilder.asString( theReceiverData ) );
					}
					catch ( TimeoutIOException ignore ) {}
					throw e;
				}
				// DEBUG <--|
				String[] theReceiverData = theReceiverPayloadSec.getPayload();
				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( VerboseTextBuilder.asString( theReceiverData ) );
				}
				assertArrayEquals( thePayload, theReceiverData );
			}

			theTransmitPort.close();
			theReceiverPort.close();
		}
		else {
			System.out.println( "Skipping test, please connect your null modem cable to two serial ports on your box, seeking for exactly two FT232 (ftdi_sio) type ports!" );
		}
	}

	@Test
	public void testAssertMagicBytesSectionFail() throws TransmissionSequenceException, IOException, NoSuchPortExcpetion, InterruptedException {
		if ( hasPorts() ) {
			Port<?> theTransmitPort = getTransmitterPort().withOpen();
			Port<?> theReceiverPort = getReceiverPort().withOpen();

			String[] thePayload = { "Hello", "World", "!!!" };
			Segment theSenderSeg = allocSegment( assertMagicBytesSection( stringArraySection( thePayload ), "MZ" ) );
			StringArraySection theReceiverPayloadSec;
			Segment theReceiverSeg = allocSegment( assertMagicBytesSection( theReceiverPayloadSec = stringArraySection(), "FEEDC0DE" ) );

			SegmentResult<?> theResult = theReceiverPort.onReceiveSegment( theReceiverSeg );
			theSenderSeg.transmitTo( theTransmitPort );
			// DEBUG |-->
			try {
				theResult.waitForResult( WAIT_TIMEOUT );
			}
			catch ( TimeoutIOException e ) {
				System.err.println( e.getMessage() );
				try {
					System.err.println( theResult.getResult() );
					String[] theReceiverData = theReceiverPayloadSec.getPayload();
					System.err.println( VerboseTextBuilder.asString( theReceiverData ) );
				}
				catch ( TimeoutIOException ignore ) {}
				throw e;
			}
			// DEBUG <--|
			try {
				theResult.getResult();
				fail( "Expected a <BadMagicBytesException> here!" );
			}
			catch ( BadMagicBytesException e ) {
				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Expected: " + e.getMessage() );
				}
			}

			theTransmitPort.close();
			theReceiverPort.close();
		}
		else {
			System.out.println( "Skipping test, please connect your null modem cable to two serial ports on your box, seeking for exactly two FT232 (ftdi_sio) type ports!" );
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
