// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.serial;

import static org.junit.jupiter.api.Assertions.*;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import org.junit.jupiter.api.Test;
import org.refcodes.runtime.RuntimeUtility;
import org.refcodes.textual.VerboseTextBuilder;

public class SequenceTest {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static boolean IS_LOG_TEST_ENABLED = Boolean.getBoolean( "log.test");

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Test construct.
	 */
	@Test
	public void testConstruct() {

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "### " + RuntimeUtility.getCallerStackTraceElement().getMethodName() + " ###\n" );
		}

		byte[] theNumbers = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };
		byte[] theChars = new byte[] { 'a', 'b', 'c' };

		ByteArraySequence theSequence = new ByteArraySequence( theNumbers, theChars );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		String theExpected = "{ { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 }, { 97, 98, 99 } }";
		assertEquals( theExpected, VerboseTextBuilder.asString( theSequence.getChunks() ) );
	}

	/**
	 * Test output stream 1.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testOutputStream1() throws IOException {

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "### " + RuntimeUtility.getCallerStackTraceElement().getMethodName() + " ###\n" );
		}

		byte[] theValues = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };

		ByteArraySequence theSequence = new ByteArraySequence();

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		OutputStream theOutputStream = theSequence.getOutputStream();
		for ( int i = 0; i < theValues.length; i++ ) {
			theOutputStream.write( theValues[i] );
		}

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		assertArrayEquals( theValues, theSequence.toBytes() );
	}

	/**
	 * Test output stream 2.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testOutputStream2() throws IOException {

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "### " + RuntimeUtility.getCallerStackTraceElement().getMethodName() + " ###\n" );
		}

		byte[] theValues = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };

		ByteArraySequence theSequence = new ByteArraySequence();

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		OutputStream theOutputStream = theSequence.getOutputStream();
		theOutputStream.write( theValues );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		assertArrayEquals( theValues, theSequence.toBytes() );
	}

	/**
	 * Test input stream 1.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testInputStream1() throws IOException {

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "### " + RuntimeUtility.getCallerStackTraceElement().getMethodName() + " ###\n" );
		}

		byte[] theNumbers = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };
		byte[] theChars = new byte[] { 'a', 'b', 'c' };

		ByteArraySequence theSequence = new ByteArraySequence( theNumbers, theChars );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		InputStream theInputStream = theSequence.getInputStream();
		int eRead;
		int i = 0;
		byte[] theBytes = theSequence.toBytes();
		byte[] theStreamBytes = new byte[theSequence.getLength()];
		while ( (eRead = theInputStream.read()) != -1 ) {
			theStreamBytes[i] = (byte) eRead;
			i++;
		}
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( VerboseTextBuilder.asString( theStreamBytes ) );
		}
		assertArrayEquals( theBytes, theStreamBytes );
	}

	/**
	 * Test input stream 2.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testInputStream2() throws IOException {

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "### " + RuntimeUtility.getCallerStackTraceElement().getMethodName() + " ###\n" );
		}

		byte[] theNumbers = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };
		byte[] theChars = new byte[] { 'a', 'b', 'c' };

		ByteArraySequence theSequence = new ByteArraySequence( theNumbers, theChars );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		InputStream theInputStream = theSequence.getInputStream();
		byte[] theBytes = theSequence.toBytes();
		byte[] theStreamBytes = new byte[theSequence.getLength()];

		theInputStream.read( theStreamBytes, 0, theInputStream.available() );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( VerboseTextBuilder.asString( theStreamBytes ) );
		}
		assertArrayEquals( theBytes, theStreamBytes );
	}

	/**
	 * Test input stream 3.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testInputStream3() throws IOException {

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "### " + RuntimeUtility.getCallerStackTraceElement().getMethodName() + " ###\n" );
		}

		byte[] theNumbers = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };
		byte[] theChars = new byte[] { 'a', 'b', 'c' };

		ByteArraySequence theSequence = new ByteArraySequence( theNumbers, theChars );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		InputStream theInputStream = theSequence.getInputStream();
		byte[] theBytes = theSequence.toBytes( 0, 10 );
		byte[] theStreamBytes = new byte[10];

		int theRead = theInputStream.read( theStreamBytes, 0, 10 );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( VerboseTextBuilder.asString( theStreamBytes ) );
		}
		assertArrayEquals( theBytes, theStreamBytes );
		assertEquals( theRead, 10 );
	}

	/**
	 * Test input stream 4.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testInputStream4() throws IOException {

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "### " + RuntimeUtility.getCallerStackTraceElement().getMethodName() + " ###\n" );
		}

		byte[] theNumbers = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };
		byte[] theChars = new byte[] { 'a', 'b', 'c' };

		ByteArraySequence theSequence = new ByteArraySequence( theNumbers, theChars );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		InputStream theInputStream = theSequence.getInputStream();
		byte[] theBytes = theSequence.toBytes();
		byte[] theStreamBytes = new byte[theSequence.getLength()];

		int theRead = theInputStream.read( theStreamBytes, 0, 20 );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( VerboseTextBuilder.asString( theStreamBytes ) );
		}
		assertArrayEquals( theBytes, theStreamBytes );
		assertEquals( theRead, theBytes.length );
	}

	/**
	 * Test append.
	 */
	@Test
	public void testAppend() {

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "### " + RuntimeUtility.getCallerStackTraceElement().getMethodName() + " ###\n" );
		}

		byte[] theNumbers = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };
		byte[] theChars = new byte[] { 'a', 'b', 'c' };

		ByteArraySequence theSequence = new ByteArraySequence();
		theSequence.append( theNumbers );
		theSequence.append( theChars );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		String theExpected = "{ { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 }, { 97, 98, 99 } }";
		assertEquals( theExpected, VerboseTextBuilder.asString( theSequence.getChunks() ) );
	}

	/**
	 * Test prepend.
	 */
	@Test
	public void testPrepend() {

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "### " + RuntimeUtility.getCallerStackTraceElement().getMethodName() + " ###\n" );
		}

		byte[] theNumbers = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };
		byte[] theChars = new byte[] { 'a', 'b', 'c' };

		ByteArraySequence theSequence = new ByteArraySequence( theNumbers );
		theSequence.prepend( theChars );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		String theExpected = "{ { 97, 98, 99 }, { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 } }";
		assertEquals( theExpected, VerboseTextBuilder.asString( theSequence.getChunks() ) );
	}

	/**
	 * Test to truncate tail.
	 */
	@Test
	public void testToTruncateTail() {

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "### " + RuntimeUtility.getCallerStackTraceElement().getMethodName() + " ###\n" );
		}

		byte[] theNumbers = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };
		byte[] theChars = new byte[] { 'a', 'b', 'c' };

		ByteArraySequence theSequence = new ByteArraySequence( theNumbers, theChars );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		ByteArraySequence theTruncated = theSequence.toTruncateTail( 1 );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theTruncated );
		}

		String theExpected = "{ { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 }, { 97, 98 } }";
		assertEquals( theExpected, VerboseTextBuilder.asString( theTruncated.getChunks() ) );
		theExpected = "{ { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 }, { 97, 98, 99 } }";
		assertEquals( theExpected, VerboseTextBuilder.asString( theSequence.getChunks() ) );
	}

	/**
	 * Test to truncate head.
	 */
	@Test
	public void testToTruncateHead() {

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "### " + RuntimeUtility.getCallerStackTraceElement().getMethodName() + " ###\n" );
		}

		byte[] theNumbers = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };
		byte[] theChars = new byte[] { 'a', 'b', 'c' };

		ByteArraySequence theSequence = new ByteArraySequence( theNumbers, theChars );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		ByteArraySequence theTruncated = theSequence.toTruncateHead( 1 );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theTruncated );
		}

		String theExpected = "{ { 1, 2, 3, 4, 5, 6, 7, 8, 9 }, { 97, 98, 99 } }";
		assertEquals( theExpected, VerboseTextBuilder.asString( theTruncated.getChunks() ) );
		theExpected = "{ { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 }, { 97, 98, 99 } }";
		assertEquals( theExpected, VerboseTextBuilder.asString( theSequence.getChunks() ) );
	}

	/**
	 * Test truncate tail 1.
	 */
	@Test
	public void testTruncateTail1() {

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "### " + RuntimeUtility.getCallerStackTraceElement().getMethodName() + " ###\n" );
		}

		byte[] theNumbers = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };
		byte[] theChars = new byte[] { 'a', 'b', 'c' };

		ByteArraySequence theSequence = new ByteArraySequence( theNumbers, theChars );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		theSequence.truncateTail( 1 );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		String theExpected = "{ { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 }, { 97, 98 } }";
		assertEquals( theExpected, VerboseTextBuilder.asString( theSequence.getChunks() ) );
	}

	/**
	 * Test truncate tail 2.
	 */
	@Test
	public void testTruncateTail2() {

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "### " + RuntimeUtility.getCallerStackTraceElement().getMethodName() + " ###\n" );
		}

		byte[] theNumbers = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };
		byte[] theChars = new byte[] { 'a', 'b', 'c' };

		ByteArraySequence theSequence = new ByteArraySequence( theNumbers, theChars );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		theSequence.truncateTail( 2 );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		String theExpected = "{ { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 }, { 97 } }";
		assertEquals( theExpected, VerboseTextBuilder.asString( theSequence.getChunks() ) );
	}

	/**
	 * Test truncate tail 3.
	 */
	@Test
	public void testTruncateTail3() {

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "### " + RuntimeUtility.getCallerStackTraceElement().getMethodName() + " ###\n" );
		}

		byte[] theNumbers = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };
		byte[] theChars = new byte[] { 'a', 'b', 'c' };

		ByteArraySequence theSequence = new ByteArraySequence( theNumbers, theChars );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		theSequence.truncateTail( 3 );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		String theExpected = "{ { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 } }";
		assertEquals( theExpected, VerboseTextBuilder.asString( theSequence.getChunks() ) );
	}

	/**
	 * Test truncate tail 4.
	 */
	@Test
	public void testTruncateTail4() {

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "### " + RuntimeUtility.getCallerStackTraceElement().getMethodName() + " ###\n" );
		}

		byte[] theNumbers = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };
		byte[] theChars = new byte[] { 'a', 'b', 'c' };

		ByteArraySequence theSequence = new ByteArraySequence( theNumbers, theChars );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		theSequence.truncateTail( 4 );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		String theExpected = "{ { 0, 1, 2, 3, 4, 5, 6, 7, 8 } }";
		assertEquals( theExpected, VerboseTextBuilder.asString( theSequence.getChunks() ) );
	}

	/**
	 * Test truncate tail 5.
	 */
	@Test
	public void testTruncateTail5() {

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "### " + RuntimeUtility.getCallerStackTraceElement().getMethodName() + " ###\n" );
		}

		byte[] theNumbers = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };
		byte[] theChars = new byte[] { 'a', 'b', 'c' };

		ByteArraySequence theSequence = new ByteArraySequence( theNumbers, theChars );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		theSequence.truncateTail( 13 );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		String theExpected = "{}";
		assertEquals( theExpected, VerboseTextBuilder.asString( theSequence.getChunks() ) );
	}

	/**
	 * Test truncate tail 6.
	 */
	@Test
	public void testTruncateTail6() {

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "### " + RuntimeUtility.getCallerStackTraceElement().getMethodName() + " ###\n" );
		}

		byte[] theNumbers = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };
		byte[] theChars = new byte[] { 'a', 'b', 'c' };

		ByteArraySequence theSequence = new ByteArraySequence( theNumbers, theChars );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		theSequence.truncateTail( 14 );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		String theExpected = "{}";
		assertEquals( theExpected, VerboseTextBuilder.asString( theSequence.getChunks() ) );
	}

	/**
	 * Test truncate head 1.
	 */
	@Test
	public void testTruncateHead1() {

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "### " + RuntimeUtility.getCallerStackTraceElement().getMethodName() + " ###\n" );
		}

		byte[] theNumbers = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };
		byte[] theChars = new byte[] { 'a', 'b', 'c' };

		ByteArraySequence theSequence = new ByteArraySequence( theNumbers, theChars );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		theSequence.truncateHead( 1 );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		String theExpected = "{ { 1, 2, 3, 4, 5, 6, 7, 8, 9 }, { 97, 98, 99 } }";
		assertEquals( theExpected, VerboseTextBuilder.asString( theSequence.getChunks() ) );
	}

	/**
	 * Test truncate head 2.
	 */
	@Test
	public void testTruncateHead2() {

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "### " + RuntimeUtility.getCallerStackTraceElement().getMethodName() + " ###\n" );
		}

		byte[] theNumbers = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };
		byte[] theChars = new byte[] { 'a', 'b', 'c' };

		ByteArraySequence theSequence = new ByteArraySequence( theNumbers, theChars );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		theSequence.truncateHead( 9 );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		String theExpected = "{ { 9 }, { 97, 98, 99 } }";
		assertEquals( theExpected, VerboseTextBuilder.asString( theSequence.getChunks() ) );
	}

	/**
	 * Test truncate head 3.
	 */
	@Test
	public void testTruncateHead3() {

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "### " + RuntimeUtility.getCallerStackTraceElement().getMethodName() + " ###\n" );
		}

		byte[] theNumbers = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };
		byte[] theChars = new byte[] { 'a', 'b', 'c' };

		ByteArraySequence theSequence = new ByteArraySequence( theNumbers, theChars );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		theSequence.truncateHead( 10 );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		String theExpected = "{ { 97, 98, 99 } }";
		assertEquals( theExpected, VerboseTextBuilder.asString( theSequence.getChunks() ) );
	}

	/**
	 * Test truncate head 4.
	 */
	@Test
	public void testTruncateHead4() {

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "### " + RuntimeUtility.getCallerStackTraceElement().getMethodName() + " ###\n" );
		}

		byte[] theNumbers = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };
		byte[] theChars = new byte[] { 'a', 'b', 'c' };

		ByteArraySequence theSequence = new ByteArraySequence( theNumbers, theChars );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		theSequence.truncateHead( 11 );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		String theExpected = "{ { 98, 99 } }";
		assertEquals( theExpected, VerboseTextBuilder.asString( theSequence.getChunks() ) );
	}

	/**
	 * Test truncate head 5.
	 */
	@Test
	public void testTruncateHead5() {

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "### " + RuntimeUtility.getCallerStackTraceElement().getMethodName() + " ###\n" );
		}

		byte[] theNumbers = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };
		byte[] theChars = new byte[] { 'a', 'b', 'c' };

		ByteArraySequence theSequence = new ByteArraySequence( theNumbers, theChars );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		theSequence.truncateHead( 13 );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		String theExpected = "{}";
		assertEquals( theExpected, VerboseTextBuilder.asString( theSequence.getChunks() ) );
	}

	/**
	 * Test truncate head 6.
	 */
	@Test
	public void testTruncateHead6() {

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "### " + RuntimeUtility.getCallerStackTraceElement().getMethodName() + " ###\n" );
		}

		byte[] theNumbers = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };
		byte[] theChars = new byte[] { 'a', 'b', 'c' };

		ByteArraySequence theSequence = new ByteArraySequence( theNumbers, theChars );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		theSequence.truncateHead( 14 );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		String theExpected = "{}";
		assertEquals( theExpected, VerboseTextBuilder.asString( theSequence.getChunks() ) );
	}

	/**
	 * Test write 1.
	 */
	@Test
	public void testWrite1() {

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "### " + RuntimeUtility.getCallerStackTraceElement().getMethodName() + " ###\n" );
		}

		byte[] theNumbers = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };

		ByteArraySequence theSequence = new ByteArraySequence();

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		theSequence.overwrite( 1, theNumbers );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		String theExpected = "{ { 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 } }";
		assertEquals( theExpected, VerboseTextBuilder.asString( theSequence.getChunks() ) );
	}

	/**
	 * Test write 2.
	 */
	@Test
	public void testWrite2() {

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "### " + RuntimeUtility.getCallerStackTraceElement().getMethodName() + " ###\n" );
		}

		byte[] theChars = new byte[] { 'a', 'b', 'c' };
		byte[] theNumbers = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };

		ByteArraySequence theSequence = new ByteArraySequence( theChars );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		theSequence.overwrite( 1, theNumbers );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		String theExpected = "{ { 97, 0, 1 }, { 2, 3, 4, 5, 6, 7, 8, 9 } }";
		assertEquals( theExpected, VerboseTextBuilder.asString( theSequence.getChunks() ) );
	}

	/**
	 * Test write 3.
	 */
	@Test
	public void testWrite3() {

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "### " + RuntimeUtility.getCallerStackTraceElement().getMethodName() + " ###\n" );
		}

		byte[] theChars = new byte[] { 'a', 'b', 'c' };
		byte[] theNumbers = new byte[] { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };

		ByteArraySequence theSequence = new ByteArraySequence( theChars );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		theSequence.overwrite( 5, theNumbers );

		if ( IS_LOG_TEST_ENABLED ) {
			printSequence( theSequence );
		}

		String theExpected = "{ { 97, 98, 99 }, { 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 } }";
		assertEquals( theExpected, VerboseTextBuilder.asString( theSequence.getChunks() ) );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	private void printSequence( ByteArraySequence theSequence ) {
		System.out.println( theSequence.toString() );
		System.out.println( VerboseTextBuilder.asString( theSequence.getChunks() ) );
		System.out.println();
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
