// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.serial;

import static org.junit.jupiter.api.Assertions.*;

import org.junit.jupiter.api.Test;
import org.refcodes.data.Text;
import org.refcodes.numerical.CrcAlgorithmConfig;
import org.refcodes.numerical.Endianess;

import static org.refcodes.serial.SerialSugar.*;

public class AllocSegmentTest {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static boolean IS_LOG_TEST_ENABLED = Boolean.getBoolean( "log.test");

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Test alloc segment decorator.
	 *
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testAllocSegmentDecorator() throws TransmissionException {
		for ( int eLengthWidth = 2; eLengthWidth <= 8; eLengthWidth++ ) {
			for ( Endianess eEndianess : Endianess.values() ) {

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "---------- Length width = <" + eLengthWidth + ">, endianess = <" + eEndianess + "> ----------" );
				}

				// @formatter:off
				AllocSectionDecoratorSegment<StringSection> theSegment = allocSegment(
					stringSection(Text.ARECIBO_MESSAGE.toString()), eLengthWidth, eEndianess
				);
				// @formatter:on

				Sequence theSequence = theSegment.toSequence();

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Sequence (hex) = " + theSequence.toHexString() );
					System.out.println( "Transmission = " + theSegment.toString() );
					System.out.println( "AbstractSchema = " + theSegment.toSchema() );
				}

				// @formatter:off
				Segment theOtherSegment = allocSegment(
					stringSection(), eLengthWidth, eEndianess
				);
				// @formatter:on

				theOtherSegment.fromTransmission( theSequence );

				assertEquals( theSegment, theOtherSegment );

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Other segment = " + theOtherSegment.toString() );
					System.out.println( "Other schema = " + theOtherSegment.toSchema() );
				}
			}
		}
	}

	/**
	 * Test length segment.
	 *
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testLengthSegment() throws TransmissionException {
		for ( int eLengthWidth = 2; eLengthWidth <= 8; eLengthWidth++ ) {
			for ( Endianess eEndianess : Endianess.values() ) {

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "---------- Length width = <" + eLengthWidth + ">, endianess = <" + eEndianess + "> ----------" );
				}
				LengthSegment<Segment> theLen;

				// @formatter:off
				Segment theSegment = crcPrefixSegment(							//    2 bytes
					segmentComposite(
						theLen = segmentLength(eLengthWidth, eEndianess),		//  + x bytes
								theLen.letReferencee( segmentComposite(
								booleanSegment( true ),							//  + 1 byte
								intSegment(5161 ),								//  + 4 bytes
								allocSegment(									//  + 4 bytes
									stringSection("Hello World!")				// + 12 bytes
								)
							)
						)
					), CrcAlgorithmConfig.CRC_16_CCITT_FALSE
				);                                                              // = 23 + x bytes
				// @formatter:on

				Sequence theSequence = theSegment.toSequence();

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Sequence (hex) = " + theSequence.toHexString() );
					System.out.println( "Transmission = " + theSegment.toString() );
					System.out.println( "AbstractSchema = " + theSegment.toSchema() );
				}

				int theLength = theSegment.getLength();
				assertEquals( 23 + eLengthWidth, theLength );
				assertEquals( 23 + eLengthWidth, theSequence.getLength() );

				LengthSegment<Segment> theOtherLen;

				// @formatter:off
				Segment theOtherSegment = crcPrefixSegment(
					segmentComposite(
						theOtherLen = segmentLength(eLengthWidth, eEndianess),
							theOtherLen.letReferencee( segmentComposite(
								booleanSegment( true ),
								intSegment(), 
								allocSegment( 
									stringSection()
								)
							)
						)
					), CrcAlgorithmConfig.CRC_16_CCITT_FALSE
				);
				// @formatter:on

				theOtherSegment.fromTransmission( theSequence );

				assertEquals( theSegment, theOtherSegment );

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Other segment = " + theOtherSegment.toString() );
					System.out.println( "Other schema = " + theOtherSegment.toSchema() );
				}

				assertEquals( 23 + eLengthWidth, theOtherSegment.getLength() );
			}
		}
	}

	/**
	 * Test alloc head body segment decorator 1.
	 *
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testAllocHeadBodySegmentDecorator1() throws TransmissionException {
		for ( int eLengthWidth = 2; eLengthWidth <= 8; eLengthWidth++ ) {
			for ( Endianess eEndianess : Endianess.values() ) {

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "---------- Length width = <" + eLengthWidth + ">, endianess = <" + eEndianess + "> ----------" );
				}

				AllocSegmentHead theLen;

				// @formatter:off
				Segment theSegment = crcPrefixSegment(                    		//    2 bytes
					segmentComposite(
						theLen = allocSegmentHead(eLengthWidth, eEndianess),    //  + x bytes
						theLen.letBody( 
							allocSegmentBody( 
								stringSection("Hello World!")                   // + 12 bytes
							)
						)
					), CrcAlgorithmConfig.CRC_16_CCITT_FALSE
				);                                                              // = 14 + x bytes
				// @formatter:on

				Sequence theSequence = theSegment.toSequence();

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Sequence (hex) = " + theSequence.toHexString() );
					System.out.println( "Transmission = " + theSegment.toString() );
					System.out.println( "AbstractSchema = " + theSegment.toSchema() );
				}

				assertEquals( 14 + eLengthWidth, theSegment.getLength() );
				assertEquals( 14 + eLengthWidth, theSequence.getLength() );

				AllocSegmentHead theOtherLen;

				// @formatter:off
				Segment theOtherSegment = crcPrefixSegment(
					segmentComposite(
						theOtherLen = allocSegmentHead(eLengthWidth, eEndianess),
						theOtherLen.letBody( 
							allocSegmentBody( 
								stringSection()
							)
						)
					), CrcAlgorithmConfig.CRC_16_CCITT_FALSE
				);
				// @formatter:on

				theOtherSegment.fromTransmission( theSequence );

				assertEquals( theOtherSegment, theOtherSegment );

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Other segment = " + theOtherSegment.toString() );
					System.out.println( "Other schema = " + theOtherSegment.toSchema() );
				}

				assertEquals( 14 + eLengthWidth, theOtherSegment.getLength() );
			}
		}
	}

	/**
	 * Test alloc head body segment decorator 2.
	 *
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testAllocHeadBodySegmentDecorator2() throws TransmissionException {
		for ( int eLengthWidth = 2; eLengthWidth <= 8; eLengthWidth++ ) {
			for ( Endianess eEndianess : Endianess.values() ) {

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "---------- Length width = <" + eLengthWidth + ">, endianess = <" + eEndianess + "> ----------" );
				}

				AllocSegmentHead theLen;

				// @formatter:off
				Segment theSegment = crcPrefixSegment(
					segmentComposite(
						theLen = allocSegmentHead(eLengthWidth, eEndianess),
						booleanSegment( true ),
						intSegment( 5161 ), 
						theLen.letBody(
							allocSegmentBody( 
								stringSection(Text.ARECIBO_MESSAGE.toString())
							)
						)
					), CrcAlgorithmConfig.CRC_16_CCITT_FALSE
				);
				// @formatter:on

				Sequence theSequence = theSegment.toSequence();

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Sequence (hex) = " + theSequence.toHexString() );
					System.out.println( "Transmission = " + theSegment.toString() );
					System.out.println( "AbstractSchema = " + theSegment.toSchema() );
				}

				AllocSegmentHead theOtherLen;

				// @formatter:off
				Segment theOtherSegment = crcPrefixSegment(
					segmentComposite(
						theOtherLen = allocSegmentHead(eLengthWidth, eEndianess),
						booleanSegment(),
						intSegment(), 
						theOtherLen.letBody( 
							allocSegmentBody( 
								stringSection()
							)
						)
					), CrcAlgorithmConfig.CRC_16_CCITT_FALSE
				);
				// @formatter:on

				theOtherSegment.fromTransmission( theSequence );

				assertEquals( theOtherSegment, theOtherSegment );

				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( "Other segment = " + theOtherSegment.toString() );
					System.out.println( "Other schema = " + theOtherSegment.toSchema() );
				}
			}
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
