// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.serial;

import static org.junit.jupiter.api.Assertions.*;
import static org.refcodes.serial.SerialSugar.*;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;

import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.Test;
import org.refcodes.data.Text;
import org.refcodes.mixin.ConcatenateMode;
import org.refcodes.numerical.CrcAlgorithmConfig;
import org.refcodes.numerical.Endianess;
import org.refcodes.numerical.NumericalUtility;
import org.refcodes.serial.TestFixures.WeatherData;
import org.refcodes.textual.VerboseTextBuilder;

public class PacketStreamTest extends AbstractLoopbackPortTest {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	private static boolean IS_LOG_TEST_ENABLED = Boolean.getBoolean( "log.test");

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final int LOOPS = 3;

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * GOTCHA!
	 */
	@Test
	@Disabled
	public void testPackeStream() throws IOException, InterruptedException {
		if ( hasPorts() ) {
			Port<?> theTransmitPort = getTransmitterPort().withOpen();
			Port<?> theReceiverPort = getReceiverPort().withOpen();

			int theBlockSize = 64;
			PacketInputStream thePacketIn = PacketInputStream.builder().withInputStream( theReceiverPort.getInputStream() ).withBlockSize( theBlockSize ).build();
			PacketOutputStream thePacketOut = PacketOutputStream.builder().withOutputStream( theReceiverPort.getOutputStream() ).withBlockSize( theBlockSize ).build();

			byte[] theOutBytes = "Hallo Welt!".getBytes();
			byte[] theInBytes = new byte[theOutBytes.length];

			Thread t = new Thread( new Runnable() {
				@Override
				public void run() {
					int eRead;
					int i = 0;
					try {
						while ( (eRead = thePacketIn.read()) != -1 && i < theOutBytes.length ) {
							if ( IS_LOG_TEST_ENABLED ) {
								System.out.println( i + " - " + (char) eRead );
							}
							theInBytes[i] = (byte) eRead;
							i++;
						}
						if ( IS_LOG_TEST_ENABLED ) {
							System.out.println( "EOT = " + eRead );
							// assertEquals( -1, eRead );
						}
						synchronized ( this ) {
							notifyAll();
						}
					}
					catch ( IOException e ) {
						System.out.println( e.getClass().getSimpleName() + ": " + e.getMessage() );
					}

				}
			} );
			t.start();

			thePacketOut.write( theOutBytes, 0, 3 );
			thePacketOut.flush();
			thePacketOut.write( theOutBytes, 3, theOutBytes.length - 3 );
			thePacketOut.flush();

			synchronized ( this ) {
				wait( 500 );
			}
			theTransmitPort.close();
			theReceiverPort.close();
			assertArrayEquals( theOutBytes, theInBytes );
		}
		else {
			System.out.println( "Skipping test, please connect your null modem cable to two serial ports on your box, seeking for exactly two FT232 (ftdi_sio) type ports!" );
		}
	}

	/**
	 * Test segment transmit receive 1.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testSegmentTransmitReceive1() throws IOException, TransmissionException {
		ByteArrayOutputStream theOutputStream = new ByteArrayOutputStream();
		StringSection theStringSection = new StringSection( Text.ARECIBO_MESSAGE.getText() );
		TransmissionMetrics theMetrics = TransmissionMetrics.builder().withEndianess( Endianess.LITTLE ).withCrcAlgorithm( CrcAlgorithmConfig.CRC_16_CCITT_FALSE ).withCrcChecksumConcatenateMode( ConcatenateMode.PREPEND ).withSequenceNumberWidth( 2 ).withSequenceNumberInitValue( 511 ).withBlockSize( 5 ).withAcknowledgeSegmentPackager( dummySegmentPackager() ).withPacketSegmentPackager( dummySegmentPackager() ).build();
		StopAndWaitPacketStreamSegmentDecorator<Segment> thePacketSection = stopAndWaitPacketStreamSegment( allocSegment( theStringSection ), theMetrics );
		thePacketSection.transmitTo( theOutputStream );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( NumericalUtility.toHexString( theOutputStream.toByteArray(), ":", 7 ) );
		}

		StringSection theOtherStringSection = new StringSection();
		StopAndWaitPacketStreamSegmentDecorator<Segment> theOtherPacketSection = stopAndWaitPacketStreamSegment( allocSegment( theOtherStringSection ), theMetrics );
		ByteArrayInputStream theInputStream = new ByteArrayInputStream( theOutputStream.toByteArray() );
		theOtherPacketSection.receiveFrom( theInputStream );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "String section: \"" + theStringSection.getPayload() + "\"" );
			System.out.println( "Other string section: \"" + theOtherStringSection.getPayload() + "\"" );
		}

		assertEquals( theStringSection, theOtherStringSection );
	}

	/**
	 * Test segment transmit receive 1 crc.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testSegmentTransmitReceive1Crc() throws IOException, TransmissionException {
		ByteArrayOutputStream theOutputStream = new ByteArrayOutputStream();
		StringSection theStringSection = new StringSection( Text.ARECIBO_MESSAGE.getText() );
		TransmissionMetrics theMetrics = TransmissionMetrics.builder().withEndianess( Endianess.LITTLE ).withCrcAlgorithm( CrcAlgorithmConfig.CRC_16_CCITT_FALSE ).withCrcChecksumConcatenateMode( ConcatenateMode.PREPEND ).withSequenceNumberWidth( 2 ).withSequenceNumberInitValue( 511 ).withBlockSize( 5 ).build();
		StopAndWaitPacketStreamSegmentDecorator<Segment> thePacketSection = stopAndWaitPacketStreamSegment( allocSegment( theStringSection ), theMetrics );
		thePacketSection.transmitTo( theOutputStream );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( NumericalUtility.toHexString( theOutputStream.toByteArray(), ":", 9 ) );
		}

		StringSection theOtherStringSection = new StringSection();
		StopAndWaitPacketStreamSegmentDecorator<Segment> theOtherPacketSection = stopAndWaitPacketStreamSegment( allocSegment( theOtherStringSection ), theMetrics );
		ByteArrayInputStream theInputStream = new ByteArrayInputStream( theOutputStream.toByteArray() );
		theOtherPacketSection.receiveFrom( theInputStream );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "String section: \"" + theStringSection.getPayload() + "\"" );
			System.out.println( "Other string section: \"" + theOtherStringSection.getPayload() + "\"" );
		}

		assertEquals( theStringSection, theOtherStringSection );
	}

	/**
	 * Test stop and wait segment transmit receive 1.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws TransmissionException the transmission exception
	 * @throws InterruptedException the interrupted exception
	 */
	@Test
	public void testStopAndWaitSegmentTransmitReceive1() throws IOException, TransmissionException, InterruptedException {
		if ( hasPorts() ) {
			Port<?> theTransmitPort = getTransmitterPort().withOpen();
			Port<?> theReceiverPort = getReceiverPort().withOpen();

			StringSection theStringSection = new StringSection( Text.ARECIBO_MESSAGE.getText() );

			TransmissionMetrics theMetrics = TransmissionMetrics.builder().withEndianess( Endianess.LITTLE ).withCrcAlgorithm( CrcAlgorithmConfig.CRC_16_CCITT_FALSE ).withCrcChecksumConcatenateMode( ConcatenateMode.PREPEND ).withSequenceNumberWidth( 2 ).withSequenceNumberInitValue( 511 ).withBlockSize( 256 ).withAcknowledgeSegmentPackager( dummySegmentPackager() ).withPacketSegmentPackager( dummySegmentPackager() ).build();
			StopAndWaitPacketStreamSegmentDecorator<Segment> thePacketSection = stopAndWaitPacketStreamSegment( allocSegment( theStringSection ), theMetrics );

			StringSection theOtherStringSection = new StringSection();
			StopAndWaitPacketStreamSegmentDecorator<Segment> theOtherPacketSection = stopAndWaitPacketStreamSegment( allocSegment( theOtherStringSection ), theMetrics );

			SegmentResult<Segment> theResult = theReceiverPort.onReceiveSegment( theOtherPacketSection );
			thePacketSection.transmitTo( theTransmitPort );
			theResult.waitForResult();

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "String section: \"" + theStringSection.getPayload() + "\"" );
				System.out.println( "Other string section: \"" + theOtherStringSection.getPayload() + "\"" );
			}

			assertEquals( theStringSection, theOtherStringSection );

			theTransmitPort.close();
			theReceiverPort.close();
		}
		else {
			System.out.println( "Skipping test, please connect your null modem cable to two serial ports on your box, seeking for exactly two FT232 (ftdi_sio) type ports!" );
		}
	}

	/**
	 * Test stop and wait segment transmit receive 1 crc.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws TransmissionException the transmission exception
	 * @throws InterruptedException the interrupted exception
	 */
	@Test
	public void testStopAndWaitSegmentTransmitReceive1Crc() throws IOException, TransmissionException, InterruptedException {
		if ( hasPorts() ) {
			Port<?> theTransmitPort = getTransmitterPort().withOpen();
			Port<?> theReceiverPort = getReceiverPort().withOpen();

			StringSection theStringSection = new StringSection( Text.ARECIBO_MESSAGE.getText() );

			TransmissionMetrics theMetrics = TransmissionMetrics.builder().withEndianess( Endianess.LITTLE ).withCrcAlgorithm( CrcAlgorithmConfig.CRC_16_CCITT_FALSE ).withCrcChecksumConcatenateMode( ConcatenateMode.PREPEND ).withSequenceNumberWidth( 2 ).withSequenceNumberInitValue( 511 ).withBlockSize( 256 ).build();
			StopAndWaitPacketStreamSegmentDecorator<Segment> thePacketSection = stopAndWaitPacketStreamSegment( allocSegment( theStringSection ), theMetrics );

			StringSection theOtherStringSection = new StringSection();
			StopAndWaitPacketStreamSegmentDecorator<Segment> theOtherPacketSection = stopAndWaitPacketStreamSegment( allocSegment( theOtherStringSection ), theMetrics );

			SegmentResult<Segment> theResult = theReceiverPort.onReceiveSegment( theOtherPacketSection );
			thePacketSection.transmitTo( theTransmitPort );
			theResult.waitForResult();

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "String section: \"" + theStringSection.getPayload() + "\"" );
				System.out.println( "Other string section: \"" + theOtherStringSection.getPayload() + "\"" );
			}

			assertEquals( theStringSection, theOtherStringSection );

			theTransmitPort.close();
			theReceiverPort.close();
		}
		else {
			System.out.println( "Skipping test, please connect your null modem cable to two serial ports on your box, seeking for exactly two FT232 (ftdi_sio) type ports!" );
		}
	}

	/**
	 * Test segment transmit receive 2.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testSegmentTransmitReceive2() throws IOException, TransmissionException {
		ByteArrayOutputStream theOutputStream = new ByteArrayOutputStream();
		StringArraySection theStringSection = new StringArraySection( "AAAAA", "BBBBB", "CCCCC", "DDDDD", "EEEEE", "" );
		TransmissionMetrics theMetrics = TransmissionMetrics.builder().withEndianess( Endianess.LITTLE ).withCrcAlgorithm( CrcAlgorithmConfig.CRC_16_CCITT_FALSE ).withCrcChecksumConcatenateMode( ConcatenateMode.PREPEND ).withSequenceNumberWidth( 2 ).withSequenceNumberInitValue( 511 ).withBlockSize( 5 ).withAcknowledgeSegmentPackager( dummySegmentPackager() ).withPacketSegmentPackager( dummySegmentPackager() ).build();
		StopAndWaitPacketStreamSegmentDecorator<Segment> thePacketSection = stopAndWaitPacketStreamSegment( allocSegment( theStringSection ), theMetrics );
		thePacketSection.transmitTo( theOutputStream );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( NumericalUtility.toHexString( theOutputStream.toByteArray(), ":", 7 ) );
		}

		StringArraySection theOtherStringSection = new StringArraySection();
		StopAndWaitPacketStreamSegmentDecorator<Segment> theOtherPacketSection = stopAndWaitPacketStreamSegment( allocSegment( theOtherStringSection ), theMetrics );
		ByteArrayInputStream theInputStream = new ByteArrayInputStream( theOutputStream.toByteArray() );
		theOtherPacketSection.receiveFrom( theInputStream );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( VerboseTextBuilder.asString( theOtherStringSection.getPayload() ) );
		}

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "String section: " + VerboseTextBuilder.asString( theStringSection.getPayload() ) );
			System.out.println( "Other string section: " + VerboseTextBuilder.asString( theOtherStringSection.getPayload() ) );
		}

		assertArrayEquals( theStringSection.getPayload(), theOtherStringSection.getPayload() );
	}

	/**
	 * Test segment transmit receive 2 crc.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testSegmentTransmitReceive2Crc() throws IOException, TransmissionException {
		ByteArrayOutputStream theOutputStream = new ByteArrayOutputStream();
		StringArraySection theStringSection = new StringArraySection( "AAAAA", "BBBBB", "CCCCC", "DDDDD", "EEEEE", "" );
		TransmissionMetrics theMetrics = TransmissionMetrics.builder().withEndianess( Endianess.LITTLE ).withCrcAlgorithm( CrcAlgorithmConfig.CRC_16_CCITT_FALSE ).withCrcChecksumConcatenateMode( ConcatenateMode.PREPEND ).withSequenceNumberWidth( 2 ).withSequenceNumberInitValue( 511 ).withBlockSize( 5 ).build();
		StopAndWaitPacketStreamSegmentDecorator<Segment> thePacketSection = stopAndWaitPacketStreamSegment( allocSegment( theStringSection ), theMetrics );
		thePacketSection.transmitTo( theOutputStream );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( NumericalUtility.toHexString( theOutputStream.toByteArray(), ":", 9 ) );
		}

		StringArraySection theOtherStringSection = new StringArraySection();
		StopAndWaitPacketStreamSegmentDecorator<Segment> theOtherPacketSection = stopAndWaitPacketStreamSegment( allocSegment( theOtherStringSection ), theMetrics );
		ByteArrayInputStream theInputStream = new ByteArrayInputStream( theOutputStream.toByteArray() );
		theOtherPacketSection.receiveFrom( theInputStream );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( VerboseTextBuilder.asString( theOtherStringSection.getPayload() ) );
		}

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "String section: " + VerboseTextBuilder.asString( theStringSection.getPayload() ) );
			System.out.println( "Other string section: " + VerboseTextBuilder.asString( theOtherStringSection.getPayload() ) );
		}

		assertArrayEquals( theStringSection.getPayload(), theOtherStringSection.getPayload() );
	}

	/**
	 * Test stop and wait segment transmit receive 2.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws TransmissionException the transmission exception
	 * @throws InterruptedException the interrupted exception
	 */
	@Test
	public void testStopAndWaitSegmentTransmitReceive2() throws IOException, TransmissionException, InterruptedException {
		if ( hasPorts() ) {
			Port<?> theTransmitPort = getTransmitterPort().withOpen();
			Port<?> theReceiverPort = getReceiverPort().withOpen();

			StringArraySection theStringSection = new StringArraySection( "AAAAA", "BBBBB", "CCCCC", "DDDDD", "EEEEE", "" );

			TransmissionMetrics theMetrics = TransmissionMetrics.builder().withEndianess( Endianess.LITTLE ).withCrcAlgorithm( CrcAlgorithmConfig.CRC_16_CCITT_FALSE ).withCrcChecksumConcatenateMode( ConcatenateMode.PREPEND ).withSequenceNumberWidth( 2 ).withSequenceNumberInitValue( 0 ).withBlockSize( 5 ).withAcknowledgeSegmentPackager( dummySegmentPackager() ).withPacketSegmentPackager( dummySegmentPackager() ).build();
			StopAndWaitPacketStreamSegmentDecorator<Segment> thePacketSection = stopAndWaitPacketStreamSegment( allocSegment( theStringSection ), theMetrics );

			StringArraySection theOtherStringSection = new StringArraySection();
			StopAndWaitPacketStreamSegmentDecorator<Segment> theOtherPacketSection = stopAndWaitPacketStreamSegment( allocSegment( theOtherStringSection ), theMetrics );

			SegmentResult<Segment> theResult = theReceiverPort.onReceiveSegment( theOtherPacketSection );
			thePacketSection.transmitTo( theTransmitPort );
			theResult.waitForResult();

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "String section: " + VerboseTextBuilder.asString( theStringSection.getPayload() ) );
				System.out.println( "Other string section: " + VerboseTextBuilder.asString( theOtherStringSection.getPayload() ) );
			}

			assertArrayEquals( theStringSection.getPayload(), theOtherStringSection.getPayload() );

			theTransmitPort.close();
			theReceiverPort.close();
		}
		else {
			System.out.println( "Skipping test, please connect your null modem cable to two serial ports on your box, seeking for exactly two FT232 (ftdi_sio) type ports!" );
		}
	}

	/**
	 * Test stop and wait segment transmit receive 2 crc.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws TransmissionException the transmission exception
	 * @throws InterruptedException the interrupted exception
	 */
	@Test
	public void testStopAndWaitSegmentTransmitReceive2Crc() throws IOException, TransmissionException, InterruptedException {
		if ( hasPorts() ) {
			Port<?> theTransmitPort = getTransmitterPort().withOpen();
			Port<?> theReceiverPort = getReceiverPort().withOpen();

			StringArraySection theStringSection = new StringArraySection( "AAAAA", "BBBBB", "CCCCC", "DDDDD", "EEEEE", "" );

			TransmissionMetrics theMetrics = TransmissionMetrics.builder().withEndianess( Endianess.LITTLE ).withCrcAlgorithm( CrcAlgorithmConfig.CRC_16_CCITT_FALSE ).withCrcChecksumConcatenateMode( ConcatenateMode.PREPEND ).withSequenceNumberWidth( 2 ).withSequenceNumberInitValue( 0 ).withBlockSize( 5 ).build();
			StopAndWaitPacketStreamSegmentDecorator<Segment> thePacketSection = stopAndWaitPacketStreamSegment( allocSegment( theStringSection ), theMetrics );

			StringArraySection theOtherStringSection = new StringArraySection();
			StopAndWaitPacketStreamSegmentDecorator<Segment> theOtherPacketSection = stopAndWaitPacketStreamSegment( allocSegment( theOtherStringSection ), theMetrics );

			SegmentResult<Segment> theResult = theReceiverPort.onReceiveSegment( theOtherPacketSection );
			thePacketSection.transmitTo( theTransmitPort );
			theResult.waitForResult();

			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "String section: " + VerboseTextBuilder.asString( theStringSection.getPayload() ) );
				System.out.println( "Other string section: " + VerboseTextBuilder.asString( theOtherStringSection.getPayload() ) );
			}

			assertArrayEquals( theStringSection.getPayload(), theOtherStringSection.getPayload() );

			theTransmitPort.close();
			theReceiverPort.close();
		}
		else {
			System.out.println( "Skipping test, please connect your null modem cable to two serial ports on your box, seeking for exactly two FT232 (ftdi_sio) type ports!" );
		}
	}

	// -------------------------------------------------------------------------

	/**
	 * Test section transmit receive 1.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testSectionTransmitReceive1() throws IOException, TransmissionException {
		ByteArrayOutputStream theOutputStream = new ByteArrayOutputStream();
		StringSection theStringSection = new StringSection( Text.ARECIBO_MESSAGE.getText() );
		TransmissionMetrics theMetrics = TransmissionMetrics.builder().withEndianess( Endianess.LITTLE ).withCrcAlgorithm( CrcAlgorithmConfig.CRC_16_CCITT_FALSE ).withCrcChecksumConcatenateMode( ConcatenateMode.PREPEND ).withSequenceNumberWidth( 2 ).withSequenceNumberInitValue( 0 ).withBlockSize( 5 ).withAcknowledgeSegmentPackager( dummySegmentPackager() ).withPacketSegmentPackager( dummySegmentPackager() ).build();
		StopAndWaitPacketStreamSectionDecorator<StringSection> thePacketSection = stopAndWaitPacketStreamSection( theStringSection, theMetrics );
		thePacketSection.transmitTo( theOutputStream );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( NumericalUtility.toHexString( theOutputStream.toByteArray(), ":", 7 ) );
		}

		StringSection theOtherStringSection = new StringSection();
		StopAndWaitPacketStreamSectionDecorator<StringSection> theOtherPacketSection = stopAndWaitPacketStreamSection( theOtherStringSection, theMetrics );
		ByteArrayInputStream theInputStream = new ByteArrayInputStream( theOutputStream.toByteArray() );
		theOtherPacketSection.receiveFrom( theInputStream, thePacketSection.getLength() );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "String section: \"" + theStringSection.getPayload() + "\"" );
			System.out.println( "Other string section: \"" + theOtherStringSection.getPayload() + "\"" );
		}

		assertEquals( theStringSection, theOtherStringSection );
	}

	/**
	 * Test section transmit receive 1 crc.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testSectionTransmitReceive1Crc() throws IOException, TransmissionException {
		ByteArrayOutputStream theOutputStream = new ByteArrayOutputStream();
		StringSection theStringSection = new StringSection( Text.ARECIBO_MESSAGE.getText() );
		TransmissionMetrics theMetrics = TransmissionMetrics.builder().withEndianess( Endianess.LITTLE ).withCrcAlgorithm( CrcAlgorithmConfig.CRC_16_CCITT_FALSE ).withCrcChecksumConcatenateMode( ConcatenateMode.PREPEND ).withSequenceNumberWidth( 2 ).withSequenceNumberInitValue( 511 ).withBlockSize( 5 ).build();
		StopAndWaitPacketStreamSectionDecorator<StringSection> thePacketSection = stopAndWaitPacketStreamSection( theStringSection, theMetrics );
		thePacketSection.transmitTo( theOutputStream );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( NumericalUtility.toHexString( theOutputStream.toByteArray(), ":", 9 ) );
		}

		StringSection theOtherStringSection = new StringSection();
		StopAndWaitPacketStreamSectionDecorator<StringSection> theOtherPacketSection = stopAndWaitPacketStreamSection( theOtherStringSection, theMetrics );
		ByteArrayInputStream theInputStream = new ByteArrayInputStream( theOutputStream.toByteArray() );
		theOtherPacketSection.receiveFrom( theInputStream, thePacketSection.getLength() );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "String section: \"" + theStringSection.getPayload() + "\"" );
			System.out.println( "Other string section: \"" + theOtherStringSection.getPayload() + "\"" );
		}

		assertEquals( theStringSection, theOtherStringSection );
	}

	/**
	 * Test section transmit receive 2.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testSectionTransmitReceive2() throws IOException, TransmissionException {
		ByteArrayOutputStream theOutputStream = new ByteArrayOutputStream();
		StringArraySection theStringSection = new StringArraySection( "AAAAA", "BBBBB", "CCCCC", "DDDDD", "EEEEE", "" );
		TransmissionMetrics theMetrics = TransmissionMetrics.builder().withEndianess( Endianess.LITTLE ).withCrcAlgorithm( CrcAlgorithmConfig.CRC_16_CCITT_FALSE ).withCrcChecksumConcatenateMode( ConcatenateMode.PREPEND ).withSequenceNumberWidth( 2 ).withSequenceNumberInitValue( 511 ).withBlockSize( 5 ).withAcknowledgeSegmentPackager( dummySegmentPackager() ).withPacketSegmentPackager( dummySegmentPackager() ).build();
		StopAndWaitPacketStreamSectionDecorator<StringArraySection> thePacketSection = stopAndWaitPacketStreamSection( theStringSection, theMetrics );
		thePacketSection.transmitTo( theOutputStream );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( NumericalUtility.toHexString( theOutputStream.toByteArray(), ":", 7 ) );
		}

		StringArraySection theOtherStringSection = new StringArraySection();
		StopAndWaitPacketStreamSectionDecorator<StringArraySection> theOtherPacketSection = stopAndWaitPacketStreamSection( theOtherStringSection, theMetrics );
		ByteArrayInputStream theInputStream = new ByteArrayInputStream( theOutputStream.toByteArray() );
		theOtherPacketSection.receiveFrom( theInputStream, thePacketSection.getLength() );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( VerboseTextBuilder.asString( theOtherStringSection.getPayload() ) );
		}

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "String section: " + VerboseTextBuilder.asString( theStringSection.getPayload() ) );
			System.out.println( "Other string section: " + VerboseTextBuilder.asString( theOtherStringSection.getPayload() ) );
		}

		assertArrayEquals( theStringSection.getPayload(), theOtherStringSection.getPayload() );
	}

	/**
	 * Test section transmit receive 2 crc.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws TransmissionException the transmission exception
	 */
	@Test
	public void testSectionTransmitReceive2Crc() throws IOException, TransmissionException {
		ByteArrayOutputStream theOutputStream = new ByteArrayOutputStream();
		StringArraySection theStringSection = new StringArraySection( "AAAAA", "BBBBB", "CCCCC", "DDDDD", "EEEEE", "" );
		TransmissionMetrics theMetrics = TransmissionMetrics.builder().withEndianess( Endianess.LITTLE ).withCrcAlgorithm( CrcAlgorithmConfig.CRC_16_CCITT_FALSE ).withCrcChecksumConcatenateMode( ConcatenateMode.PREPEND ).withSequenceNumberWidth( 2 ).withSequenceNumberInitValue( 511 ).withBlockSize( 5 ).build();
		StopAndWaitPacketStreamSectionDecorator<StringArraySection> thePacketSection = stopAndWaitPacketStreamSection( theStringSection, theMetrics );
		thePacketSection.transmitTo( theOutputStream );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( NumericalUtility.toHexString( theOutputStream.toByteArray(), ":", 9 ) );
		}

		StringArraySection theOtherStringSection = new StringArraySection();
		StopAndWaitPacketStreamSectionDecorator<StringArraySection> theOtherPacketSection = stopAndWaitPacketStreamSection( theOtherStringSection, theMetrics );
		ByteArrayInputStream theInputStream = new ByteArrayInputStream( theOutputStream.toByteArray() );
		theOtherPacketSection.receiveFrom( theInputStream, thePacketSection.getLength() );

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( VerboseTextBuilder.asString( theOtherStringSection.getPayload() ) );
		}

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "String section: " + VerboseTextBuilder.asString( theStringSection.getPayload() ) );
			System.out.println( "Other string section: " + VerboseTextBuilder.asString( theOtherStringSection.getPayload() ) );
		}

		assertArrayEquals( theStringSection.getPayload(), theOtherStringSection.getPayload() );
	}

	/**
	 * Test stop and wait packet stream segment decorator.
	 *
	 * @throws TransmissionSequenceException the transmission sequence exception
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws NoSuchPortExcpetion the no such port excpetion
	 * @throws InterruptedException the interrupted exception
	 */
	@Test
	public void testStopAndWaitPacketStreamSegmentDecorator() throws TransmissionSequenceException, IOException, NoSuchPortExcpetion, InterruptedException {
		if ( hasPorts() ) {
			Port<?> theTransmitPort = getTransmitterPort().withOpen();
			Port<?> theReceiverPort = getReceiverPort().withOpen();
			WeatherData theSenderData = TestFixures.createWeatherData();
			ComplexTypeSegment<WeatherData> theReceiverComplexSegment;
			Segment theSenderSeg = stopAndWaitPacketStreamSegmentBuilder().withDecoratee( crcPrefixSegment( complexTypeSegment( theSenderData ), CrcAlgorithmConfig.CRC_16_CCITT_FALSE ) ).withBlockSize( 8 ).build();
			Segment theReceiverSeg = stopAndWaitPacketStreamSegmentBuilder().withDecoratee( crcPrefixSegment( theReceiverComplexSegment = complexTypeSegment( WeatherData.class ), CrcAlgorithmConfig.CRC_16_CCITT_FALSE ) ).withBlockSize( 8 ).build();

			for ( int i = 0; i < LOOPS; i++ ) {
				SegmentResult<Segment> theResult = theReceiverPort.onReceiveSegment( theReceiverSeg );
				theSenderSeg.transmitTo( theTransmitPort );
				theResult.waitForResult();

				WeatherData theReceiverData = theReceiverComplexSegment.getPayload();
				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( theReceiverData );
				}
				assertEquals( theSenderData, theReceiverData );
				try {
					Thread.sleep( 1000 );
				}
				catch ( InterruptedException e ) {}
			}

			theTransmitPort.close();
			theReceiverPort.close();
		}
		else {
			System.out.println( "Skipping test, please connect your null modem cable to two serial ports on your box, seeking for exactly two FT232 (ftdi_sio) type ports!" );
		}
	}

	@Test
	public void testEdgeCase1() throws IOException, InterruptedException {

		ByteArrayOutputStream theByteOut = new ByteArrayOutputStream();

		int theBlockSize = 6;

		PacketOutputStream thePacketOut = PacketOutputStream.builder().withOutputStream( theByteOut ).withBlockSize( theBlockSize ).build();
		Sequence theSeqOut = new ByteArraySequence( new byte[] { 1, 2, -1, 4, -1, 0 } );
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Sequence:" );
			System.out.println( NumericalUtility.toHexString( ":", 10, theSeqOut.toBytes() ) );
		}

		theSeqOut.writeTo( thePacketOut );

		thePacketOut.flush();

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "ByteArrayOutputStream:" );
			System.out.println( NumericalUtility.toHexString( ":", thePacketOut.getPacketSize(), theByteOut.toByteArray() ) );
		}

		ByteArrayInputStream theByteIn = new ByteArrayInputStream( theByteOut.toByteArray() );

		try (PacketInputStream thePacketIn = PacketInputStream.builder().withInputStream( theByteIn ).withBlockSize( theBlockSize ).build()) {
			Sequence theInSeq = new ByteArraySequence( thePacketIn );
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "ByteArrayInputStream:" );
				System.out.println( NumericalUtility.toHexString( ":", thePacketOut.getPacketSize(), theInSeq.toBytes() ) );
			}

			assertArrayEquals( theSeqOut.toBytes(), theInSeq.toBytes() );
		}
	}

	@Test
	public void testEdgeCase2() throws IOException, InterruptedException {

		ByteArrayOutputStream theByteOut = new ByteArrayOutputStream();

		int theBlockSize = 1;

		PacketOutputStream thePacketOut = PacketOutputStream.builder().withOutputStream( theByteOut ).withBlockSize( theBlockSize ).build();

		Sequence theSeqOut = new ByteArraySequence( new byte[] { 1, 2, -1, 4, -1, 0 } );
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Sequence:" );
			System.out.println( NumericalUtility.toHexString( ":", thePacketOut.getPacketSize(), theSeqOut.toBytes() ) );
		}

		theSeqOut.writeTo( thePacketOut );

		thePacketOut.flush();

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "ByteArrayOutputStream:" );
			System.out.println( NumericalUtility.toHexString( ":", thePacketOut.getPacketSize(), theByteOut.toByteArray() ) );
		}

		ByteArrayInputStream theByteIn = new ByteArrayInputStream( theByteOut.toByteArray() );

		try (PacketInputStream thePacketIn = PacketInputStream.builder().withInputStream( theByteIn ).withBlockSize( theBlockSize ).build()) {
			Sequence theInSeq = new ByteArraySequence( thePacketIn );
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "ByteArrayInputStream:" );
				System.out.println( NumericalUtility.toHexString( ":", thePacketOut.getPacketSize(), theInSeq.toBytes() ) );
			}

			assertArrayEquals( theSeqOut.toBytes(), theInSeq.toBytes() );
		}
	}

	@Test
	public void testEdgeCase3() throws IOException, InterruptedException {

		ByteArrayOutputStream theByteOut = new ByteArrayOutputStream();

		int theBlockSize = 3;

		PacketOutputStream thePacketOut = PacketOutputStream.builder().withOutputStream( theByteOut ).withBlockSize( theBlockSize ).build();

		Sequence theSeqOut = new ByteArraySequence( new byte[] { 1, 2, -1, 4, -1, 0 } );
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Sequence:" );
			System.out.println( NumericalUtility.toHexString( ":", 10, theSeqOut.toBytes() ) );
		}

		theSeqOut.writeTo( thePacketOut );

		thePacketOut.flush();

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "ByteArrayOutputStream:" );
			System.out.println( NumericalUtility.toHexString( ":", thePacketOut.getPacketSize(), theByteOut.toByteArray() ) );
		}

		ByteArrayInputStream theByteIn = new ByteArrayInputStream( theByteOut.toByteArray() );

		try (PacketInputStream thePacketIn = PacketInputStream.builder().withInputStream( theByteIn ).withBlockSize( theBlockSize ).build()) {
			Sequence theInSeq = new ByteArraySequence( thePacketIn );
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "ByteArrayInputStream:" );
				System.out.println( NumericalUtility.toHexString( ":", 10, theInSeq.toBytes() ) );
			}

			assertArrayEquals( theSeqOut.toBytes(), theInSeq.toBytes() );
		}
	}

	@Test
	public void testEdgeCase4() throws IOException, InterruptedException {

		ByteArrayOutputStream theByteOut = new ByteArrayOutputStream();

		int theBlockSize = 10;

		PacketOutputStream thePacketOut = PacketOutputStream.builder().withOutputStream( theByteOut ).withBlockSize( theBlockSize ).build();

		Sequence theSeqOut = new ByteArraySequence( new byte[] { 1, 2, -1, 4, -1, 0 } );
		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "Sequence:" );
			System.out.println( NumericalUtility.toHexString( ":", 10, theSeqOut.toBytes() ) );
		}

		theSeqOut.writeTo( thePacketOut );

		thePacketOut.flush();

		if ( IS_LOG_TEST_ENABLED ) {
			System.out.println( "ByteArrayOutputStream:" );
			System.out.println( NumericalUtility.toHexString( ":", thePacketOut.getPacketSize(), theByteOut.toByteArray() ) );
		}

		ByteArrayInputStream theByteIn = new ByteArrayInputStream( theByteOut.toByteArray() );

		try (PacketInputStream thePacketIn = PacketInputStream.builder().withInputStream( theByteIn ).withBlockSize( theBlockSize ).build()) {
			Sequence theInSeq = new ByteArraySequence( thePacketIn );
			if ( IS_LOG_TEST_ENABLED ) {
				System.out.println( "ByteArrayInputStream:" );
				System.out.println( NumericalUtility.toHexString( ":", 10, theInSeq.toBytes() ) );
			}

			assertArrayEquals( theSeqOut.toBytes(), theInSeq.toBytes() );
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
