// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.serial;

import static org.junit.jupiter.api.Assertions.*;
import static org.refcodes.serial.SerialSugar.*;

import java.io.IOException;

import org.junit.jupiter.api.Test;
import org.refcodes.numerical.CrcAlgorithmConfig;
import org.refcodes.serial.TestFixures.WeatherData;
import org.refcodes.textual.VerboseTextBuilder;

public class ReadyToSendTest extends AbstractLoopbackPortTest {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static boolean IS_LOG_TEST_ENABLED = Boolean.getBoolean( "log.test");
	private static final int LOOPS = 3;

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@Test
	public void testReadyToSendSegmentDecorator1() throws IOException, InterruptedException {
		if ( hasPorts() ) {
			Port<?> theTransmitPort = getTransmitterPort().withOpen();
			Port<?> theReceiverPort = getReceiverPort().withOpen();
			WeatherData theSenderData = TestFixures.createWeatherData();
			ComplexTypeSegment<WeatherData> theReceiverSegment;
			Segment theSenderSeg = readyToSendSegment( stopAndWaitSegment( crcPrefixSegment( complexTypeSegment( theSenderData ), CrcAlgorithmConfig.CRC_16_CCITT_FALSE ) ) );
			Segment theReceiverSeg = readyToSendSegment( stopAndWaitSegment( crcPrefixSegment( theReceiverSegment = complexTypeSegment( WeatherData.class ), CrcAlgorithmConfig.CRC_16_CCITT_FALSE ) ) );

			for ( int i = 0; i < LOOPS; i++ ) {
				SegmentResult<Segment> theResult = theReceiverPort.onReceiveSegment( theReceiverSeg );
				theSenderSeg.transmitTo( theTransmitPort );
				theResult.waitForResult();
				WeatherData theReceiverData = theReceiverSegment.getPayload();
				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( theReceiverData );
				}
				assertEquals( theSenderData, theReceiverData );
			}

			theTransmitPort.close();
			theReceiverPort.close();
		}
		else {
			System.out.println( "Skipping test, please connect your null modem cable to two serial ports on your box, seeking for exactly two FT232 (ftdi_sio) type ports!" );
		}
	}

	@Test
	public void testReadyToSendSegmentDecorator1Breaker() throws IOException, InterruptedException {
		if ( hasPorts() ) {
			Port<?> theTransmitPort = getTransmitterPort().withOpen();
			Port<?> theReceiverPort = getReceiverPort().withOpen();
			WeatherData theSenderData = TestFixures.createWeatherData();
			ComplexTypeSegment<WeatherData> theReceiverSegment;
			BreakerSegmentDecorator<?> theBreaker;
			Segment theSenderSeg = readyToSendSegment( stopAndWaitSegment( crcPrefixSegment( complexTypeSegment( theSenderData ), CrcAlgorithmConfig.CRC_16_CCITT_FALSE ) ) );
			Segment theReceiverSeg = readyToSendSegment( stopAndWaitSegment( theBreaker = breakerSegment( crcPrefixSegment( theReceiverSegment = complexTypeSegment( WeatherData.class ), CrcAlgorithmConfig.CRC_16_CCITT_FALSE ), 3 ) ) );

			for ( int i = 0; i < LOOPS; i++ ) {
				SegmentResult<Segment> theResult = theReceiverPort.onReceiveSegment( theReceiverSeg );
				theSenderSeg.transmitTo( theTransmitPort );
				theResult.waitForResult();
				WeatherData theReceiverData = theReceiverSegment.getPayload();
				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( theReceiverData );
				}
				assertEquals( theSenderData, theReceiverData );
				theBreaker.reset();
			}

			theTransmitPort.close();
			theReceiverPort.close();
		}
		else {
			System.out.println( "Skipping test, please connect your null modem cable to two serial ports on your box, seeking for exactly two FT232 (ftdi_sio) type ports!" );
		}
	}

	@Test
	public void testReadyToSendSegmentDecorator2() throws IOException, InterruptedException {
		if ( hasPorts() ) {
			Port<?> theTransmitPort = getTransmitterPort().withOpen();
			Port<?> theReceiverPort = getReceiverPort().withOpen();
			WeatherData theSenderData = TestFixures.createWeatherData();
			ComplexTypeSegment<WeatherData> theReceiverSegment;
			Segment theSenderSeg = readyToSendSegment( stopAndWaitPacketStreamSegmentBuilder().withDecoratee( crcPrefixSegment( complexTypeSegment( theSenderData ), CrcAlgorithmConfig.CRC_16_CCITT_FALSE ) ).withBlockSize( 16 ).build() );
			Segment theReceiverSeg = readyToSendSegment( stopAndWaitPacketStreamSegmentBuilder().withDecoratee( crcPrefixSegment( theReceiverSegment = complexTypeSegment( WeatherData.class ), CrcAlgorithmConfig.CRC_16_CCITT_FALSE ) ).withBlockSize( 16 ).build() );

			for ( int i = 0; i < LOOPS; i++ ) {
				SegmentResult<Segment> theResult = theReceiverPort.onReceiveSegment( theReceiverSeg );
				theSenderSeg.transmitTo( theTransmitPort );
				theResult.waitForResult();
				WeatherData theReceiverData = theReceiverSegment.getPayload();
				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( theReceiverData );
				}
				assertEquals( theSenderData, theReceiverData );
			}

			theTransmitPort.close();
			theReceiverPort.close();
		}
		else {
			System.out.println( "Skipping test, please connect your null modem cable to two serial ports on your box, seeking for exactly two FT232 (ftdi_sio) type ports!" );
		}
	}

	@Test
	public void testReadyToSendSegmentDecorator3() throws IOException, InterruptedException {
		if ( hasPorts() ) {
			Port<?> theTransmitPort = getTransmitterPort().withOpen();
			Port<?> theReceiverPort = getReceiverPort().withOpen();
			WeatherData theSenderData = TestFixures.createWeatherData();
			ComplexTypeSegment<WeatherData> theReceiverSegment;
			Segment theSenderSeg = readyToSendSegment( stopAndWaitSegment( crcPrefixSegment( complexTypeSegment( theSenderData ), CrcAlgorithmConfig.CRC_16_CCITT_FALSE ) ), 5000 );
			Segment theReceiverSeg = readyToSendSegment( stopAndWaitSegment( crcPrefixSegment( theReceiverSegment = complexTypeSegment( WeatherData.class ), CrcAlgorithmConfig.CRC_16_CCITT_FALSE ) ), 5000 );

			for ( int i = 0; i < LOOPS; i++ ) {
				SegmentResult<Segment> theResult = theTransmitPort.doTransmitSegment( theSenderSeg );
				Thread.sleep( 400 );
				theReceiverPort.receiveSegment( theReceiverSeg );
				WeatherData theReceiverData = theReceiverSegment.getPayload();
				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( theReceiverData );
				}
				theResult.waitForResult();
				assertEquals( theSenderData, theReceiverData );
			}

			theTransmitPort.close();
			theReceiverPort.close();
		}
		else {
			System.out.println( "Skipping test, please connect your null modem cable to two serial ports on your box, seeking for exactly two FT232 (ftdi_sio) type ports!" );
		}
	}

	@Test
	public void testReadyToSendSegmentDecorator4() throws IOException, InterruptedException {
		if ( hasPorts() ) {
			Port<?> theTransmitPort = getTransmitterPort().withOpen();
			Port<?> theReceiverPort = getReceiverPort().withOpen();
			WeatherData theSenderData = TestFixures.createWeatherData();
			Segment theSenderSeg = readyToSendSegmentBuilder().withDecoratee( crcPrefixSegment( complexTypeSegment( theSenderData ), CrcAlgorithmConfig.CRC_16_CCITT_FALSE ) ).withReadyToSendRetryNumber( 2 ).withReadyToSendTimeoutMillis( 200 ).build();

			for ( int i = 0; i < LOOPS; i++ ) {
				try {
					theTransmitPort.transmitSegment( theSenderSeg );
					fail( "Expected a <" + FlowControlTimeoutException.class.getSimpleName() + ">!" );
				}
				catch ( FlowControlTimeoutException e ) {
					if ( IS_LOG_TEST_ENABLED ) {
						System.out.println( e.getMessage() );
						// e.printStackTrace();
					}
					/* expected */
				}
			}

			theTransmitPort.close();
			theReceiverPort.close();
		}
		else {
			System.out.println( "Skipping test, please connect your null modem cable to two serial ports on your box, seeking for exactly two FT232 (ftdi_sio) type ports!" );
		}
	}

	@Test
	public void testReadyToSendSegmentDecorator5() throws IOException, InterruptedException {
		if ( hasPorts() ) {
			Port<?> theTransmitPort = getTransmitterPort().withOpen();
			Port<?> theReceiverPort = getReceiverPort().withOpen();
			String[] theSenderData = { "This", "is", "not", "a", "love", "song" };
			StringArraySection theReceiverSection;
			Segment theSenderSeg = allocSegment( readyToSendSection( (crcPrefixSection( stringArraySection( theSenderData ), CrcAlgorithmConfig.CRC_16_CCITT_FALSE )) ) );
			Segment theReceiverSeg = allocSegment( readyToSendSection( (crcPrefixSection( theReceiverSection = stringArraySection(), CrcAlgorithmConfig.CRC_16_CCITT_FALSE )) ) );

			for ( int i = 0; i < LOOPS; i++ ) {
				theReceiverSection.setPayload( null );
				SegmentResult<Segment> theResult = theReceiverPort.onReceiveSegment( theReceiverSeg );
				theSenderSeg.transmitTo( theTransmitPort );
				theResult.waitForResult();
				String[] theReceiverData = theReceiverSection.getPayload();
				if ( IS_LOG_TEST_ENABLED ) {
					System.out.println( VerboseTextBuilder.asString( theReceiverData ) );
				}
				assertArrayEquals( theSenderData, theReceiverData );
			}

			theTransmitPort.close();
			theReceiverPort.close();
		}
		else {
			System.out.println( "Skipping test, please connect your null modem cable to two serial ports on your box, seeking for exactly two FT232 (ftdi_sio) type ports!" );
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
