// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.servicebus;

/**
 * Provides an accessor for a {@link Service} property.
 *
 * @param <S> The {@link Service} type property.
 */
public interface ServiceAccessor<S extends Service<?>> {

	/**
	 * Retrieves the {@link Service} property from the property.
	 * 
	 * @return Returns the {@link Service} property stored by the property.
	 */
	S getService();

	/**
	 * Provides a mutator for a {@link Service} property.
	 * 
	 * @param <S> The {@link Service} type property.
	 */
	public interface ServiceMutator<S extends Service<?>> {

		/**
		 * Sets the {@link Service} property for the property.
		 * 
		 * @param aService The {@link Service} property to be stored by the
		 *        property.
		 */
		void setService( S aService );
	}

	/**
	 * Provides a {@link Service} property.
	 * 
	 * @param <S> The {@link Service} type property.
	 */
	public interface ServiceProperty<S extends Service<?>> extends ServiceAccessor<S>, ServiceMutator<S> {

		/**
		 * This method stores and passes through the given argument, which is
		 * very useful for builder APIs: Sets the given value (setter) as of
		 * {@link #setService(Service)} and returns the very same value
		 * (getter).
		 * 
		 * @param aService The value to set (via {@link #setService(Service)}).
		 * 
		 * @return Returns the value passed for it to be used in conclusive
		 *         processing steps.
		 */
		default S letService( S aService ) {
			setService( aService );
			return aService;
		}
	}
}
