package org.refcodes.servicebus;

import java.util.ServiceConfigurationError;

/**
 * The {@link ServiceBus} defines the functionality for {@link Service}
 * instances to look up other {@link Service} instances.
 *
 * @param <S> the generic type
 */
public interface ServiceBus<S extends Service<?>> {

	/**
	 * Looks for a service matching the given service descriptor and returns it
	 * if found.
	 * 
	 * @param aServiceMatcher The {@link ServiceMatcher} describing the service
	 *        for which to look for.
	 * 
	 * @return The desired {@link ServiceConfigurationError}.
	 * 
	 * @throws UnknownServiceRuntimeException in case no {@link Service}
	 *         matched.
	 * @throws AmbiguousServiceRuntimeException in case more than one
	 *         {@link Service} matched the {@link ServiceMatcher}.
	 */
	S lookupService( ServiceMatcher<S> aServiceMatcher ) throws UnknownServiceRuntimeException, AmbiguousServiceRuntimeException;

	/**
	 * Determines whether the given {@link ServiceMatcher} will result in a
	 * single {@link Service} to be looked up. I.e. the {@link ServiceMatcher}
	 * must result in an unambiguous {@link Service} to be identified.
	 * 
	 * @param aServiceMatcher The {@link ServiceMatcher} describing the
	 *        {@link Service} for which to look for.
	 * 
	 * @return True in case if exactly one {@link Service} is matching the
	 *         matcher. Else false is returned (either more than one service was
	 *         identified or none service was identified).
	 */
	boolean hasService( ServiceMatcher<S> aServiceMatcher );
}
