// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.servicebus;

import org.refcodes.exception.AbstractRuntimeException;

/**
 * Base exception for {@link ServiceBus} related runtime exceptions.
 */
public abstract class ServiceBusRuntimeException extends AbstractRuntimeException {

	private static final long serialVersionUID = 1L;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new service bus runtime exception.
	 *
	 * @param aMessage The aMessage describing this exception.
	 * @param aErrorCode The error code identifying this exception.
	 */
	public ServiceBusRuntimeException( String aMessage, String aErrorCode ) {
		super( aMessage, aErrorCode );
	}

	/**
	 * Instantiates a new service bus runtime exception.
	 *
	 * @param aMessage The aMessage describing this exception.
	 * @param aCause The {@link Throwable} ({@link Exception}) causing this
	 *        exception.
	 * @param aErrorCode The error code identifying this exception.
	 */
	public ServiceBusRuntimeException( String aMessage, Throwable aCause, String aErrorCode ) {
		super( aMessage, aCause, aErrorCode );
	}

	/**
	 * Instantiates a new service bus runtime exception.
	 *
	 * @param aMessage The aMessage describing this exception.
	 * @param aCause The {@link Throwable} ({@link Exception}) causing this
	 *        exception.
	 */
	public ServiceBusRuntimeException( String aMessage, Throwable aCause ) {
		super( aMessage, aCause );
	}

	/**
	 * Instantiates a new service bus runtime exception.
	 *
	 * @param aMessage The aMessage describing this exception.
	 */
	public ServiceBusRuntimeException( String aMessage ) {
		super( aMessage );
	}

	/**
	 * Instantiates a new service bus runtime exception.
	 *
	 * @param aCause The {@link Throwable} ({@link Exception}) causing this
	 *        exception.
	 * @param aErrorCode The error code identifying this exception.
	 */
	public ServiceBusRuntimeException( Throwable aCause, String aErrorCode ) {
		super( aCause, aErrorCode );
	}

	/**
	 * Instantiates a new service bus runtime exception.
	 *
	 * @param aCause The {@link Throwable} ({@link Exception}) causing this
	 *        exception.
	 */
	public ServiceBusRuntimeException( Throwable aCause ) {
		super( aCause );
	}

	/**
	 * This exception is the exception for {@link ServiceDescriptor} related
	 * problems.
	 */
	@SuppressWarnings("rawtypes")
	static abstract class ServiceBusServiceDescriptorRuntimeException extends ServiceBusRuntimeException implements ServiceDescriptorAccessor {

		private static final long serialVersionUID = 1L;

		// /////////////////////////////////////////////////////////////////////////
		// VARIABLES:
		// /////////////////////////////////////////////////////////////////////////

		private ServiceDescriptor _serviceDescriptor;

		// /////////////////////////////////////////////////////////////////////////
		// CONSTRUCTORS:
		// /////////////////////////////////////////////////////////////////////////

		/**
		 * Instantiates a new service bus service descriptor runtime exception.
		 *
		 * @param aServiceDescriptor the service descriptor
		 * @param aMessage The aMessage describing this exception.
		 * @param aErrorCode The error code identifying this exception.
		 */
		public ServiceBusServiceDescriptorRuntimeException( ServiceDescriptor aServiceDescriptor, String aMessage, String aErrorCode ) {
			super( aMessage, aErrorCode );
			_serviceDescriptor = aServiceDescriptor;
		}

		/**
		 * Instantiates a new service bus service descriptor runtime exception.
		 *
		 * @param aServiceDescriptor the service descriptor
		 * @param aMessage The aMessage describing this exception.
		 * @param aCause The {@link Throwable} ({@link Exception}) causing this
		 *        exception.
		 * @param aErrorCode The error code identifying this exception.
		 */
		public ServiceBusServiceDescriptorRuntimeException( ServiceDescriptor aServiceDescriptor, String aMessage, Throwable aCause, String aErrorCode ) {
			super( aMessage, aCause, aErrorCode );
			_serviceDescriptor = aServiceDescriptor;
		}

		/**
		 * Instantiates a new service bus service descriptor runtime exception.
		 *
		 * @param aServiceDescriptor the service descriptor
		 * @param aMessage The aMessage describing this exception.
		 * @param aCause The {@link Throwable} ({@link Exception}) causing this
		 *        exception.
		 */
		public ServiceBusServiceDescriptorRuntimeException( ServiceDescriptor aServiceDescriptor, String aMessage, Throwable aCause ) {
			super( aMessage, aCause );
			_serviceDescriptor = aServiceDescriptor;
		}

		/**
		 * Instantiates a new service bus service descriptor runtime exception.
		 *
		 * @param aServiceDescriptor the service descriptor
		 * @param aMessage The aMessage describing this exception.
		 */
		public ServiceBusServiceDescriptorRuntimeException( ServiceDescriptor aServiceDescriptor, String aMessage ) {
			super( aMessage );
			_serviceDescriptor = aServiceDescriptor;
		}

		/**
		 * Instantiates a new service bus service descriptor runtime exception.
		 *
		 * @param aServiceDescriptor the service descriptor
		 * @param aCause The {@link Throwable} ({@link Exception}) causing this
		 *        exception.
		 * @param aErrorCode The error code identifying this exception.
		 */
		public ServiceBusServiceDescriptorRuntimeException( ServiceDescriptor aServiceDescriptor, Throwable aCause, String aErrorCode ) {
			super( aCause, aErrorCode );
			_serviceDescriptor = aServiceDescriptor;

		}

		/**
		 * Instantiates a new service bus service descriptor runtime exception.
		 *
		 * @param aServiceDescriptor the service descriptor
		 * @param aCause The {@link Throwable} ({@link Exception}) causing this
		 *        exception.
		 */
		public ServiceBusServiceDescriptorRuntimeException( ServiceDescriptor aServiceDescriptor, Throwable aCause ) {
			super( aCause );
			_serviceDescriptor = aServiceDescriptor;
		}

		// /////////////////////////////////////////////////////////////////////////
		// METHODS:
		// /////////////////////////////////////////////////////////////////////////

		/**
		 * Gets the service descriptor.
		 *
		 * @return the service descriptor
		 */
		@Override
		public ServiceDescriptor getServiceDescriptor() {
			return _serviceDescriptor;
		}
	}

	/**
	 * This exception is the exception for {@link ServiceMatcher} related
	 * problems.
	 */
	static abstract class ServiceBusServiceMatcherRuntimeException extends ServiceBusRuntimeException implements ServiceMatcherAccessor {

		private static final long serialVersionUID = 1L;

		// /////////////////////////////////////////////////////////////////////
		// VARIABLES:
		// /////////////////////////////////////////////////////////////////////

		private ServiceMatcher<?> _serviceMatcher;

		// /////////////////////////////////////////////////////////////////////
		// CONSTRUCTORS:
		// /////////////////////////////////////////////////////////////////////

		/**
		 * Instantiates a new service bus service matcher runtime exception.
		 *
		 * @param aServiceMatcher the service matcher
		 * @param aMessage The aMessage describing this exception.
		 * @param aErrorCode The error code identifying this exception.
		 */
		public ServiceBusServiceMatcherRuntimeException( ServiceMatcher<?> aServiceMatcher, String aMessage, String aErrorCode ) {
			super( aMessage, aErrorCode );
			_serviceMatcher = aServiceMatcher;
		}

		/**
		 * Instantiates a new service bus service matcher runtime exception.
		 *
		 * @param aServiceMatcher the service matcher
		 * @param aMessage The aMessage describing this exception.
		 * @param aCause The {@link Throwable} ({@link Exception}) causing this
		 *        exception.
		 * @param aErrorCode The error code identifying this exception.
		 */
		public ServiceBusServiceMatcherRuntimeException( ServiceMatcher<?> aServiceMatcher, String aMessage, Throwable aCause, String aErrorCode ) {
			super( aMessage, aCause, aErrorCode );
			_serviceMatcher = aServiceMatcher;
		}

		/**
		 * Instantiates a new service bus service matcher runtime exception.
		 *
		 * @param aServiceMatcher the service matcher
		 * @param aMessage The aMessage describing this exception.
		 * @param aCause The {@link Throwable} ({@link Exception}) causing this
		 *        exception.
		 */
		public ServiceBusServiceMatcherRuntimeException( ServiceMatcher<?> aServiceMatcher, String aMessage, Throwable aCause ) {
			super( aMessage, aCause );
			_serviceMatcher = aServiceMatcher;
		}

		/**
		 * Instantiates a new service bus service matcher runtime exception.
		 *
		 * @param aServiceMatcher the service matcher
		 * @param aMessage The aMessage describing this exception.
		 */
		public ServiceBusServiceMatcherRuntimeException( ServiceMatcher<?> aServiceMatcher, String aMessage ) {
			super( aMessage );
			_serviceMatcher = aServiceMatcher;
		}

		/**
		 * Instantiates a new service bus service matcher runtime exception.
		 *
		 * @param aServiceMatcher the service matcher
		 * @param aCause The {@link Throwable} ({@link Exception}) causing this
		 *        exception.
		 * @param aErrorCode The error code identifying this exception.
		 */
		public ServiceBusServiceMatcherRuntimeException( ServiceMatcher<?> aServiceMatcher, Throwable aCause, String aErrorCode ) {
			super( aCause, aErrorCode );
			_serviceMatcher = aServiceMatcher;

		}

		/**
		 * Instantiates a new service bus service matcher runtime exception.
		 *
		 * @param aServiceMatcher the service matcher
		 * @param aCause The {@link Throwable} ({@link Exception}) causing this
		 *        exception.
		 */
		public ServiceBusServiceMatcherRuntimeException( ServiceMatcher<?> aServiceMatcher, Throwable aCause ) {
			super( aCause );
			_serviceMatcher = aServiceMatcher;
		}

		// /////////////////////////////////////////////////////////////////////
		// METHODS:
		// /////////////////////////////////////////////////////////////////////

		/**
		 * Gets the service matcher.
		 *
		 * @return the service matcher
		 */
		@Override
		public ServiceMatcher<?> getServiceMatcher() {
			return _serviceMatcher;
		}
	}

	/**
	 * This exception is the exception for {@link Service} related problems.
	 */
	@SuppressWarnings("rawtypes")
	static abstract class ServiceBusServiceRuntimeException extends ServiceBusRuntimeException implements ServiceAccessor {

		private static final long serialVersionUID = 1L;

		// /////////////////////////////////////////////////////////////////////
		// VARIABLES:
		// /////////////////////////////////////////////////////////////////////

		private Service _service;

		// /////////////////////////////////////////////////////////////////////
		// CONSTRUCTORS:
		// /////////////////////////////////////////////////////////////////////

		/**
		 * Instantiates a new service bus service runtime exception.
		 *
		 * @param aService the service
		 * @param aMessage The aMessage describing this exception.
		 * @param aErrorCode The error code identifying this exception.
		 */
		public ServiceBusServiceRuntimeException( Service aService, String aMessage, String aErrorCode ) {
			super( aMessage, aErrorCode );
			_service = aService;
		}

		/**
		 * Instantiates a new service bus service runtime exception.
		 *
		 * @param aService the service
		 * @param aMessage The aMessage describing this exception.
		 * @param aCause The {@link Throwable} ({@link Exception}) causing this
		 *        exception.
		 * @param aErrorCode The error code identifying this exception.
		 */
		public ServiceBusServiceRuntimeException( Service aService, String aMessage, Throwable aCause, String aErrorCode ) {
			super( aMessage, aCause, aErrorCode );
			_service = aService;
		}

		/**
		 * Instantiates a new service bus service runtime exception.
		 *
		 * @param aService the service
		 * @param aMessage The aMessage describing this exception.
		 * @param aCause The {@link Throwable} ({@link Exception}) causing this
		 *        exception.
		 */
		public ServiceBusServiceRuntimeException( Service aService, String aMessage, Throwable aCause ) {
			super( aMessage, aCause );
			_service = aService;
		}

		/**
		 * Instantiates a new service bus service runtime exception.
		 *
		 * @param aService the service
		 * @param aMessage The aMessage describing this exception.
		 */
		public ServiceBusServiceRuntimeException( Service aService, String aMessage ) {
			super( aMessage );
			_service = aService;
		}

		/**
		 * Instantiates a new service bus service runtime exception.
		 *
		 * @param aService the service
		 * @param aCause The {@link Throwable} ({@link Exception}) causing this
		 *        exception.
		 * @param aErrorCode The error code identifying this exception.
		 */
		public ServiceBusServiceRuntimeException( Service aService, Throwable aCause, String aErrorCode ) {
			super( aCause, aErrorCode );
			_service = aService;

		}

		/**
		 * Instantiates a new service bus service runtime exception.
		 *
		 * @param aService the service
		 * @param aCause The {@link Throwable} ({@link Exception}) causing this
		 *        exception.
		 */
		public ServiceBusServiceRuntimeException( Service aService, Throwable aCause ) {
			super( aCause );
			_service = aService;
		}

		// /////////////////////////////////////////////////////////////////////
		// METHODS:
		// /////////////////////////////////////////////////////////////////////

		/**
		 * Gets the service.
		 *
		 * @return the service
		 */
		@Override
		public Service getService() {
			return _service;
		}
	}
}
