// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany, distributed
// on an "AS IS" BASIS WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, and licen-
// sed under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/TEXT-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.servicebus;

import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import org.refcodes.factory.CollectionFactory;

/**
 * The {@link ServiceLookupImpl} implements the {@link ServiceLookup}.
 *
 * @param <S> The type to be used for the {@link Service}.
 * @param <SCTX> The type to be used for the {@link Service}'s context.
 */
public class ServiceLookupImpl<S extends Service<?>, SCTX> implements ServiceLookup<S, SCTX> {

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private Set<ServiceDescriptor<S, SCTX>> _serviceDescriptors = new HashSet<ServiceDescriptor<S, SCTX>>();

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Creates a {@link ServiceLookupImpl} for the required type.
	 *
	 * @param aTypedTypeFactory The factory to be used to lookup the given
	 *        type's instances.
	 */
	public ServiceLookupImpl( CollectionFactory<Set<ServiceDescriptor<S, SCTX>>> aTypedTypeFactory ) {
		_serviceDescriptors = aTypedTypeFactory.create();
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Set<ServiceDescriptor<S, SCTX>> getServiceDescriptors() {
		return Collections.unmodifiableSet( _serviceDescriptors );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean hasService( ServiceDescriptor<S, SCTX> aServiceDescriptor ) {
		return _serviceDescriptors.contains( aServiceDescriptor );
	}
}
