package org.refcodes.servicebus;

import org.refcodes.matcher.Matcher;
import org.refcodes.matcher.MatcherSchema;
import org.refcodes.matcher.MatcherSugar;

/**
 * Declarative syntactic sugar which may be statically imported in order to
 * allow declarative definitions for the {@link ServiceMatcher} related
 * elements.
 */
public class ServiceMatcherSugar {

	/**
	 * Factory method to create an service matcher by service type.
	 * 
	 * @param <S> The type of the service to be matched
	 * @param aServiceType The service type to be matched by this matcher.
	 * 
	 * @return An service matcher by service type.
	 */
	public static <S extends Service<?>> ServiceMatcher<S> isAssignableFrom( Class<?> aServiceType ) {
		Matcher<S> thrAssignableFrom = MatcherSugar.isAssignableFrom( aServiceType );
		return new ServiceMatcherWrapper<S>( thrAssignableFrom );
	}

	/**
	 * Factory method to create an "OR" matcher for the given matchers.
	 * 
	 * @param <S> The type of the service applied to the matcher
	 * @param aServiceMatchers The matchers to be combined by an "OR".
	 * 
	 * @return An "OR" matcher.
	 */
	@SafeVarargs
	public static <S extends Service<?>> ServiceMatcher<S> or( ServiceMatcher<S>... aServiceMatchers ) {
		return new ServiceMatcherWrapper<S>( MatcherSugar.or( aServiceMatchers ) );
	}

	/**
	 * Factory method to create an "AND" matcher for the given matchers.
	 * 
	 * @param <S> The type of the service applied to the matcher
	 * @param aServiceMatchers The matchers to be combined by an "AND".
	 * 
	 * @return An "AND" matcher.
	 */
	@SafeVarargs
	public static <S extends Service<?>> ServiceMatcher<S> and( ServiceMatcher<S>... aServiceMatchers ) {
		return new ServiceMatcherWrapper<S>( MatcherSugar.and( aServiceMatchers ) );
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

	private static class ServiceMatcherWrapper<S extends Service<?>> implements ServiceMatcher<S> {

		private Matcher<S> _serviceMatcher;

		public ServiceMatcherWrapper( Matcher<S> aMatcher ) {
			assert (aMatcher != null);
			_serviceMatcher = aMatcher;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public boolean isMatching( S aService ) {
			return _serviceMatcher.isMatching( aService );
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public MatcherSchema toSchema() {
			return new MatcherSchema( getClass(), _serviceMatcher.toSchema() );
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public String getAlias() {
			return _serviceMatcher.getAlias();
		}
	}
}