package org.refcodes.servicebus;

import org.refcodes.component.InitializeException;
import org.refcodes.component.LifecycleRequest;
import org.refcodes.component.LifecycleStatus;
import org.refcodes.component.PauseException;
import org.refcodes.component.ResumeException;
import org.refcodes.component.StartException;
import org.refcodes.component.StopException;
import org.refcodes.component.ext.observer.DestroyedEvent;
import org.refcodes.component.ext.observer.InitializedEvent;
import org.refcodes.component.ext.observer.LifecycleRequestEvent;
import org.refcodes.component.ext.observer.LifecycleStatusObserver;
import org.refcodes.component.ext.observer.ObservableLifecycleStatusAutomaton;
import org.refcodes.component.ext.observer.PausedEvent;
import org.refcodes.component.ext.observer.ResumedEvent;
import org.refcodes.component.ext.observer.StartedEvent;
import org.refcodes.component.ext.observer.StoppedEvent;
import org.refcodes.data.Text;
import org.refcodes.mixin.Loggable;
import org.refcodes.observer.Observer;

/**
 * The {@link SimpleServiceBus} is a basic implementation of the
 * {@link ServiceBus} interface.
 *
 * @param <S> the generic type of the services to be served.
 * @param <CTX> the generic service's context type
 */
public class SimpleServiceBus<S extends Service<?>, CTX extends ServiceContext<S>> implements ServiceBus<S>, Loggable {

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private ServiceLookup<S, CTX> _serviceLookup;

	private ServiceBusObserver _serviceBusObserver = new ServiceBusObserver();

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs the subscriber part of the event bus. For doing its job, it
	 * needs the list of services which is accessible via the provided service
	 * lookup part of the service bus.
	 *
	 * @param aServiceLookup the service lookup
	 * @param aObservableLifecycleAutomaton the observable lifecycle automaton
	 */
	public SimpleServiceBus( ServiceLookup<S, CTX> aServiceLookup, ObservableLifecycleStatusAutomaton aObservableLifecycleAutomaton ) {
		assert (aServiceLookup != null);
		assert (aObservableLifecycleAutomaton != null);
		_serviceLookup = aServiceLookup;
		registerLifecycleEventDispatcher( aObservableLifecycleAutomaton );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public S lookupService( ServiceMatcher<S> aServiceMatcher ) throws NoMatchingServiceRuntimeException, AmbiguousServiceRuntimeException {
		assert (aServiceMatcher != null);
		S theMatchedService = null;
		for ( ServiceDescriptor<S, CTX> theServiceDescriptor : _serviceLookup.getServiceDescriptors() ) {
			S eService = theServiceDescriptor.getService();
			if ( aServiceMatcher.isMatching( eService ) ) {
				if ( theMatchedService != null ) {
					throw new AmbiguousServiceRuntimeException( aServiceMatcher, "The given service matcher matched more than one service." );
				}
				theMatchedService = eService;
			}
		}
		if ( theMatchedService == null ) {
			throw new NoMatchingServiceRuntimeException( aServiceMatcher, "A service for the given matcher is not known by this service bus." );
		}
		return theMatchedService;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean hasService( ServiceMatcher<S> aServiceMatcher ) {
		assert (aServiceMatcher != null);
		boolean hasMatchedService = false;
		for ( ServiceDescriptor<S, CTX> theServiceDescriptor : _serviceLookup.getServiceDescriptors() ) {
			S eService = theServiceDescriptor.getService();
			if ( aServiceMatcher.isMatching( eService ) ) {
				if ( hasMatchedService ) {
					return false;
				}
				hasMatchedService = true;
			}
		}
		return hasMatchedService;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * This method registers a life-cycle event listener which dispatches any
	 * incoming life-cycle events to the according dispatcher method below.
	 *
	 * @param aObservableLifecycleAutomaton the observable lifecycle automaton
	 */
	protected void registerLifecycleEventDispatcher( ObservableLifecycleStatusAutomaton aObservableLifecycleAutomaton ) {
		aObservableLifecycleAutomaton.subscribeObserver( _serviceBusObserver );
	}

	/**
	 * This method is used to handle life-cycle events and inform the according
	 * services of any life-cycle state changes. Use the
	 * {@link #registerLifecycleEventDispatcher(ObservableLifecycleStatusAutomaton)}
	 * method to register a dispatcher for any life-cycle events from a provided
	 * observable life-cycle instance.
	 *
	 * @param aEvent the event
	 */
	protected void onLifecycleEvent( LifecycleRequestEvent aEvent ) {
		switch ( aEvent.getLifecycleRequest() ) {
		case INITIALIZE:
			initializeServices();
			break;
		case START:
			startServices();
			break;
		case PAUSE:
			pauseServices();
			break;
		case RESUME:
			resumeServices();
			break;
		case STOP:
			stopServices();
			break;
		case DESTROY:
			destroyServices();
			break;
		}
	}

	/**
	 * Initialize services.
	 */
	protected void initializeServices() {
		info( "About to " + LifecycleRequest.INITIALIZE + " services ..." );
		ObservableLifecycleStatusAutomaton eServiceLifecycle;
		for ( ServiceDescriptor<S, CTX> eServiceDescriptor : _serviceLookup.getServiceDescriptors() ) {
			eServiceLifecycle = eServiceDescriptor.getServiceContext().getObservableLifecycleAutomaton();
			if ( eServiceLifecycle.isInitalizable() ) {
				try {
					eServiceLifecycle.initialize();
				}
				catch ( InitializeException e ) {
					warn( "Unable to \"" + LifecycleRequest.INITIALIZE + "\" the service \"" + eServiceDescriptor.getClass().getName() + "\".", e );
				}
			}
		}
		info( "Services \"" + LifecycleStatus.INITIALIZED + "\"." );
	}

	/**
	 * Start services.
	 */
	protected void startServices() {
		info( "About to " + LifecycleRequest.START + " services ..." );
		ObservableLifecycleStatusAutomaton eServiceLifecycle;
		for ( ServiceDescriptor<S, CTX> eServiceDescriptor : _serviceLookup.getServiceDescriptors() ) {
			eServiceLifecycle = eServiceDescriptor.getServiceContext().getObservableLifecycleAutomaton();
			if ( eServiceLifecycle.isStartable() ) {
				try {
					eServiceLifecycle.start();
				}
				catch ( StartException e ) {
					warn( "Unable to \"" + LifecycleRequest.START + "\" the service \"" + eServiceDescriptor.getClass().getName() + "\".", e );
				}
			}
		}
		info( "Services \"" + LifecycleStatus.STARTED + "\"." );
	}

	/**
	 * Pause services.
	 */
	protected void pauseServices() {
		info( "About to " + LifecycleRequest.PAUSE + " services ..." );
		ObservableLifecycleStatusAutomaton eServiceLifecycle;
		for ( ServiceDescriptor<S, CTX> eServiceDescriptor : _serviceLookup.getServiceDescriptors() ) {
			eServiceLifecycle = eServiceDescriptor.getServiceContext().getObservableLifecycleAutomaton();
			if ( eServiceLifecycle.isPausable() ) {
				try {
					eServiceLifecycle.pause();
				}
				catch ( PauseException e ) {
					warn( "Unable to \"" + LifecycleRequest.PAUSE + "\" the service \"" + eServiceDescriptor.getClass().getName() + "\".", e );
				}
			}
		}
		info( "Services \"" + LifecycleStatus.PAUSED + "\"." );
	}

	/**
	 * Resume services.
	 */
	protected void resumeServices() {
		info( "About to " + LifecycleRequest.RESUME + " services ..." );
		ObservableLifecycleStatusAutomaton eServiceLifecycle;
		for ( ServiceDescriptor<S, CTX> eServiceDescriptor : _serviceLookup.getServiceDescriptors() ) {
			eServiceLifecycle = eServiceDescriptor.getServiceContext().getObservableLifecycleAutomaton();
			if ( eServiceLifecycle.isResumable() ) {
				try {
					eServiceLifecycle.resume();
				}
				catch ( ResumeException e ) {
					warn( "Unable to \"" + LifecycleRequest.RESUME + "\" the service \"" + eServiceDescriptor.getClass().getName() + "\".", e );
				}
			}
		}
		info( "Services \"" + LifecycleStatus.STARTED + "\"." );
	}

	/**
	 * Stop services.
	 */
	protected void stopServices() {
		info( "About to " + LifecycleRequest.STOP + " services ..." );
		ObservableLifecycleStatusAutomaton eServiceLifecycle;
		for ( ServiceDescriptor<S, CTX> eServiceDescriptor : _serviceLookup.getServiceDescriptors() ) {
			eServiceLifecycle = eServiceDescriptor.getServiceContext().getObservableLifecycleAutomaton();
			if ( eServiceLifecycle.isStoppable() ) {
				try {
					eServiceLifecycle.stop();
				}
				catch ( StopException e ) {
					warn( "Unable to \"" + LifecycleRequest.STOP + "\" the service \"" + eServiceDescriptor.getClass().getName() + "\".", e );
				}
			}
		}
		info( "Services \"" + LifecycleStatus.STOPPED + "\"." );
	}

	/**
	 * Destroy services.
	 */
	protected void destroyServices() {
		info( "About to " + LifecycleRequest.DESTROY + " services ..." );
		for ( ServiceDescriptor<S, CTX> theServiceDescriptor : _serviceLookup.getServiceDescriptors() ) {
			ObservableLifecycleStatusAutomaton theServiceLifecycle = theServiceDescriptor.getServiceContext().getObservableLifecycleAutomaton();
			if ( theServiceLifecycle.isDestroyable() ) {
				theServiceLifecycle.destroy();
			}
		}
		info( "Services \"" + LifecycleStatus.DESTROYED + "\"." );
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * The {@link Observer} used by the {@link ServiceBus} to monitor the
	 * {@link ObservableLifecycleStatusAutomaton} instances as provided to the
	 * {@link Service} instances.
	 */
	private class ServiceBusObserver implements LifecycleStatusObserver {

		/**
		 * This method is called when information about an ServiceBus which was
		 * previously requested using an asynchronous interface becomes
		 * available.
		 *
		 * @param aEvent the event
		 */
		@Override
		public void onInitialized( InitializedEvent aEvent ) {
			throw new RuntimeException( Text.NOT_YET_IMPLEMENTED.getText() );
		}

		/**
		 * This method is called when information about an ServiceBus which was
		 * previously requested using an asynchronous interface becomes
		 * available.
		 *
		 * @param aEvent the event
		 */
		@Override
		public void onStarted( StartedEvent aEvent ) {
			throw new RuntimeException( Text.NOT_YET_IMPLEMENTED.getText() );
		}

		/**
		 * This method is called when information about an ServiceBus which was
		 * previously requested using an asynchronous interface becomes
		 * available.
		 *
		 * @param aEvent the event
		 */
		@Override
		public void onResumed( ResumedEvent aEvent ) {
			throw new RuntimeException( Text.NOT_YET_IMPLEMENTED.getText() );
		}

		/**
		 * This method is called when information about an ServiceBus which was
		 * previously requested using an asynchronous interface becomes
		 * available.
		 *
		 * @param aEvent the event
		 */
		@Override
		public void onPaused( PausedEvent aEvent ) {
			throw new RuntimeException( Text.NOT_YET_IMPLEMENTED.getText() );
		}

		/**
		 * This method is called when information about an ServiceBus which was
		 * previously requested using an asynchronous interface becomes
		 * available.
		 *
		 * @param aEvent the event
		 */
		@Override
		public void onStopped( StoppedEvent aEvent ) {
			throw new RuntimeException( Text.NOT_YET_IMPLEMENTED.getText() );
		}

		/**
		 * This method is called when information about an ServiceBus which was
		 * previously requested using an asynchronous interface becomes
		 * available.
		 *
		 * @param aEvent the event
		 */
		@Override
		public void onDestroyed( DestroyedEvent aEvent ) {
			throw new RuntimeException( Text.NOT_YET_IMPLEMENTED.getText() );
		}

	}
}
