/*
 * Decompiled with CFR 0.152.
 */
package org.revapi.configuration;

import java.io.IOException;
import java.io.Reader;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.jboss.dmr.ModelNode;
import org.jboss.dmr.ModelType;
import org.revapi.Revapi;
import org.revapi.configuration.Configurable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class XmlToJson<Xml> {
    private static final Logger LOG = LoggerFactory.getLogger(XmlToJson.class);
    private final Function<Xml, String> getName;
    private final BiFunction<Xml, String, String> getAttributeValue;
    private final Function<Xml, String> getValue;
    private final Function<Xml, List<Xml>> getChildren;
    private final Map<String, ModelNode> knownExtensionSchemas;

    public XmlToJson(Revapi revapi, Function<Xml, String> getName, Function<Xml, String> getValue, BiFunction<Xml, String, String> getAttributeValue, Function<Xml, List<Xml>> getChildren) {
        this(XmlToJson.getKnownExtensionSchemas(revapi), getName, getValue, getAttributeValue, getChildren);
    }

    public XmlToJson(Map<String, ModelNode> knownExtensionSchemas, Function<Xml, String> getName, Function<Xml, String> getValue, BiFunction<Xml, String, String> getAttributeValue, Function<Xml, List<Xml>> getChildren) {
        this.getName = getName;
        this.getValue = getValue;
        this.getAttributeValue = getAttributeValue;
        this.getChildren = getChildren;
        this.knownExtensionSchemas = knownExtensionSchemas;
    }

    public ModelNode convert(Xml xml) {
        ModelNode fullConfiguration = new ModelNode();
        fullConfiguration.setEmptyList();
        for (Xml c : this.getChildren.apply(xml)) {
            String extensionId = this.getName.apply(c);
            String id = this.getAttributeValue.apply(c, "id");
            ModelNode schema = this.knownExtensionSchemas.get(extensionId);
            if (schema == null) {
                LOG.warn("Extension '" + extensionId + "' doesn't declare a JSON schema but XML contains its configuration. Cannot convert it into JSON and will ignore it!");
                continue;
            }
            ModelNode config = this.convert(c, schema);
            ModelNode instanceConfig = new ModelNode();
            instanceConfig.get("extension").set(extensionId);
            if (id != null) {
                instanceConfig.get("id").set(id);
            }
            instanceConfig.get("configuration").set(config);
            fullConfiguration.add(instanceConfig);
        }
        return fullConfiguration;
    }

    private ModelNode convert(Xml configuration, ModelNode jsonSchema) {
        return this.convert(configuration, jsonSchema, jsonSchema);
    }

    private ModelNode convert(Xml configuration, ModelNode jsonSchema, ModelNode rootSchema) {
        String type;
        ModelNode typeNode = jsonSchema.get("type");
        if (!typeNode.isDefined()) {
            ModelNode ret = null;
            if (jsonSchema.get("enum").isDefined()) {
                ret = this.convertByEnum(configuration, jsonSchema.get("enum"));
            } else if (jsonSchema.get("$ref").isDefined()) {
                jsonSchema = XmlToJson.findRef(rootSchema, jsonSchema.get("$ref").asString());
                ret = this.convert(configuration, jsonSchema, rootSchema);
            } else if (jsonSchema.hasDefined("oneOf")) {
                ret = this.convertByOneOf(configuration, jsonSchema.get("oneOf").asList());
            } else if (jsonSchema.hasDefined("anyOf")) {
                ret = this.convertByAnyOf(configuration, jsonSchema.get("anyOf").asList());
            } else if (jsonSchema.hasDefined("allOf")) {
                ret = this.convertByAllOf(configuration, jsonSchema.get("allOf").asList());
            }
            if (ret == null) {
                throw new IllegalArgumentException("Could not convert the configuration.");
            }
            return ret;
        }
        if (typeNode.getType() != ModelType.STRING) {
            throw new IllegalArgumentException("JSON schema allows for multiple possible types. This is not supported by the XML-to-JSON conversion yet.");
        }
        switch (type = typeNode.asString()) {
            case "boolean": {
                return this.convertBoolean(configuration);
            }
            case "integer": {
                return this.convertInteger(configuration);
            }
            case "number": {
                return this.convertNumber(configuration);
            }
            case "string": {
                return this.convertString(configuration);
            }
            case "array": {
                return this.convertArray(configuration, jsonSchema, rootSchema);
            }
            case "object": {
                return this.convertObject(configuration, jsonSchema, rootSchema);
            }
        }
        throw new IllegalArgumentException("Unsupported json value type: " + type);
    }

    private static ModelNode findRef(ModelNode rootSchema, String ref) {
        return JSONPointer.parse(ref).navigate(rootSchema);
    }

    private ModelNode convertByEnum(Xml configuration, ModelNode enumValues) {
        String xmlValue = this.getValue.apply(configuration);
        Map<String, ModelType> jsonValues = enumValues.asList().stream().collect(Collectors.toMap(ModelNode::asString, ModelNode::getType));
        for (Map.Entry<String, ModelType> e : jsonValues.entrySet()) {
            String jsonValue = e.getKey();
            ModelType jsonType = e.getValue();
            if (!Objects.equals(xmlValue, jsonValue)) continue;
            switch (jsonType) {
                case BIG_INTEGER: 
                case INT: 
                case LONG: {
                    return new ModelNode(Long.parseLong(xmlValue));
                }
                case BIG_DECIMAL: 
                case DOUBLE: {
                    return new ModelNode(Double.parseDouble(xmlValue));
                }
                case BOOLEAN: {
                    return new ModelNode(Boolean.parseBoolean(xmlValue));
                }
                case STRING: {
                    return new ModelNode(xmlValue);
                }
            }
            throw new IllegalArgumentException("Unsupported type of enum value defined in schema: " + jsonValue);
        }
        throw new IllegalArgumentException("XML value '" + xmlValue + " doesn't match any of the allowed: " + jsonValues.keySet());
    }

    private ModelNode convertObject(Xml configuration, ModelNode jsonSchema, ModelNode rootSchema) {
        ModelNode object = new ModelNode();
        object.setEmptyObject();
        ModelNode propertySchemas = jsonSchema.get("properties");
        ModelNode additionalPropSchemas = jsonSchema.get("additionalProperties");
        for (Xml childConfig : this.getChildren.apply(configuration)) {
            String name = this.getName.apply(childConfig);
            ModelNode childSchema = propertySchemas.get(name);
            if (!childSchema.isDefined()) {
                if (additionalPropSchemas.getType() == ModelType.BOOLEAN) {
                    throw new IllegalArgumentException("Cannot determine the format for the '" + name + "' XML tag during the XML-to-JSON conversion.");
                }
                childSchema = additionalPropSchemas;
            }
            if (!childSchema.isDefined()) {
                throw new IllegalArgumentException("Could not determine the format for the '" + name + "' XML tag during the XML-to-JSON conversion.");
            }
            ModelNode jsonChild = this.convert(childConfig, childSchema, rootSchema);
            object.get(name).set(jsonChild);
        }
        return object;
    }

    private ModelNode convertArray(Xml configuration, ModelNode jsonSchema, ModelNode rootSchema) {
        ModelNode itemsSchema = jsonSchema.get("items");
        if (!itemsSchema.isDefined()) {
            throw new IllegalArgumentException("No schema found for items of a list. Cannot continue with XML-to-JSON conversion.");
        }
        if (this.getValue.apply(configuration) != null) {
            throw new IllegalArgumentException("Array is not allowed to have a text node: <" + this.getName.apply(configuration) + ">");
        }
        ModelNode list = new ModelNode();
        list.setEmptyList();
        for (Xml childConfig : this.getChildren.apply(configuration)) {
            ModelNode child = this.convert(childConfig, itemsSchema, rootSchema);
            list.add(child);
        }
        return list;
    }

    private ModelNode convertString(Xml configuration) {
        return new ModelNode(this.getValue.apply(configuration));
    }

    private ModelNode convertNumber(Xml configuration) {
        try {
            double floatVal = Double.parseDouble(this.getValue.apply(configuration));
            return new ModelNode(floatVal);
        }
        catch (NumberFormatException e) {
            throw new IllegalArgumentException("Invalid value.", e);
        }
    }

    private ModelNode convertInteger(Xml configuration) {
        try {
            long intVal = Long.parseLong(this.getValue.apply(configuration));
            return new ModelNode(intVal);
        }
        catch (NumberFormatException e) {
            throw new IllegalArgumentException("Invalid value.", e);
        }
    }

    private ModelNode convertBoolean(Xml configuration) {
        Boolean boolVal;
        String v = this.getValue.apply(configuration);
        Boolean bl = "true".equalsIgnoreCase(v) ? Boolean.TRUE : (boolVal = "false".equalsIgnoreCase(v) ? Boolean.FALSE : null);
        if (boolVal == null) {
            throw new IllegalArgumentException("'true' or 'false' expected as a boolean value.");
        }
        return new ModelNode(boolVal.booleanValue());
    }

    private ModelNode convertByOneOf(Xml configuration, Iterable<ModelNode> candidateSchemas) {
        boolean matched = false;
        ModelNode parsed = null;
        for (ModelNode candidateSchema : candidateSchemas) {
            try {
                parsed = this.convert(configuration, candidateSchema);
                if (matched) {
                    return null;
                }
                matched = true;
            }
            catch (IllegalArgumentException illegalArgumentException) {}
        }
        return parsed;
    }

    private ModelNode convertByAnyOf(Xml configuration, Iterable<ModelNode> candidateSchemas) {
        for (ModelNode candidateSchema : candidateSchemas) {
            try {
                return this.convert(configuration, candidateSchema);
            }
            catch (IllegalArgumentException illegalArgumentException) {
            }
        }
        return null;
    }

    private ModelNode convertByAllOf(Xml configuration, Iterable<ModelNode> candidateSchemas) {
        ModelNode parsed = null;
        for (ModelNode candidateSchema : candidateSchemas) {
            try {
                parsed = this.convert(configuration, candidateSchema);
            }
            catch (IllegalArgumentException __) {
                return null;
            }
        }
        return parsed;
    }

    private static Map<String, ModelNode> getKnownExtensionSchemas(Revapi revapi) {
        HashMap<String, ModelNode> knownSchemas = new HashMap<String, ModelNode>();
        XmlToJson.extractKnownSchemas(knownSchemas, revapi.getPipelineConfiguration().getApiAnalyzerTypes());
        XmlToJson.extractKnownSchemas(knownSchemas, revapi.getPipelineConfiguration().getTransformTypes());
        XmlToJson.extractKnownSchemas(knownSchemas, revapi.getPipelineConfiguration().getFilterTypes());
        XmlToJson.extractKnownSchemas(knownSchemas, revapi.getPipelineConfiguration().getReporterTypes());
        return knownSchemas;
    }

    private static <T extends Configurable> void extractKnownSchemas(Map<String, ModelNode> schemaByExtensionId, Set<Class<? extends T>> types) {
        for (Class<T> extensionType : types) {
            try {
                Reader schema;
                Configurable c = (Configurable)extensionType.newInstance();
                String extensionId = c.getExtensionId();
                if (extensionId == null || (schema = c.getJSONSchema()) == null) continue;
                String schemaString = XmlToJson.readFull(schema);
                ModelNode schemaNode = ModelNode.fromJSONString((String)schemaString);
                schemaByExtensionId.put(extensionId, schemaNode);
            }
            catch (IllegalAccessException | InstantiationException e) {
                throw new IllegalStateException("Extension " + extensionType + " is not default-constructable.");
            }
            catch (IOException e) {
                throw new IllegalArgumentException("Failed to read the schema of extension " + extensionType);
            }
        }
    }

    private static String readFull(Reader reader) throws IOException {
        int cnt;
        char[] buffer = new char[512];
        StringBuilder bld = new StringBuilder();
        while ((cnt = reader.read(buffer)) != -1) {
            bld.append(buffer, 0, cnt);
        }
        return bld.toString();
    }

    public static final class JSONPointer {
        private final List<String> tokens;

        public static JSONPointer parse(String pointer) {
            if (pointer.isEmpty() || pointer.equals("#")) {
                return new JSONPointer(Collections.emptyList());
            }
            if (pointer.startsWith("#/")) {
                pointer = pointer.substring(2);
                try {
                    pointer = URLDecoder.decode(pointer, "UTF-8");
                }
                catch (UnsupportedEncodingException e) {
                    throw new IllegalStateException("UTF-8 not supported. What?");
                }
            } else if (pointer.startsWith("/")) {
                pointer = pointer.substring(1);
            } else {
                throw new IllegalArgumentException("a JSON pointer should start with '/' or '#/'");
            }
            ArrayList<String> tokens = new ArrayList<String>();
            for (String token : pointer.split("/")) {
                tokens.add(JSONPointer.unescape(token));
            }
            return new JSONPointer(tokens);
        }

        private JSONPointer(List<String> tokens) {
            this.tokens = tokens;
        }

        public ModelNode navigate(ModelNode root) {
            if (this.tokens.isEmpty()) {
                return root;
            }
            ModelNode current = root;
            for (String token : this.tokens) {
                if (current.getType() == ModelType.OBJECT) {
                    current = current.get(JSONPointer.unescape(token));
                    continue;
                }
                if (current.getType() == ModelType.LIST) {
                    int idx = Integer.parseInt(token);
                    current = current.get(idx);
                    continue;
                }
                throw new IllegalArgumentException("Cannot navigate to '" + this + "' in " + root);
            }
            return current;
        }

        public String toString() {
            StringBuilder ret = new StringBuilder();
            for (String token : this.tokens) {
                ret.append('/').append(JSONPointer.escape(token));
            }
            return ret.toString();
        }

        private static String unescape(String token) {
            return token.replace("~1", "/").replace("~0", "~").replace("\\\"", "\"").replace("\\\\", "\\");
        }

        private static String escape(String token) {
            return token.replace("~", "~0").replace("/", "~1").replace("\\", "\\\\").replace("\"", "\\\"");
        }
    }
}

